# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['dexom_python', 'dexom_python.cluster_utils', 'dexom_python.enum_functions']

package_data = \
{'': ['*']}

install_requires = \
['cobra>=0.25,<0.26',
 'matplotlib>=3.5.1,<4.0.0',
 'numpy==1.20',
 'scikit-learn>=1.0.2,<2.0.0',
 'scipy==1.7',
 'six>=1.16,<2.0',
 'statsmodels>=0.13.2,<0.14.0',
 'symengine>=0.9.2,<0.10.0']

setup_kwargs = {
    'name': 'dexom-python',
    'version': '0.5',
    'description': 'DEXOM implementation in python using cobrapy',
    'long_description': '# DEXOM in python\n\n<a href = "https://github.com/MetExplore/dexom-python/blob/master/LICENSE"><img alt="GitHub license" src="https://img.shields.io/github/license/maximiliansti/dexom_python"></a>\n<a href="https://pypi.org/project/dexom-python/"><img alt = "PyPI Package" src = "https://img.shields.io/pypi/v/dexom-python"/></a>  \n\nThis is a python implementation of DEXOM (Diversity-based enumeration of optimal context-specific metabolic networks)  \nThe original project, which was developped in MATLAB, can be found here: https://github.com/MetExplore/dexom  \nParts of the imat code were taken from the driven package for data-driven constraint-based analysis: https://github.com/opencobra/driven\n\nAPI documentation is available here: https://dexom-python.readthedocs.io/en/stable/\n\nThe package can be installed using pip: `pip install dexom-python`\n\nYou can also clone the git repository with `git clone https://github.com/MetExplore/dexom-python` and then install dependencies with `python setup.py install`\n\n## Requirements\n- Python 3.7 - 3.9\n- CPLEX 12.10 - 22.10\n\n### Installing CPLEX\n\n[Free license (Trial version)](https://www.ibm.com/analytics/cplex-optimizer): this version is limited to 1000 variables and 1000 constraints, and is therefore not useable on larger models\n\n[Academic license](https://www.ibm.com/academic/technology/data-science): for this, you must sign up using an academic email address.\n - after logging in, you can access the download for "ILOG CPLEX Optimization Studio"\n - download version 12.10 or higher of the appropriate installer for your operating system\n - install the solver \n\nYou must then update the PYTHONPATH environment variable by adding the directory containing the `setup.py` file appropriate for your OS and python version   \nAlternatively, run `python "C:\\Program Files\\IBM\\ILOG\\CPLEX_Studio1210\\python\\setup.py" install` and/or `pip install cplex==12.10`\n\n## Functions\n\nThese are the different functions which are available for context-specific metabolic subnetwork extraction\n\n### apply_gpr\nThe `gpr_rules.py` script can be used to transform gene expression data into reaction weights, for a limited selection of models.  \nIt uses the gene identifiers and gene-protein-reaction rules present in the model to connect the genes and reactions.  \nBy default, continuous gene expression values/weights will be transformed into continuous reaction weights.  \nUsing the `--convert` flag will instead create semi-quantitative reaction weights with values in {-1, 0, 1}. By default, the proportion of these three weights will be {25%, 50%, 25%}.\n\n### iMAT\n`imat.py` contains a modified version of the iMAT algorithm as defined by [(Shlomi et al. 2008)](https://pubmed.ncbi.nlm.nih.gov/18711341/).  \nThe main inputs of this algorithm are a model file, which must be supplied in a cobrapy-compatible format (SBML, JSON or MAT), and a reaction_weight file in which each reaction is attributed a score.  \nThese reaction weights must be determined prior to launching imat, for example with GPR rules present in the metabolic model.  \n\nThe remaining inputs of imat are:\n- `epsilon`: the activation threshold of reactions with weight > 0\n- `threshold`: the activation threshold for unweighted reactions\n- `full`: a bool parameter for switching between the partial & full-DEXOM implementation\n\nIn addition, the following solver parameters have been made available through the solver API:\n- `timelimit`: the maximum amount of time allowed for solver optimization (in seconds)\n- `feasibility`: the solver feasbility tolerance\n- `mipgaptol`: the solver MIP gap tolerance\nnote: the feasibility determines the solver\'s capacity to return correct results. \nIn particular, it is necessary that `epsilon` > `threshold` > `ub*feasibility` (where `ub` is the maximal upper bound for reaction flux in the model)\n\nBy default, imat uses the `create_new_partial_variables` function. In this version, binary flux indicator variables are created for each reaction with a non-zero weight.  \nIn the full-DEXOM implementation, binary flux indicator variables are created for every reaction in the model. This does not change the result of the imat function, but can be used for the enumeration methods below.\n\n### enum_functions\n\nFour methods for enumerating context-specific networks are available:\n- `rxn_enum_functions.py` contains reaction-enumeration (function name: `rxn_enum`)\n- `icut_functions.py` contains integer-cut (function name: `icut`)\n- `maxdist_functions.py` contains distance-maximization (function name: `maxdistm`)\n- `diversity_enum_functions.py` contains diversity-enumeration  (function name: `diversity_enum`)\n\nAn explanation of these methods can be found in [(Rodriguez-Mier et al. 2021)](https://doi.org/10.1371/journal.pcbi.1008730).  \nEach of these methods can be used on its own. The same model and reaction_weights inputs must be provided as for the imat function.\n\nAdditional parameters for all 4 methods are:\n- `prev_sol`: an imat solution used as a starting point (if none is provided, a new one will be computed)  \n- `obj_tol`: the relative tolerance on the imat objective value for the optimality of the solutions  \nicut, maxdist, and diversity-enum also have two additional parameters:\n- `maxiter`: the maximum number of iterations to run\n- `full`: set to True to use the full-DEXOM implementation  \nAs previously explained, the full-DEXOM implementation defines binary indicator variables for all reactions in the model. Although only the reactions with non-zero weights have an impact on the imat objective function, the distance maximization function which is used in maxdist and diversity-enum can utilize the binary indicators for all reactions. This increases the distance between the solutions and their diversity, but requires significantly more computation time.  \nmaxdist and div-enum also have one additional parameter:  \n- `icut`: if True, an icut constraint will be applied to prevent duplicate solutions\n\n## Parallelized DEXOM\n\nThe DEXOM algorithm is a combination of several network enumeration methods.  \n`write_cluster_scripts.py` contains functions which are used for creating a parallelization of DEXOM on a slurm computation cluster.\nThe default function is `write_batch_script1`.\nThe main inputs of this function are:\n- `filenums`: the number of parallel batches which should be launched on slurm\n- `iters`: the number of div-enum iterations per batch  \n\nOther inputs are used for personalizing the directories and filenames on the cluster.\n\nAfter executing the script, the target directory should contain several bash files named `file_0.sh`, `file_1.sh` etc. depending on the `filenum` parameter that was provided.  \nIn addition, there should be one `runfiles.sh` file. This file contains the commands to submit the other files as job batches on the slurm cluster.\n\nThe results of a DEXOM run can be evaluated with the following scripts:  \n- `dexom_cluster_results.py`compiles and removes duplicate solutions from the results of a parallel DEXOM run.  \n- `pathway_enrichment.py` can be used to perform a pathway enrichment analysis using a one-sided hypergeometric test  \n- `result_functions.py` contains the `plot_pca` function, which performs Principal Component Analysis on the enumeration solutions\n\n## Examples\n\n### Toy models\nThe `toy_models.py` script contains code for generating some small metabolic models and reaction weights.  \nThe toy_models folder contains some ready-to-use models and reaction weight files.  \nThe `main.py` script contains a simple example of the DEXOM workflow using one of the toy models.\n\n### Recon 2.2\nThe example_data folder contains the model and the differential gene expression data which was used to test this new implementation.  \nIn order to produce reaction weights, you can call the `gpr_rules` script from the command line.  \nThis will create a file named "pval_0-01_reactionweights.csv" in the recon2v2 folder:  \n```\npython dexom_python/gpr_rules -m example_data/recon2v2_corrected.json -g example_data/pval_0-01_geneweights.csv -o example_data/pval_0-01_reactionweights\n```\n \nThen, call imat to produce a first context-specific subnetwork. This will create a file named "imat_solution.csv" in the example_data folder:  \n```\npython dexom_python/imat_functions.py -m example_data/recon2v2_corrected.json -r example_data/pval_0-01_reactionweights.csv -o example_data/imat_solution\n```\nTo run DEXOM on a slurm cluster, call the enumeration.py script to create the necessary batch files (here: 100 batches with 100 iterations).   \nBe careful to put the path to your installation of the CPLEX solver as the `-c` argument.   \nThis script assumes that you have cloned the `dexom-python` project on the cluster, which contains the `dexom_python` folder and the `example_data` folder in the same directory.  \nNote that this step creates a file called "recon2v2_reactions_shuffled.csv", which shows the order in which rxn-enum will call the reactions from the model.  \n```\npython dexom_python/cluster_utils/write_cluster_scripts.py -m example_data/recon2v2_corrected.json -r example_data/pval_0-01_reactionweights.csv -p example_data/imat_solution.csv -o example_data/ -n 100 -i 100 -c /home/mstingl/save/CPLEX_Studio1210/cplex/python/3.7/x86-64_linux\n```\nThen, submit the job to the slurm cluster.  \nNote that if you created the files on a Windows pc, you must use the command `dos2unix runfiles.sh` before `sbatch runfiles.sh`:  \n```\ncd example_data/\nsbatch runfiles.sh\ncd ..\n```\nAfter all jobs are completed, you can analyze the results using the following scripts:  \n```\npython dexom_python/cluster_utils/dexom_cluster_results.py -i example_data/ -o example_data/ -n 100\npython dexom_python/pathway_enrichment.py -s example_data/all_dexom_sols.csv -m example_data/recon2v2_corrected.json -o example_data/\npython dexom_python/result_functions.py -s example_data/all_dexom_sols.csv -o example_data/\n```\nThe file `all_dexom_sols.csv` contains all unique solutions enumerated with DEXOM.  \nThe file `output.txt` contains the average computation time per iteration and the proportion of duplicate solutions.  \nThe `.png` files contain boxplots of the pathway enrichment tests as well as a 2D PCA plot of the binary solution vectors.\n',
    'author': 'Maximilian Stingl',
    'author_email': 'maximilian.h.a.stingl@gmail.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://forgemia.inra.fr/metexplore/cbm/dexom-python',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7,<3.10',
}


setup(**setup_kwargs)
