# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_view_composer',
 'django_view_composer.migrations',
 'django_view_composer.templatetags']

package_data = \
{'': ['*']}

install_requires = \
['Django>=3.2,<4.0']

setup_kwargs = {
    'name': 'django-view-composer',
    'version': '0.1.0',
    'description': 'Django template tags to include views inside other views',
    'long_description': '# Django View Composer\n\nExtending the Django template system so that you can include a view inside another view\n\n- **Simple view composition** - based on template tags\n- **Reusable components** - turn any class based view into a reusable component\n- **Lightweight** - no framework, no core Javascript and no additional Python dependencies\n- **Reactive, if you want** - integrates nicely with HTMX\n\n```html+django\n{% load view_composer %}\n\n<nav>\n   {% view \'myapp.views.NavBar\' %}\n</nav>\n\n{% viewblock \'myapp.views.ListView\' %}\n   {% view \'myapp.views.ListFilter\' %}\n{% endviewblock %}\n```\n\n## Quick start\n\nDjango View Composer is released on PyPi so you can install using Pip:\n\n```sh\npip install django-view-composer\n```\n\nor Poetry:\n\n```sh\npoetry install django-view-composer\n```\n\nOnce installed, add to your `INSTALLED_APPS` setting to register the template tags:\n\n```python\nINSTALLED_APPS = [\n    ...\n    "django_view_composer",\n    ...\n]\n```\n\n## Using the `view` tag\n\nThe `{% view %}` tag renders a class based view and includes the content in the current template.\n\nIt has 1 required argument which is the import string for the view:\n\n```html+django\n{% load view_composer %}\n\n{% view \'myapp.views.MyView\' %}\n```\n\nYou can also provide a variable for the import string which will be resolved from the current template\'s context:\n\n```html+django\n{% view view_to_render %}\n```\n\n### Context variables\n\nSimilar to the `{% include %}` tag which operates on templates, the included view will be provided with the same context variables from the current template it is being rendered into.\n\nThese are provided as extra, so they won\'t replace any context variables the child view might be setting itself.\n\nIf any context variable names conflict, whatever the child view sets in it\'s own `get_context_data` will take precedence.\n\nYou can pass additional context from the template tag:\n\n```html+django\n{% view \'myapp.views.MyView\' with foo=\'bar\' %}\n```\n\nAdditional variables can be resolved from the current template\'s context to pass to the included view:\n\n```html+django\n{% view \'myapp.views.MyView\' with foo=foo %}\n```\n\n### Use `only` to limit context\n\nIf you want to render the included view only with the variables provided (or even no variables at all), use the only option. No other variables will be provided to the included view.\n\n```html+django\n{% view \'myapp.views.MyView\' with foo=\'bar\' only %}\n```\n\n## Using the `viewblock` tag\n\nThe `{% viewblock %}` tag renders a class based view and includes the content in the current template, but provides a block for additional nodes which are rendered first and made available in the included view’s context.\n\nThis tag must be closed with a corresponding endviewblock. It has 1 required argument which is the import string for the view:\n\n```html+django\n{% load view_composer %}\n\n{% viewblock \'myapp.views.MyView\' %}\n    <h2>An excellent view!</h2>\n{% endviewblock %}\n```\n\nIn the template for the `myapp.views.MyView`, you can use the children context variable to decide where to render the block content:\n\n```html+django\n<div>\n    {{ children }}\n</div>\n```\n\nContext variables are supported in the same way as the `view` tag.\nHowever, since the block content is rendered _before_ the included view, the additional nodes in the block can only access the current template\'s context - not the context of the view being included.\n\n## Nesting views\n\nTwo different forms of nesting are possible\n\n- you can put a `{% view %}` or a `{% viewblock %}` inside a `{% viewblock %}` in the same template\n- you can include a view where its own template includes other views\n\nViews are rendered in the order that the tags appear in the template.\n\nA `{% viewblock %}` renders the block content _first_ and then renders the view being included.\n\n## Handling POST views\n\nMost of the time the views being composed will have GET handlers which return a template response.\n\nOne of the powerful features of view composition (compared to template includes) is the ability to bring in additional logic, such as including a view which handles a form.\n\nHowever, included views are all rendered with the same HTTP request object which originates from the root view - the top most one which was handled by a URL pattern.\n\nIf you need to handle a different method in an included view, such as a child view that contains a form POST, you must\n\n- map the included view to a URL pattern as well\n- make the request to the view\'s direct URL, instead of the current URL, when it is submitted\n\nUsing the form as an example:\n\n```python\nfrom .views import ItemCreateView\n\napp_name = "myapp"\nurl_patterns = [\n    path("item/create", ItemCreateView.as_view(), name="item-create-view"),\n]\n```\n\n```html+django\n<form method="post" action="{% url \'myapp:item-create-view\' %}">\n    ...\n</form>\n```\n\nNow you can include this view inside another one like `{% view \'myapp.views.ItemCreateView\' %}` and when the form is POSTed it will send the request to the correct view.\n\n### Modify view dispatch\n\nIn some cases you might need to modify how the view is dispatched so it works well when included inside another (or several levels of) view.\n\nFor example, the Django generic editing views usually return a redirect response. You might want to return another blank instance of the form in the response to a successful POST, or you might want to return a confirmation with an \'Add another\' button to bring up a new form.\n\nDjango View Composer will not render anything if the view being included does not return a template response.\n\n## Reactive\n\nAn important concept in the view composer is that _the same views can be included via a template tag or rendered normally via a URL pattern_. \n\nThis provides a foundation for reactivity when coupled with [HTMX](https://htmx.org):\n\n- **initial render of the page** - including views via template tags\n- **reactive updates of individual views on a page** - calling the view\'s direct URL pattern and replacing the content in the page\n\nRemember, the view composer is not a framework - its just a template tag that lets you render a view inside another view.\nIt does not fundamentally change the Django request lifecycle even if you choose to include some reactivity with HTMX.\n\n## Running tests\n\nThere is a growing test suite which can be run \n\n```\n$ poetry install\n$ poetry shell\n$ cd tests\n$ ./manage.py test\n```\n\n## Contributing\n\nWelcome!',
    'author': 'Josh Higgins',
    'author_email': 'josh@joshh.info',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/joshiggins/django-view-composer',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
