"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Ingestion = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const constants_1 = require("./constants");
const ingestion_1 = require("./ingestion");
/**
 * The ingestion function receives messages from discovery integrations and
 * processes their payloads into the provided S3 Bucket.
 *
 * This function is also an `IGrantable`, so that it can be granted permissions
 * to read from the source S3 buckets.
 */
class Ingestion extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.queueRetentionPeriod = core_1.Duration.days(14);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        this.queue = new aws_sqs_1.Queue(this, 'Queue', {
            deadLetterQueue: {
                maxReceiveCount: 5,
                queue: this.deadLetterQueue,
            },
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: this.queueRetentionPeriod,
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        const handler = new ingestion_1.Ingestion(this, 'Default', {
            description: '[ConstructHub/Ingestion] Ingests new package versions into the Construct Hub',
            environment: {
                BUCKET_NAME: props.bucket.bucketName,
                STATE_MACHINE_ARN: props.orchestration.stateMachine.stateMachineArn,
                PACKAGE_LINKS: JSON.stringify((_a = props.packageLinks) !== null && _a !== void 0 ? _a : []),
            },
            logRetention: (_b = props.logRetention) !== null && _b !== void 0 ? _b : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.function = handler;
        props.bucket.grantWrite(this.function);
        props.orchestration.stateMachine.grantStartExecution(this.function);
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.queue, { batchSize: 1 }));
        // This event source is disabled, and can be used to re-process dead-letter-queue messages
        this.function.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.deadLetterQueue, { batchSize: 1, enabled: false }));
        this.grantPrincipal = this.function.grantPrincipal;
        props.monitoring.addHighSeverityAlarm('Ingestion Dead-Letter Queue not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.node.path}/DLQNotEmpty`,
            alarmDescription: [
                'The dead-letter queue for the Ingestion function is not empty!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            // SQS does not emit metrics if the queue has been empty for a while, which is GOOD.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
        props.monitoring.addHighSeverityAlarm('Ingestion failures', this.function.metricErrors().createAlarm(this, 'FailureAlarm', {
            alarmName: `${this.node.path}/Failure`,
            alarmDescription: [
                'The Ingestion function is failing!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to the function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // Lambda only emits metrics when the function is invoked. No invokation => no errors.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        }));
    }
    metricFoundLicenseFile(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "FoundLicenseFile" /* FOUND_LICENSE_FILE */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricIneligibleLicense(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "IneligibleLicense" /* INELIGIBLE_LICENSE */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricInvalidAssembly(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidAssembly" /* INVALID_ASSEMBLY */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    metricInvalidTarball(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "InvalidTarball" /* INVALID_TARBALL */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
    /**
     * This metrics is the total count of packages that were rejected due to
     * mismatched identity (name, version, license) between the `package.json`
     * file and te `.jsii` attribute.
     */
    metricMismatchedIdentityRejections(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */,
            namespace: constants_1.METRICS_NAMESPACE,
        });
    }
}
exports.Ingestion = Ingestion;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW5kZXguanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi8uLi8uLi9zcmMvYmFja2VuZC9pbmdlc3Rpb24vaW5kZXgudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsNERBQWlJO0FBRWpJLG9EQUF5RDtBQUN6RCxnRkFBbUU7QUFDbkUsZ0RBQWtEO0FBRWxELDhDQUFrRTtBQUNsRSx3Q0FBb0Q7QUFDcEQsK0NBQWlFO0FBRWpFLG1EQUFnRDtBQUdoRCwyQ0FBNEQ7QUFDNUQsMkNBQW1EO0FBZ0NuRDs7Ozs7O0dBTUc7QUFDSCxNQUFhLFNBQVUsU0FBUSxnQkFBUztJQW1CdEMsWUFBbUIsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBcUI7O1FBQ3BFLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFMSCx5QkFBb0IsR0FBRyxlQUFRLENBQUMsSUFBSSxDQUFDLEVBQUUsQ0FBQyxDQUFDO1FBT3ZELElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxlQUFLLENBQUMsSUFBSSxFQUFFLEtBQUssRUFBRTtZQUM1QyxVQUFVLEVBQUUseUJBQWUsQ0FBQyxXQUFXO1lBQ3ZDLGVBQWUsRUFBRSxJQUFJLENBQUMsb0JBQW9CO1lBQzFDLGlCQUFpQixFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsRUFBRSxDQUFDO1NBQ3hDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxLQUFLLEdBQUcsSUFBSSxlQUFLLENBQUMsSUFBSSxFQUFFLE9BQU8sRUFBRTtZQUNwQyxlQUFlLEVBQUU7Z0JBQ2YsZUFBZSxFQUFFLENBQUM7Z0JBQ2xCLEtBQUssRUFBRSxJQUFJLENBQUMsZUFBZTthQUM1QjtZQUNELFVBQVUsRUFBRSx5QkFBZSxDQUFDLFdBQVc7WUFDdkMsZUFBZSxFQUFFLElBQUksQ0FBQyxvQkFBb0I7WUFDMUMsaUJBQWlCLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7U0FDeEMsQ0FBQyxDQUFDO1FBRUgsTUFBTSxPQUFPLEdBQUcsSUFBSSxxQkFBTyxDQUFDLElBQUksRUFBRSxTQUFTLEVBQUU7WUFDM0MsV0FBVyxFQUFFLDhFQUE4RTtZQUMzRixXQUFXLEVBQUU7Z0JBQ1gsV0FBVyxFQUFFLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVTtnQkFDcEMsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsZUFBZTtnQkFDbkUsYUFBYSxFQUFFLElBQUksQ0FBQyxTQUFTLE9BQUMsS0FBSyxDQUFDLFlBQVksbUNBQUksRUFBRSxDQUFDO2FBQ3hEO1lBQ0QsWUFBWSxRQUFFLEtBQUssQ0FBQyxZQUFZLG1DQUFJLHdCQUFhLENBQUMsU0FBUztZQUMzRCxVQUFVLEVBQUUsS0FBTTtZQUNsQixPQUFPLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxFQUFFLENBQUM7WUFDN0IsT0FBTyxFQUFFLG9CQUFPLENBQUMsTUFBTTtTQUN4QixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsUUFBUSxHQUFHLE9BQU8sQ0FBQztRQUV4QixLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUM7UUFDdkMsS0FBSyxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsbUJBQW1CLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBRXBFLElBQUksQ0FBQyxRQUFRLENBQUMsY0FBYyxDQUFDLElBQUkseUNBQWMsQ0FBQyxJQUFJLENBQUMsS0FBSyxFQUFFLEVBQUUsU0FBUyxFQUFFLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUMvRSwwRkFBMEY7UUFDMUYsSUFBSSxDQUFDLFFBQVEsQ0FBQyxjQUFjLENBQUMsSUFBSSx5Q0FBYyxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsRUFBRSxTQUFTLEVBQUUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFFekcsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLENBQUMsUUFBUSxDQUFDLGNBQWMsQ0FBQztRQUVuRCxLQUFLLENBQUMsVUFBVSxDQUFDLG9CQUFvQixDQUNuQyx1Q0FBdUMsRUFDdkMsSUFBSSwrQkFBYyxDQUFDO1lBQ2pCLFVBQVUsRUFBRSxTQUFTO1lBQ3JCLFlBQVksRUFBRTtnQkFDWixFQUFFLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyx3Q0FBd0MsQ0FBQyxFQUFFLE1BQU0sRUFBRSxlQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7Z0JBQ2xHLEVBQUUsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLDJDQUEyQyxDQUFDLEVBQUUsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQzthQUN0RztTQUNGLENBQUMsQ0FBQyxXQUFXLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMvQixTQUFTLEVBQUUsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLElBQUksY0FBYztZQUMxQyxnQkFBZ0IsRUFBRTtnQkFDaEIsZ0VBQWdFO2dCQUNoRSxFQUFFO2dCQUNGLFlBQVkseUJBQVcsRUFBRTtnQkFDekIsRUFBRTtnQkFDRiw2QkFBNkIsdUJBQVcsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLEVBQUU7Z0JBQ2hFLGdDQUFnQyw2QkFBaUIsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7YUFDbkUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO1lBQ1osa0JBQWtCLEVBQUUsbUNBQWtCLENBQUMsa0NBQWtDO1lBQ3pFLGlCQUFpQixFQUFFLENBQUM7WUFDcEIsU0FBUyxFQUFFLENBQUM7WUFDWixvRkFBb0Y7WUFDcEYsZ0JBQWdCLEVBQUUsaUNBQWdCLENBQUMsYUFBYTtTQUNqRCxDQUFDLENBQ0gsQ0FBQztRQUNGLEtBQUssQ0FBQyxVQUFVLENBQUMsb0JBQW9CLENBQ25DLG9CQUFvQixFQUNwQixJQUFJLENBQUMsUUFBUSxDQUFDLFlBQVksRUFBRSxDQUFDLFdBQVcsQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQzdELFNBQVMsRUFBRSxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsSUFBSSxVQUFVO1lBQ3RDLGdCQUFnQixFQUFFO2dCQUNoQixvQ0FBb0M7Z0JBQ3BDLEVBQUU7Z0JBQ0YsWUFBWSx5QkFBVyxFQUFFO2dCQUN6QixFQUFFO2dCQUNGLGdDQUFnQyw2QkFBaUIsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUU7YUFDbkUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDO1lBQ1osa0JBQWtCLEVBQUUsbUNBQWtCLENBQUMsa0NBQWtDO1lBQ3pFLGlCQUFpQixFQUFFLENBQUM7WUFDcEIsU0FBUyxFQUFFLENBQUM7WUFDWixzRkFBc0Y7WUFDdEYsZ0JBQWdCLEVBQUUsaUNBQWdCLENBQUMsYUFBYTtTQUNqRCxDQUFDLENBQ0gsQ0FBQztJQUNKLENBQUM7SUFFTSxzQkFBc0IsQ0FBQyxJQUFvQjtRQUNoRCxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDM0IsU0FBUyxFQUFFLDBCQUFTLENBQUMsR0FBRztZQUN4QixHQUFHLElBQUk7WUFDUCxVQUFVLDZDQUErQjtZQUN6QyxTQUFTLEVBQUUsNkJBQWlCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSx1QkFBdUIsQ0FBQyxJQUFvQjtRQUNqRCxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDM0IsU0FBUyxFQUFFLDBCQUFTLENBQUMsR0FBRztZQUN4QixHQUFHLElBQUk7WUFDUCxVQUFVLDhDQUErQjtZQUN6QyxTQUFTLEVBQUUsNkJBQWlCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSxxQkFBcUIsQ0FBQyxJQUFvQjtRQUMvQyxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDM0IsU0FBUyxFQUFFLDBCQUFTLENBQUMsR0FBRztZQUN4QixHQUFHLElBQUk7WUFDUCxVQUFVLDBDQUE2QjtZQUN2QyxTQUFTLEVBQUUsNkJBQWlCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFTSxvQkFBb0IsQ0FBQyxJQUFvQjtRQUM5QyxPQUFPLElBQUksdUJBQU0sQ0FBQztZQUNoQixNQUFNLEVBQUUsZUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDM0IsU0FBUyxFQUFFLDBCQUFTLENBQUMsR0FBRztZQUN4QixHQUFHLElBQUk7WUFDUCxVQUFVLHdDQUE0QjtZQUN0QyxTQUFTLEVBQUUsNkJBQWlCO1NBQzdCLENBQUMsQ0FBQztJQUNMLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksa0NBQWtDLENBQUMsSUFBb0I7UUFDNUQsT0FBTyxJQUFJLHVCQUFNLENBQUM7WUFDaEIsTUFBTSxFQUFFLGVBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1lBQzNCLFNBQVMsRUFBRSwwQkFBUyxDQUFDLEdBQUc7WUFDeEIsR0FBRyxJQUFJO1lBQ1AsVUFBVSxxRUFBMkM7WUFDckQsU0FBUyxFQUFFLDZCQUFpQjtTQUM3QixDQUFDLENBQUM7SUFDTCxDQUFDO0NBQ0Y7QUFoS0QsOEJBZ0tDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgQ29tcGFyaXNvbk9wZXJhdG9yLCBNYXRoRXhwcmVzc2lvbiwgTWV0cmljLCBNZXRyaWNPcHRpb25zLCBTdGF0aXN0aWMsIFRyZWF0TWlzc2luZ0RhdGEgfSBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCc7XG5pbXBvcnQgeyBJR3JhbnRhYmxlLCBJUHJpbmNpcGFsIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSc7XG5pbXBvcnQgeyBJRnVuY3Rpb24sIFRyYWNpbmcgfSBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJztcbmltcG9ydCB7IFNxc0V2ZW50U291cmNlIH0gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYS1ldmVudC1zb3VyY2VzJztcbmltcG9ydCB7IFJldGVudGlvbkRheXMgfSBmcm9tICdAYXdzLWNkay9hd3MtbG9ncyc7XG5pbXBvcnQgeyBJQnVja2V0IH0gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IElRdWV1ZSwgUXVldWUsIFF1ZXVlRW5jcnlwdGlvbiB9IGZyb20gJ0Bhd3MtY2RrL2F3cy1zcXMnO1xuaW1wb3J0IHsgQ29uc3RydWN0LCBEdXJhdGlvbiB9IGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgbGFtYmRhRnVuY3Rpb25VcmwsIHNxc1F1ZXVlVXJsIH0gZnJvbSAnLi4vLi4vZGVlcC1saW5rJztcbmltcG9ydCB7IE1vbml0b3JpbmcgfSBmcm9tICcuLi8uLi9tb25pdG9yaW5nJztcbmltcG9ydCB7IFJVTkJPT0tfVVJMIH0gZnJvbSAnLi4vLi4vcnVuYm9vay11cmwnO1xuaW1wb3J0IHR5cGUgeyBQYWNrYWdlTGlua0NvbmZpZyB9IGZyb20gJy4uLy4uL3dlYmFwcCc7XG5pbXBvcnQgeyBPcmNoZXN0cmF0aW9uIH0gZnJvbSAnLi4vb3JjaGVzdHJhdGlvbic7XG5pbXBvcnQgeyBNZXRyaWNOYW1lLCBNRVRSSUNTX05BTUVTUEFDRSB9IGZyb20gJy4vY29uc3RhbnRzJztcbmltcG9ydCB7IEluZ2VzdGlvbiBhcyBIYW5kbGVyIH0gZnJvbSAnLi9pbmdlc3Rpb24nO1xuXG5leHBvcnQgaW50ZXJmYWNlIEluZ2VzdGlvblByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBidWNrZXQgaW4gd2hpY2ggaW5nZXN0ZWQgb2JqZWN0cyBhcmUgZHVlIHRvIGJlIGluc2VydGVkLlxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0OiBJQnVja2V0O1xuXG4gIC8qKlxuICAgKiBUaGUgbW9uaXRvcmluZyBoYW5kbGVyIHRvIHJlZ2lzdGVyIGFsYXJtcyB3aXRoLlxuICAgKi9cbiAgcmVhZG9ubHkgbW9uaXRvcmluZzogTW9uaXRvcmluZztcblxuICAvKipcbiAgICogVGhlIGJhY2tlbmQgb3JjaGVzdHJhdGlvbiB0byBpbnZva2Ugb25jZSB0aGUgcGFja2FnZSBtZXRhZGF0YSBoYXMgYmVlblxuICAgKiBzdWNjZXNzZnVsbHkgcmVnaXN0ZXJlZC5cbiAgICovXG4gIHJlYWRvbmx5IG9yY2hlc3RyYXRpb246IE9yY2hlc3RyYXRpb247XG5cbiAgLyoqXG4gICAqIEhvdyBsb25nIHRvIHJldGFpbiB0aGUgQ2xvdWRXYXRjaCBsb2dzLlxuICAgKlxuICAgKiBAZGVmYXVsdCBSZXRlbnRpb25EYXlzLlRFTl9ZRUFSU1xuICAgKi9cbiAgcmVhZG9ubHkgbG9nUmV0ZW50aW9uPzogUmV0ZW50aW9uRGF5cztcblxuICAvKipcbiAgICogQ29uZmlndXJhdGlvbiBmb3IgY3VzdG9tIHBhY2thZ2UgcGFnZSBsaW5rcy5cbiAgICovXG4gIHJlYWRvbmx5IHBhY2thZ2VMaW5rcz86IFBhY2thZ2VMaW5rQ29uZmlnW107XG59XG5cbi8qKlxuICogVGhlIGluZ2VzdGlvbiBmdW5jdGlvbiByZWNlaXZlcyBtZXNzYWdlcyBmcm9tIGRpc2NvdmVyeSBpbnRlZ3JhdGlvbnMgYW5kXG4gKiBwcm9jZXNzZXMgdGhlaXIgcGF5bG9hZHMgaW50byB0aGUgcHJvdmlkZWQgUzMgQnVja2V0LlxuICpcbiAqIFRoaXMgZnVuY3Rpb24gaXMgYWxzbyBhbiBgSUdyYW50YWJsZWAsIHNvIHRoYXQgaXQgY2FuIGJlIGdyYW50ZWQgcGVybWlzc2lvbnNcbiAqIHRvIHJlYWQgZnJvbSB0aGUgc291cmNlIFMzIGJ1Y2tldHMuXG4gKi9cbmV4cG9ydCBjbGFzcyBJbmdlc3Rpb24gZXh0ZW5kcyBDb25zdHJ1Y3QgaW1wbGVtZW50cyBJR3JhbnRhYmxlIHtcbiAgcHVibGljIHJlYWRvbmx5IGdyYW50UHJpbmNpcGFsOiBJUHJpbmNpcGFsO1xuXG4gIC8qKlxuICAgKiBUaGUgU1FTIHF1ZXVlIHRoYXQgdHJpZ2dlcnMgdGhlIGluZ2VzdGlvbiBmdW5jdGlvbi5cbiAgICovXG4gIHB1YmxpYyByZWFkb25seSBxdWV1ZTogSVF1ZXVlO1xuXG4gIC8qKlxuICAgKiBUaGUgaW5nZXN0aW9uIGRlYWQgbGV0dGVyIHF1ZXVlLCB3aGljaCB3aWxsIGhvbGQgbWVzc2FnZXMgdGhhdCBmYWlsZWRcbiAgICogaW5nZXN0aW9uIG9uZSB0b28gbWFueSB0aW1lcywgc28gdGhhdCBwb2lzb24gcGlsbHMgZG9uJ3QgZW5kbGVzc2x5IGNvbnN1bWVcbiAgICogcmVzb3VyY2VzLlxuICAgKi9cbiAgcHVibGljIHJlYWRvbmx5IGRlYWRMZXR0ZXJRdWV1ZTogSVF1ZXVlO1xuXG4gIHB1YmxpYyByZWFkb25seSBxdWV1ZVJldGVudGlvblBlcmlvZCA9IER1cmF0aW9uLmRheXMoMTQpO1xuXG4gIHB1YmxpYyByZWFkb25seSBmdW5jdGlvbjogSUZ1bmN0aW9uO1xuXG4gIHB1YmxpYyBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogSW5nZXN0aW9uUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgdGhpcy5kZWFkTGV0dGVyUXVldWUgPSBuZXcgUXVldWUodGhpcywgJ0RMUScsIHtcbiAgICAgIGVuY3J5cHRpb246IFF1ZXVlRW5jcnlwdGlvbi5LTVNfTUFOQUdFRCxcbiAgICAgIHJldGVudGlvblBlcmlvZDogdGhpcy5xdWV1ZVJldGVudGlvblBlcmlvZCxcbiAgICAgIHZpc2liaWxpdHlUaW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICB9KTtcblxuICAgIHRoaXMucXVldWUgPSBuZXcgUXVldWUodGhpcywgJ1F1ZXVlJywge1xuICAgICAgZGVhZExldHRlclF1ZXVlOiB7XG4gICAgICAgIG1heFJlY2VpdmVDb3VudDogNSxcbiAgICAgICAgcXVldWU6IHRoaXMuZGVhZExldHRlclF1ZXVlLFxuICAgICAgfSxcbiAgICAgIGVuY3J5cHRpb246IFF1ZXVlRW5jcnlwdGlvbi5LTVNfTUFOQUdFRCxcbiAgICAgIHJldGVudGlvblBlcmlvZDogdGhpcy5xdWV1ZVJldGVudGlvblBlcmlvZCxcbiAgICAgIHZpc2liaWxpdHlUaW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICB9KTtcblxuICAgIGNvbnN0IGhhbmRsZXIgPSBuZXcgSGFuZGxlcih0aGlzLCAnRGVmYXVsdCcsIHtcbiAgICAgIGRlc2NyaXB0aW9uOiAnW0NvbnN0cnVjdEh1Yi9Jbmdlc3Rpb25dIEluZ2VzdHMgbmV3IHBhY2thZ2UgdmVyc2lvbnMgaW50byB0aGUgQ29uc3RydWN0IEh1YicsXG4gICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICBCVUNLRVRfTkFNRTogcHJvcHMuYnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIFNUQVRFX01BQ0hJTkVfQVJOOiBwcm9wcy5vcmNoZXN0cmF0aW9uLnN0YXRlTWFjaGluZS5zdGF0ZU1hY2hpbmVBcm4sXG4gICAgICAgIFBBQ0tBR0VfTElOS1M6IEpTT04uc3RyaW5naWZ5KHByb3BzLnBhY2thZ2VMaW5rcyA/PyBbXSksXG4gICAgICB9LFxuICAgICAgbG9nUmV0ZW50aW9uOiBwcm9wcy5sb2dSZXRlbnRpb24gPz8gUmV0ZW50aW9uRGF5cy5URU5fWUVBUlMsXG4gICAgICBtZW1vcnlTaXplOiAxMF8yNDAsIC8vIEN1cnJlbnRseSB0aGUgbWF4aW11bSBwb3NzaWJsZSBzZXR0aW5nXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5taW51dGVzKDE1KSxcbiAgICAgIHRyYWNpbmc6IFRyYWNpbmcuQUNUSVZFLFxuICAgIH0pO1xuICAgIHRoaXMuZnVuY3Rpb24gPSBoYW5kbGVyO1xuXG4gICAgcHJvcHMuYnVja2V0LmdyYW50V3JpdGUodGhpcy5mdW5jdGlvbik7XG4gICAgcHJvcHMub3JjaGVzdHJhdGlvbi5zdGF0ZU1hY2hpbmUuZ3JhbnRTdGFydEV4ZWN1dGlvbih0aGlzLmZ1bmN0aW9uKTtcblxuICAgIHRoaXMuZnVuY3Rpb24uYWRkRXZlbnRTb3VyY2UobmV3IFNxc0V2ZW50U291cmNlKHRoaXMucXVldWUsIHsgYmF0Y2hTaXplOiAxIH0pKTtcbiAgICAvLyBUaGlzIGV2ZW50IHNvdXJjZSBpcyBkaXNhYmxlZCwgYW5kIGNhbiBiZSB1c2VkIHRvIHJlLXByb2Nlc3MgZGVhZC1sZXR0ZXItcXVldWUgbWVzc2FnZXNcbiAgICB0aGlzLmZ1bmN0aW9uLmFkZEV2ZW50U291cmNlKG5ldyBTcXNFdmVudFNvdXJjZSh0aGlzLmRlYWRMZXR0ZXJRdWV1ZSwgeyBiYXRjaFNpemU6IDEsIGVuYWJsZWQ6IGZhbHNlIH0pKTtcblxuICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSB0aGlzLmZ1bmN0aW9uLmdyYW50UHJpbmNpcGFsO1xuXG4gICAgcHJvcHMubW9uaXRvcmluZy5hZGRIaWdoU2V2ZXJpdHlBbGFybShcbiAgICAgICdJbmdlc3Rpb24gRGVhZC1MZXR0ZXIgUXVldWUgbm90IGVtcHR5JyxcbiAgICAgIG5ldyBNYXRoRXhwcmVzc2lvbih7XG4gICAgICAgIGV4cHJlc3Npb246ICdtMSArIG0yJyxcbiAgICAgICAgdXNpbmdNZXRyaWNzOiB7XG4gICAgICAgICAgbTE6IHRoaXMuZGVhZExldHRlclF1ZXVlLm1ldHJpY0FwcHJveGltYXRlTnVtYmVyT2ZNZXNzYWdlc1Zpc2libGUoeyBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMSkgfSksXG4gICAgICAgICAgbTI6IHRoaXMuZGVhZExldHRlclF1ZXVlLm1ldHJpY0FwcHJveGltYXRlTnVtYmVyT2ZNZXNzYWdlc05vdFZpc2libGUoeyBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoMSkgfSksXG4gICAgICAgIH0sXG4gICAgICB9KS5jcmVhdGVBbGFybSh0aGlzLCAnRExRQWxhcm0nLCB7XG4gICAgICAgIGFsYXJtTmFtZTogYCR7dGhpcy5ub2RlLnBhdGh9L0RMUU5vdEVtcHR5YCxcbiAgICAgICAgYWxhcm1EZXNjcmlwdGlvbjogW1xuICAgICAgICAgICdUaGUgZGVhZC1sZXR0ZXIgcXVldWUgZm9yIHRoZSBJbmdlc3Rpb24gZnVuY3Rpb24gaXMgbm90IGVtcHR5IScsXG4gICAgICAgICAgJycsXG4gICAgICAgICAgYFJ1bkJvb2s6ICR7UlVOQk9PS19VUkx9YCxcbiAgICAgICAgICAnJyxcbiAgICAgICAgICBgRGlyZWN0IGxpbmsgdG8gdGhlIHF1ZXVlOiAke3Nxc1F1ZXVlVXJsKHRoaXMuZGVhZExldHRlclF1ZXVlKX1gLFxuICAgICAgICAgIGBEaXJlY3QgbGluayB0byB0aGUgZnVuY3Rpb246ICR7bGFtYmRhRnVuY3Rpb25VcmwodGhpcy5mdW5jdGlvbil9YCxcbiAgICAgICAgXS5qb2luKCdcXG4nKSxcbiAgICAgICAgY29tcGFyaXNvbk9wZXJhdG9yOiBDb21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICAgICAgZXZhbHVhdGlvblBlcmlvZHM6IDEsXG4gICAgICAgIHRocmVzaG9sZDogMSxcbiAgICAgICAgLy8gU1FTIGRvZXMgbm90IGVtaXQgbWV0cmljcyBpZiB0aGUgcXVldWUgaGFzIGJlZW4gZW1wdHkgZm9yIGEgd2hpbGUsIHdoaWNoIGlzIEdPT0QuXG4gICAgICAgIHRyZWF0TWlzc2luZ0RhdGE6IFRyZWF0TWlzc2luZ0RhdGEuTk9UX0JSRUFDSElORyxcbiAgICAgIH0pLFxuICAgICk7XG4gICAgcHJvcHMubW9uaXRvcmluZy5hZGRIaWdoU2V2ZXJpdHlBbGFybShcbiAgICAgICdJbmdlc3Rpb24gZmFpbHVyZXMnLFxuICAgICAgdGhpcy5mdW5jdGlvbi5tZXRyaWNFcnJvcnMoKS5jcmVhdGVBbGFybSh0aGlzLCAnRmFpbHVyZUFsYXJtJywge1xuICAgICAgICBhbGFybU5hbWU6IGAke3RoaXMubm9kZS5wYXRofS9GYWlsdXJlYCxcbiAgICAgICAgYWxhcm1EZXNjcmlwdGlvbjogW1xuICAgICAgICAgICdUaGUgSW5nZXN0aW9uIGZ1bmN0aW9uIGlzIGZhaWxpbmchJyxcbiAgICAgICAgICAnJyxcbiAgICAgICAgICBgUnVuQm9vazogJHtSVU5CT09LX1VSTH1gLFxuICAgICAgICAgICcnLFxuICAgICAgICAgIGBEaXJlY3QgbGluayB0byB0aGUgZnVuY3Rpb246ICR7bGFtYmRhRnVuY3Rpb25VcmwodGhpcy5mdW5jdGlvbil9YCxcbiAgICAgICAgXS5qb2luKCdcXG4nKSxcbiAgICAgICAgY29tcGFyaXNvbk9wZXJhdG9yOiBDb21wYXJpc29uT3BlcmF0b3IuR1JFQVRFUl9USEFOX09SX0VRVUFMX1RPX1RIUkVTSE9MRCxcbiAgICAgICAgZXZhbHVhdGlvblBlcmlvZHM6IDIsXG4gICAgICAgIHRocmVzaG9sZDogMSxcbiAgICAgICAgLy8gTGFtYmRhIG9ubHkgZW1pdHMgbWV0cmljcyB3aGVuIHRoZSBmdW5jdGlvbiBpcyBpbnZva2VkLiBObyBpbnZva2F0aW9uID0+IG5vIGVycm9ycy5cbiAgICAgICAgdHJlYXRNaXNzaW5nRGF0YTogVHJlYXRNaXNzaW5nRGF0YS5OT1RfQlJFQUNISU5HLFxuICAgICAgfSksXG4gICAgKTtcbiAgfVxuXG4gIHB1YmxpYyBtZXRyaWNGb3VuZExpY2Vuc2VGaWxlKG9wdHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IE1ldHJpYyh7XG4gICAgICBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5TVU0sXG4gICAgICAuLi5vcHRzLFxuICAgICAgbWV0cmljTmFtZTogTWV0cmljTmFtZS5GT1VORF9MSUNFTlNFX0ZJTEUsXG4gICAgICBuYW1lc3BhY2U6IE1FVFJJQ1NfTkFNRVNQQUNFLFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIG1ldHJpY0luZWxpZ2libGVMaWNlbnNlKG9wdHM/OiBNZXRyaWNPcHRpb25zKTogTWV0cmljIHtcbiAgICByZXR1cm4gbmV3IE1ldHJpYyh7XG4gICAgICBwZXJpb2Q6IER1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgICBzdGF0aXN0aWM6IFN0YXRpc3RpYy5TVU0sXG4gICAgICAuLi5vcHRzLFxuICAgICAgbWV0cmljTmFtZTogTWV0cmljTmFtZS5JTkVMSUdJQkxFX0xJQ0VOU0UsXG4gICAgICBuYW1lc3BhY2U6IE1FVFJJQ1NfTkFNRVNQQUNFLFxuICAgIH0pO1xuICB9XG5cbiAgcHVibGljIG1ldHJpY0ludmFsaWRBc3NlbWJseShvcHRzPzogTWV0cmljT3B0aW9ucyk6IE1ldHJpYyB7XG4gICAgcmV0dXJuIG5ldyBNZXRyaWMoe1xuICAgICAgcGVyaW9kOiBEdXJhdGlvbi5taW51dGVzKDUpLFxuICAgICAgc3RhdGlzdGljOiBTdGF0aXN0aWMuU1VNLFxuICAgICAgLi4ub3B0cyxcbiAgICAgIG1ldHJpY05hbWU6IE1ldHJpY05hbWUuSU5WQUxJRF9BU1NFTUJMWSxcbiAgICAgIG5hbWVzcGFjZTogTUVUUklDU19OQU1FU1BBQ0UsXG4gICAgfSk7XG4gIH1cblxuICBwdWJsaWMgbWV0cmljSW52YWxpZFRhcmJhbGwob3B0cz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiBuZXcgTWV0cmljKHtcbiAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLlNVTSxcbiAgICAgIC4uLm9wdHMsXG4gICAgICBtZXRyaWNOYW1lOiBNZXRyaWNOYW1lLklOVkFMSURfVEFSQkFMTCxcbiAgICAgIG5hbWVzcGFjZTogTUVUUklDU19OQU1FU1BBQ0UsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVGhpcyBtZXRyaWNzIGlzIHRoZSB0b3RhbCBjb3VudCBvZiBwYWNrYWdlcyB0aGF0IHdlcmUgcmVqZWN0ZWQgZHVlIHRvXG4gICAqIG1pc21hdGNoZWQgaWRlbnRpdHkgKG5hbWUsIHZlcnNpb24sIGxpY2Vuc2UpIGJldHdlZW4gdGhlIGBwYWNrYWdlLmpzb25gXG4gICAqIGZpbGUgYW5kIHRlIGAuanNpaWAgYXR0cmlidXRlLlxuICAgKi9cbiAgcHVibGljIG1ldHJpY01pc21hdGNoZWRJZGVudGl0eVJlamVjdGlvbnMob3B0cz86IE1ldHJpY09wdGlvbnMpOiBNZXRyaWMge1xuICAgIHJldHVybiBuZXcgTWV0cmljKHtcbiAgICAgIHBlcmlvZDogRHVyYXRpb24ubWludXRlcyg1KSxcbiAgICAgIHN0YXRpc3RpYzogU3RhdGlzdGljLlNVTSxcbiAgICAgIC4uLm9wdHMsXG4gICAgICBtZXRyaWNOYW1lOiBNZXRyaWNOYW1lLk1JU01BVENIRURfSURFTlRJVFlfUkVKRUNUSU9OUyxcbiAgICAgIG5hbWVzcGFjZTogTUVUUklDU19OQU1FU1BBQ0UsXG4gICAgfSk7XG4gIH1cbn1cbiJdfQ==