"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const mockDistributionId = crypto_1.pseudoRandomBytes(16).toString('hex');
const mockPathPrefix = crypto_1.pseudoRandomBytes(16).toString('base64');
beforeEach(() => {
    process.env.DISTRIBUTION_ID = mockDistributionId;
    process.env.PATH_PREFIX = mockPathPrefix;
    AWSMock.setSDKInstance(AWS);
});
afterEach(() => {
    delete process.env.DISTRIBUTION_ID;
    delete process.env.PATH_PREFIX;
    AWSMock.restore();
});
const mockEvent = {
    Records: Array.from({ length: randomInt(2, 10) }, (_, index) => ({
        eventVersion: '1337.42',
        eventSource: 's3.test',
        awsRegion: 'bermuda-triangle-1',
        eventTime: new Date().toISOString(),
        eventName: 'PutObject:Test',
        userIdentity: { principalId: 'arn::test::principal' },
        requestParameters: { sourceIPAddress: '127.0.0.1' },
        responseElements: { 'x-amz-request-id': crypto_1.pseudoRandomBytes(16).toString('hex'), 'x-amz-id-2': crypto_1.pseudoRandomBytes(16).toString('hex') },
        s3: {
            s3SchemaVersion: '1337',
            configurationId: '42',
            bucket: { arn: 'arn:aws:s3:::phony-bucket-name', name: 'phony-bucket-name', ownerIdentity: { principalId: 'arn::test::principal' } },
            object: {
                key: crypto_1.pseudoRandomBytes(16).toString('base64'),
                size: randomInt(0, 1024),
                eTag: crypto_1.pseudoRandomBytes(16).toString('hex'),
                sequencer: `${index}-${crypto_1.pseudoRandomBytes(16).toString('hex')}`,
            },
        },
    })),
};
test('creates the expected invalidation', async () => {
    const mockInvalidationLocation = crypto_1.pseudoRandomBytes(16).toString('hex');
    AWSMock.mock('CloudFront', 'createInvalidation', (req, cb) => {
        try {
            expect(req.DistributionId).toBe(mockDistributionId);
            expect(req.InvalidationBatch.Paths.Quantity).toBe(mockEvent.Records.length);
            expect(req.InvalidationBatch.Paths.Items).toEqual(mockEvent.Records.map((record) => `${mockPathPrefix}${record.s3.object.key}`));
            expect(req.InvalidationBatch.CallerReference).toBe(mockEvent.Records.map((record) => record.s3.object.eTag).join(', '));
        }
        catch (e) {
            return cb(e);
        }
        cb(null, { Location: mockInvalidationLocation });
    });
    // Requiring the handler her to ensure it sees the expected environment variables
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const { handler } = require('../../../webapp/cache-invalidator/handler.lambda');
    return expect(handler(mockEvent, {})).resolves.toEqual({ Location: mockInvalidationLocation });
});
function randomInt(min, max) {
    return min + Math.ceil(Math.random() * (max - min));
}
//# sourceMappingURL=data:application/json;base64,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