"use strict";
var __classPrivateFieldGet = (this && this.__classPrivateFieldGet) || function (receiver, privateMap) {
    if (!privateMap.has(receiver)) {
        throw new TypeError("attempted to get private field on non-instance");
    }
    return privateMap.get(receiver);
};
var __classPrivateFieldSet = (this && this.__classPrivateFieldSet) || function (receiver, privateMap, value) {
    if (!privateMap.has(receiver)) {
        throw new TypeError("attempted to set private field on non-instance");
    }
    privateMap.set(receiver, value);
    return value;
};
var _repositoryNpmEndpoint, _s3BucketArn;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Repository = void 0;
const aws_codeartifact_1 = require("@aws-cdk/aws-codeartifact");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const custom_resources_1 = require("@aws-cdk/custom-resources");
/**
 * A CodeArtifact repository with an npmjs.com upstream connection.
 */
class Repository extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        _repositoryNpmEndpoint.set(this, void 0);
        _s3BucketArn.set(this, void 0);
        const domainName = (_a = props === null || props === void 0 ? void 0 : props.domainName) !== null && _a !== void 0 ? _a : this.node.addr;
        const domain = new aws_codeartifact_1.CfnDomain(this, 'Domain', { domainName });
        const repository = new aws_codeartifact_1.CfnRepository(this, 'Default', {
            description: props === null || props === void 0 ? void 0 : props.description,
            domainName: domain.attrName,
            externalConnections: ['public:npmjs'],
            repositoryName: (_b = props === null || props === void 0 ? void 0 : props.repositoryName) !== null && _b !== void 0 ? _b : domainName,
        });
        this.repositoryDomainArn = domain.attrArn;
        this.repositoryDomainName = repository.attrDomainName;
        this.repositoryDomainOwner = repository.attrDomainOwner;
        this.repositoryArn = repository.attrArn;
        this.repositoryName = repository.attrName;
    }
    /**
     * The npm repository endpoint to use for interacting with this repository.
     */
    get repositoryNpmEndpoint() {
        if (__classPrivateFieldGet(this, _repositoryNpmEndpoint) == null) {
            const serviceCall = {
                service: 'CodeArtifact',
                action: 'getRepositoryEndpoint',
                parameters: {
                    domain: this.repositoryDomainName,
                    domainOwner: this.repositoryDomainOwner,
                    format: 'npm',
                    repository: this.repositoryName,
                },
                physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse('repositoryEndpoint'),
            };
            const endpoint = new custom_resources_1.AwsCustomResource(this, 'GetEndpoint', {
                onCreate: serviceCall,
                onUpdate: serviceCall,
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: [this.repositoryArn] }),
                resourceType: 'Custom::CodeArtifactNpmRepositoryEndpoint',
            });
            __classPrivateFieldSet(this, _repositoryNpmEndpoint, endpoint.getResponseField('repositoryEndpoint'));
        }
        return __classPrivateFieldGet(this, _repositoryNpmEndpoint);
    }
    /**
     * The S3 bucket in which CodeArtifact stores the package data. When using
     * VPC Endpoints for CodeArtifact, an S3 Gateway Endpoint must also be
     * available, which allows reading from this bucket.
     */
    get s3BucketArn() {
        if (__classPrivateFieldGet(this, _s3BucketArn) == null) {
            const domainDescription = new custom_resources_1.AwsCustomResource(this, 'DescribeDomain', {
                onCreate: {
                    service: 'CodeArtifact',
                    action: 'describeDomain',
                    parameters: {
                        domain: this.repositoryDomainName,
                        domainOwner: this.repositoryDomainOwner,
                    },
                    physicalResourceId: custom_resources_1.PhysicalResourceId.fromResponse('domain.s3BucketArn'),
                },
                policy: custom_resources_1.AwsCustomResourcePolicy.fromSdkCalls({ resources: [this.repositoryDomainArn] }),
                resourceType: 'Custom::CoreArtifactDomainDescription',
            });
            __classPrivateFieldSet(this, _s3BucketArn, domainDescription.getResponseField('domain.s3BucketArn'));
        }
        return __classPrivateFieldGet(this, _s3BucketArn);
    }
    grantReadFromRepository(grantee) {
        // The Grant API does not allow conditions
        const stsGrantResult = grantee.grantPrincipal.addToPrincipalPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: ['sts:GetServiceBearerToken'],
            conditions: { StringEquals: { 'sts:AWSServiceName': 'codeartifact.amazonaws.com' } },
            resources: ['*'],
        }));
        if (!stsGrantResult.statementAdded) {
            return aws_iam_1.Grant.drop(grantee, 'CodeArtifact:ReadFromRepository');
        }
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: [
                'codeartifact:GetAuthorizationToken',
                'codeartifact:GetRepositoryEndpoint',
                'codeartifact:ReadFromRepository',
            ],
            resourceArns: [this.repositoryDomainArn, this.repositoryArn],
        });
    }
    /**
     * Obtains a view of this repository that is intended to be accessed though
     * VPC endpoints.
     *
     * @param api          an `InterfaceVpcEndpoint` to the `codeartifact.api`
     *                     service.
     * @param repositories an `InterfaceVpcEndpoint` to the
     *                    `codeartifact.repositories` service.
     *
     * @returns a view of this repository that appropriately grants permissions on
     *          the VPC endpoint policies, too.
     */
    throughVpcEndpoint(api, repositories) {
        return new Proxy(this, {
            get(target, property, _receiver) {
                if (property === 'grantReadFromRepository') {
                    return decoratedGrantReadFromRepository.bind(target);
                }
                return target[property];
            },
            getOwnPropertyDescriptor(target, property) {
                const realDescriptor = Object.getOwnPropertyDescriptor(target, property);
                if (property === 'grantReadFromRepository') {
                    return {
                        ...realDescriptor,
                        value: decoratedGrantReadFromRepository,
                        get: undefined,
                        set: undefined,
                    };
                }
                return realDescriptor;
            },
        });
        function decoratedGrantReadFromRepository(grantee) {
            const mainGrant = this.grantReadFromRepository(grantee);
            if (mainGrant.success) {
                api.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ['sts:GetServiceBearerToken'],
                    conditions: { StringEquals: { 'sts:AWSServiceName': 'codeartifact.amazonaws.com' } },
                    resources: ['*'],
                    principals: [grantee.grantPrincipal],
                }));
                api.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ['codeartifact:GetAuthorizationToken', 'codeartifact:GetRepositoryEndpoint'],
                    resources: [this.repositoryDomainArn, this.repositoryArn],
                    principals: [grantee.grantPrincipal],
                }));
                repositories.addToPolicy(new aws_iam_1.PolicyStatement({
                    effect: aws_iam_1.Effect.ALLOW,
                    actions: ['codeartifact:ReadFromRepository'],
                    resources: [this.repositoryArn],
                    principals: [grantee.grantPrincipal],
                }));
            }
            return mainGrant;
        }
    }
}
exports.Repository = Repository;
_repositoryNpmEndpoint = new WeakMap(), _s3BucketArn = new WeakMap();
//# sourceMappingURL=data:application/json;base64,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