"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const catalog_builder_1 = require("../catalog-builder");
const constants_1 = require("../shared/constants");
const language_1 = require("../shared/language");
const transliterator_1 = require("../transliterator");
const redrive_state_machine_1 = require("./redrive-state-machine");
const SUPPORTED_LANGUAGES = [
    language_1.DocumentationLanguage.PYTHON,
    language_1.DocumentationLanguage.TYPESCRIPT,
    language_1.DocumentationLanguage.JAVA,
    language_1.DocumentationLanguage.CSHARP,
];
/**
 * This retry policy is used for all items in the state machine and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * After 30 attempts, given the parameters, the last attempt will wait just
 * under 16 minutes, which should be enough for currently running Lambda
 * functions to complete (or time out after 15 minutes). The total time spent
 * waiting between retries by this time is just over 3 hours. This is a lot of
 * time, but in extreme burst situations (i.e: reprocessing everything), this
 * is actually a good thing.
 */
const THROTTLE_RETRY_POLICY = { backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 };
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            label: 'Dead-Letter Queue not empty',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        });
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props);
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder.function,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" throttled', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['Lambda.TooManyRequestsException'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" failure', {
            parameters: { 'error.$': 'States.StringToJson($.Cause)' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.TaskFailed'] })
            .addCatch(new aws_stepfunctions_1.Pass(this, '"Add to catalog.json" fault', {
            parameters: { 'error.$': '$.Cause' },
            resultPath: '$.error',
        }).next(sendToDeadLetterQueue), { errors: ['States.ALL'] });
        const docGenResultsKey = 'DocGen';
        const sendToDlqIfNeeded = new aws_stepfunctions_1.Choice(this, 'Any Failure?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isPresent(`$.${docGenResultsKey}[${i}].error`))), sendToDeadLetterQueue)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success'));
        this.ecsCluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            containerInsights: true,
            enableFargateCapacityProviders: true,
            vpc: props.vpc,
        });
        this.transliterator = new transliterator_1.Transliterator(this, 'Transliterator', props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        }).next(new aws_stepfunctions_1.Parallel(this, 'DocGen', { resultPath: `$.${docGenResultsKey}` })
            .branch(...SUPPORTED_LANGUAGES.map((language) => {
            return new aws_stepfunctions_1.Choice(this, `Is ${language} needed?`)
                .when(aws_stepfunctions_1.Condition.or(aws_stepfunctions_1.Condition.isNotPresent('$.languages'), aws_stepfunctions_1.Condition.and(aws_stepfunctions_1.Condition.isPresent(`$.languages.${language}`), aws_stepfunctions_1.Condition.booleanEquals(`$.languages.${language}`, true))), 
            // We have to prepare the input to be a JSON string, within an array, because the ECS task integration expects
            // an array of strings (the model if that of a CLI invocation).
            // Unfortunately, we have to split this in two Pass states, because I don't know how to make it work otherwise.
            new aws_stepfunctions_1.Pass(this, `Prepare ${language}`, {
                parameters: { command: { 'bucket.$': '$.bucket', 'assembly.$': '$.assembly', 'package.$': '$.package', '$TaskExecution.$': '$.$TaskExecution' } },
                resultPath: '$',
            })
                .next(new aws_stepfunctions_1.Pass(this, `Stringify ${language} input`, {
                parameters: { 'commands.$': 'States.Array(States.JsonToString($.command))' },
                resultPath: '$',
            })
                .next(this.transliterator.createEcsRunTask(this, `Generate ${language} docs`, {
                cluster: this.ecsCluster,
                inputPath: '$.commands',
                language,
                resultSelector: { result: { 'language': language.toString(), 'success.$': '$' } },
                // Expect this to complete within one hour
                timeout: core_1.Duration.hours(1),
                vpcSubnets: props.vpcSubnets,
            })
                // Do not retry NoSpaceLeftOnDevice errors, these are typically not transient.
                .addRetry({ errors: ['jsii-docgen.NoSpaceLeftOnDevice'], maxAttempts: 0 })
                .addRetry({
                errors: [
                    'ECS.AmazonECSException',
                    'ECS.InvalidParameterException',
                    'jsii-docgen.NpmError.E429',
                    'jsii-codgen.NpmError.EPROTO',
                ],
                ...THROTTLE_RETRY_POLICY,
            })
                .addRetry({
                errors: ['jsii-docgen.NpmError.ETARGET'],
                // We'll wait longer between retries. This is to account for CodeArtifact's lag behind npm
                backoffRate: 2,
                interval: core_1.Duration.minutes(5),
                maxAttempts: 3,
            })
                .addRetry({ maxAttempts: 3 })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" timed out`, { parameters: { error: 'Timed out!', language: language.toString() } }), { errors: ['States.Timeout'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" service error`, { parameters: { 'error.$': '$.Cause', 'language': language.toString() } }), { errors: ['ECS.AmazonECSException', 'ECS.InvalidParameterException'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" failure`, { parameters: { 'error.$': 'States.StringToJson($.Cause)', 'language': language.toString() } }), { errors: ['States.TaskFailed'] })
                .addCatch(new aws_stepfunctions_1.Pass(this, `"Generate ${language} docs" fault`, { parameters: { 'error.$': '$.Cause', 'language': language.toString() } }), { errors: ['States.ALL'] }))))
                .otherwise(new aws_stepfunctions_1.Pass(this, `Skip ${language}`, { result: aws_stepfunctions_1.Result.fromObject({ language: language.toString(), success: 'NOOP (Skipped)' }), resultPath: '$' }));
        }))
            .next(new aws_stepfunctions_1.Choice(this, 'Any Success?')
            .when(aws_stepfunctions_1.Condition.or(...SUPPORTED_LANGUAGES.map((_, i) => aws_stepfunctions_1.Condition.isNotPresent(`$.${docGenResultsKey}[${i}].error`))), addToCatalog.next(sendToDlqIfNeeded))
            .otherwise(sendToDlqIfNeeded)));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // The workflow is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the orchestrator.
        this.regenerateAllDocumentation = new RegenerateAllDocumentation(this, 'RegenerateAllDocumentation', {
            bucket: props.bucket,
            stateMachine: this.stateMachine,
        }).stateMachine;
    }
    metricEcsTaskCount(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'TaskCount',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % CPU utilization from the CPU units utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mCpuUtilized, 0) / FILL(mCpuReserved, REPEAT)',
            usingMetrics: {
                mCpuReserved: this.metricEcsCpuReserved(),
                mCpuUtilized: this.metricEcsCpuUtilized(),
            },
        });
    }
    metricEcsMemoryReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % memory utilization from the RAM utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mMemoryUtilized, 0) / FILL(mMemoryReserved, REPEAT)',
            usingMetrics: {
                mMemoryReserved: this.metricEcsMemoryReserved(),
                mMemoryUtilized: this.metricEcsMemoryUtilized(),
            },
        });
    }
    metricEcsNetworkRxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkRxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsNetworkTxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkTxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
}
exports.Orchestration = Orchestration;
class RegenerateAllDocumentation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const processVersions = new aws_stepfunctions_1.Choice(this, 'Get package versions page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each key prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Start Orchestration Workflow', {
            stateMachine: props.stateMachine,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it.
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                bucket: props.bucket.bucketName,
                assembly: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.ASSEMBLY_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                metadata: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.METADATA_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                package: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.PACKAGE_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
        })));
        processVersions.next(new aws_stepfunctions_1.Choice(this, 'Has more versions?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processVersions)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success')));
        const processPackageVersions = new aws_stepfunctions_1.StateMachine(this, 'PerPackage', {
            definition: processVersions,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This workflow is broken into two sub-workflows because otherwise it hits the 25K events limit
        // of StepFunction executions relatively quickly.
        const processNamespace = new aws_stepfunctions_1.Choice(this, 'Get @scope page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each @scope/pkg', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Process scoped package', {
            stateMachine: processPackageVersions,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it,
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        })));
        processNamespace.next(new aws_stepfunctions_1.Choice(this, 'Has more packages?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processNamespace)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done')));
        const start = new aws_stepfunctions_1.Choice(this, 'Get prefix page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next prefixes page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }))
            .otherwise(new tasks.CallAwsService(this, 'First prefix page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        })).afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new aws_stepfunctions_1.Choice(this, 'Is this a @scope/ prefix?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Prefix', `${constants_1.STORAGE_KEY_PREFIX}@*`), processNamespace)
            .otherwise(new tasks.StepFunctionsStartExecution(this, 'Process unscoped package', {
            stateMachine: processPackageVersions,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                // Associate the child workflow with the execution that started it,
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: aws_stepfunctions_1.JsonPath.stringAt('$$.Execution.Id'),
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }))
            .afterwards()));
        start.next(new aws_stepfunctions_1.Choice(this, 'Has more prefixes?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), start)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition: start,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.hours(4),
            tracingEnabled: true,
        });
        props.bucket.grantRead(processPackageVersions);
        props.bucket.grantRead(this.stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,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