"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const assert_1 = require("@aws-cdk/assert");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const backend_1 = require("../../../backend");
const create_map_1 = require("../../../backend/deny-list/create-map");
const monitoring_1 = require("../../../monitoring");
const catalog_builder_mock_1 = require("./integ/catalog-builder-mock");
test('defaults - empty deny list', () => {
    const stack = new core_1.Stack();
    const denyList = new backend_1.DenyList(stack, 'DenyList', {
        rules: [],
        monitoring: new monitoring_1.Monitoring(stack, 'Monitoring'),
        packageDataBucket: new s3.Bucket(stack, 'PackageDataBucket'),
        packageDataKeyPrefix: 'my-data/',
    });
    denyList.prune.onChangeInvoke(new catalog_builder_mock_1.CatalogBuilderMock(stack, 'CatalogBuilderMock'));
    // THEN
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('pruneOnChange is disabled', () => {
    const stack = new core_1.Stack();
    const denyList = new backend_1.DenyList(stack, 'DenyList', {
        rules: [],
        monitoring: new monitoring_1.Monitoring(stack, 'Monitoring'),
        packageDataBucket: new s3.Bucket(stack, 'PackageDataBucket'),
        packageDataKeyPrefix: 'my-data/',
        pruneOnChange: false,
    });
    denyList.prune.onChangeInvoke(new catalog_builder_mock_1.CatalogBuilderMock(stack, 'CatalogBuilderMock'));
    // THEN
    expect(stack).not.toHaveResource('Custom::S3BucketNotifications');
});
test('prunePeriod controls period', () => {
    const stack = new core_1.Stack();
    new backend_1.DenyList(stack, 'DenyList', {
        rules: [],
        monitoring: new monitoring_1.Monitoring(stack, 'Monitoring'),
        packageDataBucket: new s3.Bucket(stack, 'PackageDataBucket'),
        packageDataKeyPrefix: 'my-data/',
        prunePeriod: core_1.Duration.minutes(10),
    });
    // THEN
    expect(stack).toHaveResource('AWS::Events::Rule', {
        ScheduleExpression: 'rate(10 minutes)',
    });
});
test('prunePeriod of zero disables periodical pruning', () => {
    const stack = new core_1.Stack();
    new backend_1.DenyList(stack, 'DenyList', {
        rules: [],
        monitoring: new monitoring_1.Monitoring(stack, 'Monitoring'),
        packageDataBucket: new s3.Bucket(stack, 'PackageDataBucket'),
        packageDataKeyPrefix: 'my-data/',
        prunePeriod: core_1.Duration.minutes(0),
    });
    // THEN
    expect(stack).not.toHaveResource('AWS::Events::Rule');
});
describe('createDenyListMap()', () => {
    test('no rules', () => {
        const rules = [];
        expect(create_map_1.createDenyListMap(rules)).toEqual({});
    });
    test('rule with "package"', () => {
        const rules = [
            { package: 'my-package', reason: 'my reason' },
        ];
        expect(create_map_1.createDenyListMap(rules)).toEqual({
            'my-package': { package: 'my-package', reason: 'my reason' },
        });
    });
    test('rule with scoped "package"', () => {
        const rules = [
            { package: '@my-scope/my-package', reason: 'my reason' },
        ];
        expect(create_map_1.createDenyListMap(rules)).toEqual({
            '@my-scope/my-package': { package: '@my-scope/my-package', reason: 'my reason' },
        });
    });
    test('rule with "package" and "version"', () => {
        const rules = [
            { package: 'my-package', version: '1.2.3', reason: 'my reason 1.2.3' },
        ];
        expect(create_map_1.createDenyListMap(rules)).toEqual({
            'my-package/v1.2.3': { package: 'my-package', version: '1.2.3', reason: 'my reason 1.2.3' },
        });
    });
    test('fail for duplicate rules for the same package + version', () => {
        const rules = [
            { package: 'my-package', version: '1.2.3', reason: 'my reason 1.2.3' },
            { package: 'my-package', version: '1.2.3', reason: 'your reason' },
        ];
        expect(() => create_map_1.createDenyListMap(rules)).toThrow(/Duplicate deny list entry: my-package\/v1\.2\.3/);
    });
    test('fails for duplicate rules for p+v,p (in that order)', () => {
        const rules = [
            { package: 'my-package', version: '1.2.3', reason: 'only my-package@1.2.3 is blocked' },
            { package: 'my-package', version: '3.4.5', reason: 'only my-package@3.4.5 is blocked' },
            { package: 'my-package', reason: 'all versions of my-package are denied' },
        ];
        expect(() => create_map_1.createDenyListMap(rules)).toThrow(/Found rules that match specific versions of \"my-package\" \(1\.2\.3,3\.4\.5\) but there is also a rule that matches all versions/);
    });
    test('fails for duplicate rules for p,p+v,p+v (in that order)', () => {
        const rules = [
            { package: 'my-package', reason: 'all versions of my-package are denied' },
            { package: 'my-package', version: '1.2.3', reason: 'only my-package@1.2.3 is blocked' },
            { package: 'my-package', version: '3.4.5', reason: 'only my-package@3.4.5 is blocked' },
        ];
        expect(() => create_map_1.createDenyListMap(rules)).toThrow(/Found rules that match specific versions of \"my-package\" \(1\.2\.3,3\.4\.5\) but there is also a rule that matches all versions/);
    });
    test('fails for duplicate rules for p+v,p,p+v (in that order)', () => {
        const rules = [
            { package: 'my-package', version: '1.2.3', reason: 'only my-package@1.2.3 is blocked' },
            { package: 'my-package', reason: 'all versions of my-package are denied' },
            { package: 'my-package', version: '3.4.5', reason: 'only my-package@3.4.5 is blocked' },
        ];
        expect(() => create_map_1.createDenyListMap(rules)).toThrow(/Found rules that match specific versions of \"my-package\" \(1\.2\.3,3\.4\.5\) but there is also a rule that matches all versions/);
    });
});
//# sourceMappingURL=data:application/json;base64,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