"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const events_1 = require("events");
const spec_1 = require("@jsii/spec");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const aws_lambda_shared_1 = require("../../../backend/shared/aws.lambda-shared");
const constants = require("../../../backend/shared/constants");
jest.mock('zlib');
jest.mock('aws-embedded-metrics');
jest.mock('tar-stream');
jest.mock('../../../backend/shared/env.lambda-shared');
// eslint-disable-next-line @typescript-eslint/no-require-imports
const mockMetricScope = require('aws-embedded-metrics').metricScope;
const mockPutMetric = jest.fn().mockName('MetricsLogger.putMetric');
const mockMetrics = {
    putMetric: mockPutMetric,
    setDimensions: (...args) => expect(args).toEqual([]),
};
mockMetricScope.mockImplementation((cb) => {
    const impl = cb(mockMetrics);
    return async (...args) => impl(...args);
});
beforeEach((done) => {
    AWSMock.setSDKInstance(AWS);
    done();
});
afterEach((done) => {
    AWSMock.restore();
    aws_lambda_shared_1.reset();
    done();
});
test('basic happy case', async () => {
    const mockBucketName = 'fake-bucket';
    const mockStateMachineArn = 'fake-state-machine-arn';
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockRequireEnv = require('../../../backend/shared/env.lambda-shared').requireEnv;
    mockRequireEnv.mockImplementation((name) => {
        if (name === 'BUCKET_NAME') {
            return mockBucketName;
        }
        if (name === 'STATE_MACHINE_ARN') {
            return mockStateMachineArn;
        }
        throw new Error(`Bad environment variable: "${name}"`);
    });
    const stagingBucket = 'staging-bucket';
    const stagingKey = 'staging-key';
    const stagingVersion = 'staging-version-id';
    const fakeTarGz = Buffer.from('fake-tarball-content[gzipped]');
    const fakeTar = Buffer.from('fake-tarball-content');
    const tarballUri = `s3://${stagingBucket}.test-bermuda-2.s3.amazonaws.com/${stagingKey}?versionId=${stagingVersion}`;
    const time = '2021-07-12T15:18:00.000000+02:00';
    const integrity = 'sha256-1RyNs3cDpyTqBMqJIiHbCpl8PEN6h3uWx3lzF+3qcmY=';
    const packageName = '@package-scope/package-name';
    const packageVersion = '1.2.3-pre.4';
    const packageLicense = 'Apache-2.0';
    const fakeDotJsii = JSON.stringify(fakeAssembly(packageName, packageVersion, packageLicense));
    const context = {
        awsRequestId: 'Fake-Request-ID',
        logGroupName: 'Fake-Log-Group',
        logStreamName: 'Fake-Log-Stream',
    };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(stagingBucket);
            expect(req.Key).toBe(stagingKey);
            expect(req.VersionId).toBe(stagingVersion);
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Body: fakeTarGz });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockCreateGunzip = require('zlib').createGunzip;
    mockCreateGunzip.mockImplementation(() => new FakeGunzip(fakeTarGz, fakeTar));
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockExtract = require('tar-stream').extract;
    mockExtract.mockImplementation(() => new FakeExtract(fakeTar, {
        'package/.jsii': fakeDotJsii,
        'package/index.js': '// Ignore me!',
        'package/package.json': JSON.stringify({ name: packageName, version: packageVersion, license: packageLicense }),
    }));
    let mockTarballCreated = false;
    let mockMetadataCreated = false;
    const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b, _c;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect((_a = req.Metadata) === null || _a === void 0 ? void 0 : _a['Lambda-Log-Group']).toBe(context.logGroupName);
            expect((_b = req.Metadata) === null || _b === void 0 ? void 0 : _b['Lambda-Log-Stream']).toBe(context.logStreamName);
            expect((_c = req.Metadata) === null || _c === void 0 ? void 0 : _c['Lambda-Run-Id']).toBe(context.awsRequestId);
            switch (req.Key) {
                case assemblyKey:
                    expect(req.ContentType).toBe('application/json');
                    expect(req.Body).toEqual(Buffer.from(fakeDotJsii));
                    // Must be created strictly after the tarball and metadata files have been uploaded.
                    expect(mockTarballCreated && mockMetadataCreated).toBeTruthy();
                    break;
                case metadataKey:
                    expect(req.ContentType).toBe('application/json');
                    expect(Buffer.from(req.Body)).toEqual(Buffer.from(JSON.stringify({ date: time })));
                    mockMetadataCreated = true;
                    break;
                case packageKey:
                    expect(req.ContentType).toBe('application/octet-stream');
                    expect(req.Body).toEqual(fakeTarGz);
                    mockTarballCreated = true;
                    break;
                default:
                    fail(`Unexpected key: "${req.Key}"`);
            }
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { VersionId: `${req.Key}-NewVersion` });
    });
    const executionArn = 'Fake-Execution-Arn';
    AWSMock.mock('StepFunctions', 'startExecution', (req, cb) => {
        try {
            expect(req.stateMachineArn).toBe(mockStateMachineArn);
            expect(JSON.parse(req.input)).toEqual({
                bucket: mockBucketName,
                assembly: { key: assemblyKey, versionId: `${assemblyKey}-NewVersion` },
                metadata: { key: metadataKey, versionId: `${metadataKey}-NewVersion` },
                package: { key: packageKey, versionId: `${packageKey}-NewVersion` },
            });
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { executionArn, startDate: new Date() });
    });
    const event = {
        Records: [{
                attributes: {},
                awsRegion: 'test-bermuda-1',
                body: JSON.stringify({ tarballUri, integrity, time }),
                eventSource: 'sqs',
                eventSourceARN: 'arn:aws:sqs:test-bermuda-1:123456789012:fake',
                md5OfBody: 'Fake-MD5-Of-Body',
                messageAttributes: {},
                messageId: 'Fake-Message-ID',
                receiptHandle: 'Fake-Receipt-Handke',
            }],
    };
    // We require the handler here so that any mocks to metricScope are set up
    // prior to the handler being created.
    //
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    await expect(require('../../../backend/ingestion/ingestion.lambda').handler(event, context))
        .resolves.toEqual([executionArn]);
    expect(mockPutMetric).toHaveBeenCalledWith("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, 'Count');
    expect(mockPutMetric).toHaveBeenCalledWith("FoundLicenseFile" /* FOUND_LICENSE_FILE */, 0, 'Count');
});
test('basic happy case with license file', async () => {
    const mockBucketName = 'fake-bucket';
    const mockStateMachineArn = 'fake-state-machine-arn';
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockRequireEnv = require('../../../backend/shared/env.lambda-shared').requireEnv;
    mockRequireEnv.mockImplementation((name) => {
        if (name === 'BUCKET_NAME') {
            return mockBucketName;
        }
        if (name === 'STATE_MACHINE_ARN') {
            return mockStateMachineArn;
        }
        throw new Error(`Bad environment variable: "${name}"`);
    });
    const stagingBucket = 'staging-bucket';
    const stagingKey = 'staging-key';
    const stagingVersion = 'staging-version-id';
    const fakeTarGz = Buffer.from('fake-tarball-content[gzipped]');
    const fakeTar = Buffer.from('fake-tarball-content');
    const fakeLicense = 'inscrutable-legalese';
    const tarballUri = `s3://${stagingBucket}.test-bermuda-2.s3.amazonaws.com/${stagingKey}?versionId=${stagingVersion}`;
    const time = '2021-07-12T15:18:00.000000+02:00';
    const integrity = 'sha256-1RyNs3cDpyTqBMqJIiHbCpl8PEN6h3uWx3lzF+3qcmY=';
    const packageName = '@package-scope/package-name';
    const packageVersion = '1.2.3-pre.4';
    const packageLicense = 'Apache-2.0';
    const fakeDotJsii = JSON.stringify(fakeAssembly(packageName, packageVersion, packageLicense));
    const context = {
        awsRequestId: 'Fake-Request-ID',
        logGroupName: 'Fake-Log-Group',
        logStreamName: 'Fake-Log-Stream',
    };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(stagingBucket);
            expect(req.Key).toBe(stagingKey);
            expect(req.VersionId).toBe(stagingVersion);
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Body: fakeTarGz });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockCreateGunzip = require('zlib').createGunzip;
    mockCreateGunzip.mockImplementation(() => new FakeGunzip(fakeTarGz, fakeTar));
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockExtract = require('tar-stream').extract;
    mockExtract.mockImplementation(() => new FakeExtract(fakeTar, {
        'package/.jsii': fakeDotJsii,
        'package/LICENSE.md': fakeLicense,
        'package/index.js': '// Ignore me!',
        'package/package.json': JSON.stringify({ name: packageName, version: packageVersion, license: packageLicense }),
    }));
    let mockTarballCreated = false;
    let mockMetadataCreated = false;
    const { assemblyKey, metadataKey, packageKey } = constants.getObjectKeys(packageName, packageVersion);
    AWSMock.mock('S3', 'putObject', (req, cb) => {
        var _a, _b, _c;
        try {
            expect(req.Bucket).toBe(mockBucketName);
            expect((_a = req.Metadata) === null || _a === void 0 ? void 0 : _a['Lambda-Log-Group']).toBe(context.logGroupName);
            expect((_b = req.Metadata) === null || _b === void 0 ? void 0 : _b['Lambda-Log-Stream']).toBe(context.logStreamName);
            expect((_c = req.Metadata) === null || _c === void 0 ? void 0 : _c['Lambda-Run-Id']).toBe(context.awsRequestId);
            switch (req.Key) {
                case assemblyKey:
                    expect(req.ContentType).toBe('application/json');
                    expect(req.Body).toEqual(Buffer.from(fakeDotJsii));
                    // Must be created strictly after the tarball and metadata files have been uploaded.
                    expect(mockTarballCreated && mockMetadataCreated).toBeTruthy();
                    break;
                case metadataKey:
                    expect(req.ContentType).toBe('application/json');
                    expect(Buffer.from(req.Body)).toEqual(Buffer.from(JSON.stringify({ date: time, licenseText: fakeLicense })));
                    mockMetadataCreated = true;
                    break;
                case packageKey:
                    expect(req.ContentType).toBe('application/octet-stream');
                    expect(req.Body).toEqual(fakeTarGz);
                    mockTarballCreated = true;
                    break;
                default:
                    fail(`Unexpected key: "${req.Key}"`);
            }
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { VersionId: `${req.Key}-NewVersion` });
    });
    const executionArn = 'Fake-Execution-Arn';
    AWSMock.mock('StepFunctions', 'startExecution', (req, cb) => {
        try {
            expect(req.stateMachineArn).toBe(mockStateMachineArn);
            expect(JSON.parse(req.input)).toEqual({
                bucket: mockBucketName,
                assembly: { key: assemblyKey, versionId: `${assemblyKey}-NewVersion` },
                metadata: { key: metadataKey, versionId: `${metadataKey}-NewVersion` },
                package: { key: packageKey, versionId: `${packageKey}-NewVersion` },
            });
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, {
            executionArn,
            startDate: new Date(),
        });
    });
    const event = {
        Records: [{
                attributes: {},
                awsRegion: 'test-bermuda-1',
                body: JSON.stringify({ tarballUri, integrity, time }),
                eventSource: 'sqs',
                eventSourceARN: 'arn:aws:sqs:test-bermuda-1:123456789012:fake',
                md5OfBody: 'Fake-MD5-Of-Body',
                messageAttributes: {},
                messageId: 'Fake-Message-ID',
                receiptHandle: 'Fake-Receipt-Handke',
            }],
    };
    // We require the handler here so that any mocks to metricScope are set up
    // prior to the handler being created.
    //
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    await expect(require('../../../backend/ingestion/ingestion.lambda').handler(event, context))
        .resolves.toEqual([executionArn]);
    expect(mockPutMetric).toHaveBeenCalledWith("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 0, 'Count');
    expect(mockPutMetric).toHaveBeenCalledWith("FoundLicenseFile" /* FOUND_LICENSE_FILE */, 1, 'Count');
});
test('mismatched package name', async () => {
    const mockBucketName = 'fake-bucket';
    const mockStateMachineArn = 'fake-state-machine-arn';
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockRequireEnv = require('../../../backend/shared/env.lambda-shared').requireEnv;
    mockRequireEnv.mockImplementation((name) => {
        if (name === 'BUCKET_NAME') {
            return mockBucketName;
        }
        if (name === 'STATE_MACHINE_ARN') {
            return mockStateMachineArn;
        }
        throw new Error(`Bad environment variable: "${name}"`);
    });
    const stagingBucket = 'staging-bucket';
    const stagingKey = 'staging-key';
    const stagingVersion = 'staging-version-id';
    const fakeTarGz = Buffer.from('fake-tarball-content[gzipped]');
    const fakeTar = Buffer.from('fake-tarball-content');
    const fakeLicense = 'inscrutable-legalese';
    const tarballUri = `s3://${stagingBucket}.test-bermuda-2.s3.amazonaws.com/${stagingKey}?versionId=${stagingVersion}`;
    const time = '2021-07-12T15:18:00.000000+02:00';
    const integrity = 'sha256-1RyNs3cDpyTqBMqJIiHbCpl8PEN6h3uWx3lzF+3qcmY=';
    const packageName = '@package-scope/package-name';
    const packageVersion = '1.2.3-pre.4';
    const packageLicense = 'Apache-2.0';
    const fakeDotJsii = JSON.stringify(fakeAssembly(packageName + '-oops', packageVersion, packageLicense));
    const context = {
        awsRequestId: 'Fake-Request-ID',
        logGroupName: 'Fake-Log-Group',
        logStreamName: 'Fake-Log-Stream',
    };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(stagingBucket);
            expect(req.Key).toBe(stagingKey);
            expect(req.VersionId).toBe(stagingVersion);
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Body: fakeTarGz });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockCreateGunzip = require('zlib').createGunzip;
    mockCreateGunzip.mockImplementation(() => new FakeGunzip(fakeTarGz, fakeTar));
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockExtract = require('tar-stream').extract;
    mockExtract.mockImplementation(() => new FakeExtract(fakeTar, {
        'package/.jsii': fakeDotJsii,
        'package/LICENSE.md': fakeLicense,
        'package/index.js': '// Ignore me!',
        'package/package.json': JSON.stringify({ name: packageName, version: packageVersion, license: packageLicense }),
    }));
    const event = {
        Records: [{
                attributes: {},
                awsRegion: 'test-bermuda-1',
                body: JSON.stringify({ tarballUri, integrity, time }),
                eventSource: 'sqs',
                eventSourceARN: 'arn:aws:sqs:test-bermuda-1:123456789012:fake',
                md5OfBody: 'Fake-MD5-Of-Body',
                messageAttributes: {},
                messageId: 'Fake-Message-ID',
                receiptHandle: 'Fake-Receipt-Handke',
            }],
    };
    // We require the handler here so that any mocks to metricScope are set up
    // prior to the handler being created.
    //
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    await expect(require('../../../backend/ingestion/ingestion.lambda').handler(event, context))
        .resolves.toEqual([]);
    expect(mockPutMetric).toHaveBeenCalledWith("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, 'Count');
});
test('mismatched package version', async () => {
    const mockBucketName = 'fake-bucket';
    const mockStateMachineArn = 'fake-state-machine-arn';
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockRequireEnv = require('../../../backend/shared/env.lambda-shared').requireEnv;
    mockRequireEnv.mockImplementation((name) => {
        if (name === 'BUCKET_NAME') {
            return mockBucketName;
        }
        if (name === 'STATE_MACHINE_ARN') {
            return mockStateMachineArn;
        }
        throw new Error(`Bad environment variable: "${name}"`);
    });
    const stagingBucket = 'staging-bucket';
    const stagingKey = 'staging-key';
    const stagingVersion = 'staging-version-id';
    const fakeTarGz = Buffer.from('fake-tarball-content[gzipped]');
    const fakeTar = Buffer.from('fake-tarball-content');
    const fakeLicense = 'inscrutable-legalese';
    const tarballUri = `s3://${stagingBucket}.test-bermuda-2.s3.amazonaws.com/${stagingKey}?versionId=${stagingVersion}`;
    const time = '2021-07-12T15:18:00.000000+02:00';
    const integrity = 'sha256-1RyNs3cDpyTqBMqJIiHbCpl8PEN6h3uWx3lzF+3qcmY=';
    const packageName = '@package-scope/package-name';
    const packageVersion = '1.2.3-pre.4';
    const packageLicense = 'Apache-2.0';
    const fakeDotJsii = JSON.stringify(fakeAssembly(packageName, packageVersion + '-oops', packageLicense));
    const context = {
        awsRequestId: 'Fake-Request-ID',
        logGroupName: 'Fake-Log-Group',
        logStreamName: 'Fake-Log-Stream',
    };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(stagingBucket);
            expect(req.Key).toBe(stagingKey);
            expect(req.VersionId).toBe(stagingVersion);
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Body: fakeTarGz });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockCreateGunzip = require('zlib').createGunzip;
    mockCreateGunzip.mockImplementation(() => new FakeGunzip(fakeTarGz, fakeTar));
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockExtract = require('tar-stream').extract;
    mockExtract.mockImplementation(() => new FakeExtract(fakeTar, {
        'package/.jsii': fakeDotJsii,
        'package/LICENSE.md': fakeLicense,
        'package/index.js': '// Ignore me!',
        'package/package.json': JSON.stringify({ name: packageName, version: packageVersion, license: packageLicense }),
    }));
    const event = {
        Records: [{
                attributes: {},
                awsRegion: 'test-bermuda-1',
                body: JSON.stringify({ tarballUri, integrity, time }),
                eventSource: 'sqs',
                eventSourceARN: 'arn:aws:sqs:test-bermuda-1:123456789012:fake',
                md5OfBody: 'Fake-MD5-Of-Body',
                messageAttributes: {},
                messageId: 'Fake-Message-ID',
                receiptHandle: 'Fake-Receipt-Handke',
            }],
    };
    // We require the handler here so that any mocks to metricScope are set up
    // prior to the handler being created.
    //
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    await expect(require('../../../backend/ingestion/ingestion.lambda').handler(event, context))
        .resolves.toEqual([]);
    expect(mockPutMetric).toHaveBeenCalledWith("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, 'Count');
});
test('mismatched package license', async () => {
    const mockBucketName = 'fake-bucket';
    const mockStateMachineArn = 'fake-state-machine-arn';
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockRequireEnv = require('../../../backend/shared/env.lambda-shared').requireEnv;
    mockRequireEnv.mockImplementation((name) => {
        if (name === 'BUCKET_NAME') {
            return mockBucketName;
        }
        if (name === 'STATE_MACHINE_ARN') {
            return mockStateMachineArn;
        }
        throw new Error(`Bad environment variable: "${name}"`);
    });
    const stagingBucket = 'staging-bucket';
    const stagingKey = 'staging-key';
    const stagingVersion = 'staging-version-id';
    const fakeTarGz = Buffer.from('fake-tarball-content[gzipped]');
    const fakeTar = Buffer.from('fake-tarball-content');
    const fakeLicense = 'inscrutable-legalese';
    const tarballUri = `s3://${stagingBucket}.test-bermuda-2.s3.amazonaws.com/${stagingKey}?versionId=${stagingVersion}`;
    const time = '2021-07-12T15:18:00.000000+02:00';
    const integrity = 'sha256-1RyNs3cDpyTqBMqJIiHbCpl8PEN6h3uWx3lzF+3qcmY=';
    const packageName = '@package-scope/package-name';
    const packageVersion = '1.2.3-pre.4';
    const packageLicense = 'Apache-2.0';
    const fakeDotJsii = JSON.stringify(fakeAssembly(packageName, packageVersion, packageLicense + '-oops'));
    const context = {
        awsRequestId: 'Fake-Request-ID',
        logGroupName: 'Fake-Log-Group',
        logStreamName: 'Fake-Log-Stream',
    };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(stagingBucket);
            expect(req.Key).toBe(stagingKey);
            expect(req.VersionId).toBe(stagingVersion);
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Body: fakeTarGz });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockCreateGunzip = require('zlib').createGunzip;
    mockCreateGunzip.mockImplementation(() => new FakeGunzip(fakeTarGz, fakeTar));
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockExtract = require('tar-stream').extract;
    mockExtract.mockImplementation(() => new FakeExtract(fakeTar, {
        'package/.jsii': fakeDotJsii,
        'package/LICENSE.md': fakeLicense,
        'package/index.js': '// Ignore me!',
        'package/package.json': JSON.stringify({ name: packageName, version: packageVersion, license: packageLicense }),
    }));
    const event = {
        Records: [{
                attributes: {},
                awsRegion: 'test-bermuda-1',
                body: JSON.stringify({ tarballUri, integrity, time }),
                eventSource: 'sqs',
                eventSourceARN: 'arn:aws:sqs:test-bermuda-1:123456789012:fake',
                md5OfBody: 'Fake-MD5-Of-Body',
                messageAttributes: {},
                messageId: 'Fake-Message-ID',
                receiptHandle: 'Fake-Receipt-Handke',
            }],
    };
    // We require the handler here so that any mocks to metricScope are set up
    // prior to the handler being created.
    //
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    await expect(require('../../../backend/ingestion/ingestion.lambda').handler(event, context))
        .resolves.toEqual([]);
    expect(mockPutMetric).toHaveBeenCalledWith("MismatchedIdentityRejections" /* MISMATCHED_IDENTITY_REJECTIONS */, 1, 'Count');
});
test('missing .jsii file', async () => {
    const mockBucketName = 'fake-bucket';
    const mockStateMachineArn = 'fake-state-machine-arn';
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockRequireEnv = require('../../../backend/shared/env.lambda-shared').requireEnv;
    mockRequireEnv.mockImplementation((name) => {
        if (name === 'BUCKET_NAME') {
            return mockBucketName;
        }
        if (name === 'STATE_MACHINE_ARN') {
            return mockStateMachineArn;
        }
        throw new Error(`Bad environment variable: "${name}"`);
    });
    const stagingBucket = 'staging-bucket';
    const stagingKey = 'staging-key';
    const stagingVersion = 'staging-version-id';
    const fakeTarGz = Buffer.from('fake-tarball-content[gzipped]');
    const fakeTar = Buffer.from('fake-tarball-content');
    const fakeLicense = 'inscrutable-legalese';
    const tarballUri = `s3://${stagingBucket}.test-bermuda-2.s3.amazonaws.com/${stagingKey}?versionId=${stagingVersion}`;
    const time = '2021-07-12T15:18:00.000000+02:00';
    const integrity = 'sha256-1RyNs3cDpyTqBMqJIiHbCpl8PEN6h3uWx3lzF+3qcmY=';
    const packageName = '@package-scope/package-name';
    const packageVersion = '1.2.3-pre.4';
    const packageLicense = 'Apache-2.0';
    const context = {
        awsRequestId: 'Fake-Request-ID',
        logGroupName: 'Fake-Log-Group',
        logStreamName: 'Fake-Log-Stream',
    };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(stagingBucket);
            expect(req.Key).toBe(stagingKey);
            expect(req.VersionId).toBe(stagingVersion);
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Body: fakeTarGz });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockCreateGunzip = require('zlib').createGunzip;
    mockCreateGunzip.mockImplementation(() => new FakeGunzip(fakeTarGz, fakeTar));
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockExtract = require('tar-stream').extract;
    mockExtract.mockImplementation(() => new FakeExtract(fakeTar, {
        'package/LICENSE.md': fakeLicense,
        'package/index.js': '// Ignore me!',
        'package/package.json': JSON.stringify({ name: packageName, version: packageVersion, license: packageLicense }),
    }));
    const event = {
        Records: [{
                attributes: {},
                awsRegion: 'test-bermuda-1',
                body: JSON.stringify({ tarballUri, integrity, time }),
                eventSource: 'sqs',
                eventSourceARN: 'arn:aws:sqs:test-bermuda-1:123456789012:fake',
                md5OfBody: 'Fake-MD5-Of-Body',
                messageAttributes: {},
                messageId: 'Fake-Message-ID',
                receiptHandle: 'Fake-Receipt-Handke',
            }],
    };
    // We require the handler here so that any mocks to metricScope are set up
    // prior to the handler being created.
    //
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    await expect(require('../../../backend/ingestion/ingestion.lambda').handler(event, context))
        .resolves.toBeUndefined();
});
test('missing package.json file', async () => {
    const mockBucketName = 'fake-bucket';
    const mockStateMachineArn = 'fake-state-machine-arn';
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockRequireEnv = require('../../../backend/shared/env.lambda-shared').requireEnv;
    mockRequireEnv.mockImplementation((name) => {
        if (name === 'BUCKET_NAME') {
            return mockBucketName;
        }
        if (name === 'STATE_MACHINE_ARN') {
            return mockStateMachineArn;
        }
        throw new Error(`Bad environment variable: "${name}"`);
    });
    const stagingBucket = 'staging-bucket';
    const stagingKey = 'staging-key';
    const stagingVersion = 'staging-version-id';
    const fakeTarGz = Buffer.from('fake-tarball-content[gzipped]');
    const fakeTar = Buffer.from('fake-tarball-content');
    const fakeLicense = 'inscrutable-legalese';
    const tarballUri = `s3://${stagingBucket}.test-bermuda-2.s3.amazonaws.com/${stagingKey}?versionId=${stagingVersion}`;
    const time = '2021-07-12T15:18:00.000000+02:00';
    const integrity = 'sha256-1RyNs3cDpyTqBMqJIiHbCpl8PEN6h3uWx3lzF+3qcmY=';
    const packageName = '@package-scope/package-name';
    const packageVersion = '1.2.3-pre.4';
    const packageLicense = 'Apache-2.0';
    const fakeDotJsii = JSON.stringify(fakeAssembly(packageName, packageVersion, packageLicense));
    const context = {
        awsRequestId: 'Fake-Request-ID',
        logGroupName: 'Fake-Log-Group',
        logStreamName: 'Fake-Log-Stream',
    };
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req.Bucket).toBe(stagingBucket);
            expect(req.Key).toBe(stagingKey);
            expect(req.VersionId).toBe(stagingVersion);
        }
        catch (e) {
            return cb(e);
        }
        return cb(null, { Body: fakeTarGz });
    });
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockCreateGunzip = require('zlib').createGunzip;
    mockCreateGunzip.mockImplementation(() => new FakeGunzip(fakeTarGz, fakeTar));
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const mockExtract = require('tar-stream').extract;
    mockExtract.mockImplementation(() => new FakeExtract(fakeTar, {
        'package/.jsii': fakeDotJsii,
        'package/LICENSE.md': fakeLicense,
        'package/index.js': '// Ignore me!',
    }));
    const event = {
        Records: [{
                attributes: {},
                awsRegion: 'test-bermuda-1',
                body: JSON.stringify({ tarballUri, integrity, time }),
                eventSource: 'sqs',
                eventSourceARN: 'arn:aws:sqs:test-bermuda-1:123456789012:fake',
                md5OfBody: 'Fake-MD5-Of-Body',
                messageAttributes: {},
                messageId: 'Fake-Message-ID',
                receiptHandle: 'Fake-Receipt-Handke',
            }],
    };
    // We require the handler here so that any mocks to metricScope are set up
    // prior to the handler being created.
    //
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    await expect(require('../../../backend/ingestion/ingestion.lambda').handler(event, context))
        .resolves.toBeUndefined();
});
class FakeGunzip extends events_1.EventEmitter {
    constructor(gz, result) {
        super();
        this.gz = gz;
        this.result = result;
        this.sent = 0;
    }
    end(data) {
        try {
            expect(data).toEqual(this.gz);
            setImmediate(() => this.sendData());
        }
        catch (e) {
            this.emit('error', e);
        }
    }
    sendData() {
        if (this.sent >= this.result.length) {
            this.emit('end');
            return;
        }
        this.emit('data', this.result.slice(this.sent, this.sent + 1));
        this.sent++;
        setImmediate(() => this.sendData());
    }
}
class FakeExtract extends events_1.EventEmitter {
    constructor(tar, files) {
        super();
        this.tar = tar;
        this.files = Object.entries(files);
    }
    write(data, cb) {
        try {
            expect(data).toEqual(Buffer.from(this.tar));
            cb === null || cb === void 0 ? void 0 : cb(null);
            setImmediate(() => this.sendNextEntry());
        }
        catch (e) {
            cb === null || cb === void 0 ? void 0 : cb(e);
        }
    }
    end() {
        // no-op
    }
    sendNextEntry() {
        const nextEntry = this.files.shift();
        if (nextEntry == null) {
            this.emit('finish');
            return;
        }
        const [name, content] = nextEntry;
        const headers = { name };
        const stream = new FakeStream(Buffer.from(content));
        const next = () => this.sendNextEntry();
        this.emit('entry', headers, stream, next);
    }
}
class FakeStream extends events_1.EventEmitter {
    constructor(content) {
        super();
        this.content = content;
        this.sent = 0;
    }
    resume() {
        setImmediate(() => this.sendData());
    }
    sendData() {
        if (this.sent >= this.content.length) {
            this.emit('end');
            return;
        }
        this.emit('data', this.content.slice(this.sent, this.sent + 1));
        this.sent++;
        setImmediate(() => this.sendData());
    }
}
function fakeAssembly(name, version, license) {
    return {
        schema: spec_1.SchemaVersion.LATEST,
        name,
        version,
        license,
        homepage: 'https://localhost.fake/repository',
        repository: { url: 'ssh://localhost.fake/repository.git', type: 'git' },
        author: { name: 'ACME', email: 'test@acme', organization: true, roles: ['author'] },
        description: 'This is a fake package assembly',
        jsiiVersion: '0.0.0+head',
        fingerprint: 'NOPE',
    };
}
//# sourceMappingURL=data:application/json;base64,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