"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const Case = require("case");
const client_lambda_shared_1 = require("../../../backend/license-list/client.lambda-shared");
const aws_lambda_shared_1 = require("../../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../../backend/shared/env.lambda-shared");
jest.mock('../../../backend/shared/env.lambda-shared');
const mockRequireEnv = env_lambda_shared_1.requireEnv;
const mockBucketName = 'fake-bucket';
const mockObjectKey = 'object/key';
beforeEach(() => {
    AWSMock.setSDKInstance(AWS);
    mockRequireEnv.mockImplementation((name) => {
        switch (name) {
            case "LICENSE_LIST_BUCKET_NAME" /* BUCKET_NAME */:
                return mockBucketName;
            case "LICENSE_LIST_OBJECT_KEY" /* OBJECT_KEY */:
                return mockObjectKey;
            default:
                throw new Error(`Attempted to use unexpected environment variable: ${name}`);
        }
    });
});
afterEach(() => {
    aws_lambda_shared_1.reset();
    AWSMock.restore();
});
test('basic use', async () => {
    // GIVEN
    const mockLicense = 'MockLicense-1.0';
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req).toEqual({ Bucket: mockBucketName, Key: mockObjectKey });
            cb(null, { Body: JSON.stringify([mockLicense]) });
        }
        catch (e) {
            cb(e);
        }
    });
    // WHEN
    const licenseList = await client_lambda_shared_1.LicenseListClient.newClient();
    // THEN
    expect(licenseList.lookup(Case.random(mockLicense))).toBe(mockLicense);
    expect(licenseList.lookup('NotInList')).toBeUndefined();
});
test('empty list', async () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req).toEqual({ Bucket: mockBucketName, Key: mockObjectKey });
            cb(null, { Body: JSON.stringify([]) });
        }
        catch (e) {
            cb(e);
        }
    });
    // WHEN
    const licenseList = await client_lambda_shared_1.LicenseListClient.newClient();
    // THEN
    expect(licenseList.lookup('NotInList')).toBeUndefined();
});
test('absent list', async () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req).toEqual({ Bucket: mockBucketName, Key: mockObjectKey });
            cb(null, {});
        }
        catch (e) {
            cb(e);
        }
    });
    // WHEN
    const licenseList = await client_lambda_shared_1.LicenseListClient.newClient();
    // THEN
    expect(licenseList.lookup('NotInList')).toBeUndefined();
});
test('broken list', async () => {
    // GIVEN
    AWSMock.mock('S3', 'getObject', (req, cb) => {
        try {
            expect(req).toEqual({ Bucket: mockBucketName, Key: mockObjectKey });
            cb(null, { Body: JSON.stringify('{}', null, 2) });
        }
        catch (e) {
            cb(e);
        }
    });
    // THEN
    return expect(client_lambda_shared_1.LicenseListClient.newClient()).rejects.toThrowError(/Invalid format/);
});
//# sourceMappingURL=data:application/json;base64,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