"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto_1 = require("crypto");
const AWS = require("aws-sdk");
const AWSMock = require("aws-sdk-mock");
const mockDistributionId = crypto_1.pseudoRandomBytes(16).toString('hex');
const mockPathPrefix = crypto_1.pseudoRandomBytes(16).toString('base64');
beforeEach(() => {
    process.env.DISTRIBUTION_ID = mockDistributionId;
    process.env.PATH_PREFIX = mockPathPrefix;
    AWSMock.setSDKInstance(AWS);
});
afterEach(() => {
    delete process.env.DISTRIBUTION_ID;
    delete process.env.PATH_PREFIX;
    AWSMock.restore();
});
const mockEvent = {
    Records: Array.from({ length: randomInt(2, 10) }, (_, index) => ({
        eventVersion: '1337.42',
        eventSource: 's3.test',
        awsRegion: 'bermuda-triangle-1',
        eventTime: new Date().toISOString(),
        eventName: 'PutObject:Test',
        userIdentity: { principalId: 'arn::test::principal' },
        requestParameters: { sourceIPAddress: '127.0.0.1' },
        responseElements: { 'x-amz-request-id': crypto_1.pseudoRandomBytes(16).toString('hex'), 'x-amz-id-2': crypto_1.pseudoRandomBytes(16).toString('hex') },
        s3: {
            s3SchemaVersion: '1337',
            configurationId: '42',
            bucket: { arn: 'arn:aws:s3:::phony-bucket-name', name: 'phony-bucket-name', ownerIdentity: { principalId: 'arn::test::principal' } },
            object: {
                key: crypto_1.pseudoRandomBytes(16).toString('base64'),
                size: randomInt(0, 1024),
                eTag: crypto_1.pseudoRandomBytes(16).toString('hex'),
                sequencer: `${index}-${crypto_1.pseudoRandomBytes(16).toString('hex')}`,
            },
        },
    })),
};
test('creates the expected invalidation', async () => {
    const mockInvalidationLocation = crypto_1.pseudoRandomBytes(16).toString('hex');
    AWSMock.mock('CloudFront', 'createInvalidation', (req, cb) => {
        try {
            expect(req.DistributionId).toBe(mockDistributionId);
            expect(req.InvalidationBatch.Paths.Quantity).toBe(mockEvent.Records.length);
            expect(req.InvalidationBatch.Paths.Items).toEqual(mockEvent.Records.map((record) => `${mockPathPrefix}${record.s3.object.key}`));
            expect(req.InvalidationBatch.CallerReference).toBe('MyRequestId');
        }
        catch (e) {
            return cb(e);
        }
        cb(null, { Location: mockInvalidationLocation });
    });
    // Requiring the handler her to ensure it sees the expected environment variables
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const { handler } = require('../../../webapp/cache-invalidator/handler.lambda');
    return expect(handler(mockEvent, { awsRequestId: 'MyRequestId' })).resolves.toEqual({ Location: mockInvalidationLocation });
});
function randomInt(min, max) {
    return min + Math.ceil(Math.random() * (max - min));
}
//# sourceMappingURL=data:application/json;base64,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