import { Metric, MetricOptions } from '@aws-cdk/aws-cloudwatch';
import * as lambda from '@aws-cdk/aws-lambda';
import * as s3 from '@aws-cdk/aws-s3';
import { Construct, Duration } from '@aws-cdk/core';
import { Monitoring } from '../../monitoring';
import { DenyListRule } from './api';
import { Prune } from './prune';
/**
 * Props for `DenyList`.
 */
export interface DenyListProps {
    /**
     * A set of rules that are matched against package names and versions to
     * determine if they are blocked from displaying in the Construct Hub.
     *
     * An empty list will result in no packages being blocked.
     */
    readonly rules: DenyListRule[];
    /**
     * The S3 bucket that includes the package data.
     */
    readonly packageDataBucket: s3.IBucket;
    /**
     * The S3 key prefix for all package data.
     */
    readonly packageDataKeyPrefix: string;
    /**
     * Triggers prune when deny list changes.
     *
     * @default true
     */
    readonly pruneOnChange?: boolean;
    /**
     * Prunes all S3 objects that are in the deny list periodically.
     *
     * Use `Duration.minutes(0)` to disable periodical pruning.
     *
     * @default Duration.minutes(5)
     */
    readonly prunePeriod?: Duration;
    /**
     * The monitoring system.
     */
    readonly monitoring: Monitoring;
}
/**
 * Manages the construct hub deny list.
 */
export declare class DenyList extends Construct {
    /**
     * The S3 bucket that contains the deny list.
     */
    readonly bucket: s3.Bucket;
    /**
     * The object key within the bucket with the deny list JSON file.
     */
    readonly objectKey = "deny-list.json";
    /**
     * Responsible for deleting objects that match the deny list.
     */
    readonly prune: Prune;
    private readonly upload;
    constructor(scope: Construct, id: string, props: DenyListProps);
    /**
     * Grants an AWS Lambda function permissions to read the deny list, and adds
     * the relevant environment variables expected by the `DenyListClient`.
     */
    grantRead(handler: lambda.Function): void;
    /**
     * Number of rules in the deny list.
     */
    metricDenyListRules(opts?: MetricOptions): Metric;
    /**
     * Writes the deny list to a temporary file and returns a path to a directory
     * with the JSON file. The contents of the JSON file is a map where keys are
     * package names (and optionally, version) and the value is the deny list
     * entry. This makes it easier to query by the service.
     *
     * Also performs some validation to make sure there are no duplicate entries.
     *
     * @param list The deny list
     * @returns a path to a temporary directory that can be deployed to S3
     */
    private writeToFile;
}
