"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Discovery = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const discovery_1 = require("./discovery");
/**
 * This discovery function periodically scans the CouchDB replica of npmjs.com
 * to discover newly published packages that are relevant for indexing in the
 * Construct Hub, then notifies the ingestion function about those.
 */
class Discovery extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.timeout = core_1.Duration.minutes(5);
        this.bucket = new aws_s3_1.Bucket(this, 'StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [
                {
                    prefix: "staged/" /* STAGED_KEY_PREFIX */,
                    expiration: core_1.Duration.days(30),
                },
            ],
        });
        // Note: the handler is designed to stop processing more batches about 30 seconds ahead of the timeout.
        const handler = new discovery_1.Discovery(this, 'Default', {
            description: '[ConstructHub/Discovery] Periodically query npm.js index for new construct libraries',
            environment: {
                BUCKET_NAME: this.bucket.bucketName,
                QUEUE_URL: props.queue.queueUrl,
            },
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: this.timeout,
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.function = handler;
        this.bucket.grantReadWrite(this.function);
        props.queue.grantSendMessages(this.function);
        props.denyList.grantRead(handler);
        props.licenseList.grantRead(handler);
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(this.timeout),
            targets: [new aws_events_targets_1.LambdaFunction(this.function)],
        });
        this.alarmErrors = this.function.metricErrors().createAlarm(this, 'ErrorsAlarm', {
            alarmName: `${this.node.path}/Errors`,
            alarmDescription: [
                'The discovery function (on npmjs.com) failed to run',
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 3,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        this.alarmNoInvocations = this.function.metricInvocations()
            .createAlarm(this, 'NoInvocationsAlarm', {
            alarmName: `${this.node.path}/NotRunning`,
            alarmDescription: [
                'The discovery function (on npmjs.com) is not running as scheduled',
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(this.function)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        props.monitoring.addHighSeverityAlarm('Discovery Failures', this.alarmErrors);
        props.monitoring.addHighSeverityAlarm('Discovery not Running', this.alarmNoInvocations);
    }
    /**
     * The average time it took to process a changes batch.
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of changes that were processed.
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    metricNpmJsChangeAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "NpmJsChangeAge" /* NPMJS_CHANGE_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The age of the oldest package version that was processed.
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were inspected.
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were deemed relevant.
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of time that was remaining when the lambda returned in order to
     * avoid hitting a timeout.
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of staging failures.
     */
    metricStagingFailureCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "StagingFailureCount" /* STAGING_FAILURE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The average time it took to stage a package to S3.
     */
    metricStagingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "StagingTime" /* STAGING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of changes that were not processed due to having an invalid
     * format.
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.timeout,
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
}
exports.Discovery = Discovery;
//# sourceMappingURL=data:application/json;base64,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