"use strict";
/*
 * IMPORTANT: This file is used both by bundled lambda functions and by
 * constructs. This means it should not directly or transitively import anything
 * not part of the package's `dependencies`, such as the `aws-sdk`.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.STORAGE_KEY_FORMAT_REGEX = exports.CATALOG_KEY = exports.NOT_SUPPORTED_SUFFIX = exports.docsKeySuffix = exports.getObjectKeys = exports.getObjectKeyPrefix = exports.DOCS_KEY_SUFFIX_ANY = exports.DOCS_KEY_SUFFIX_PYTHON = exports.DOCS_KEY_SUFFIX_TYPESCRIPT = exports.ASSEMBLY_KEY_SUFFIX = exports.METADATA_KEY_SUFFIX = exports.PACKAGE_KEY_SUFFIX = exports.STORAGE_KEY_PREFIX = void 0;
const language_1 = require("./language");
/**
 * Key prefix for the package data storage.
 */
exports.STORAGE_KEY_PREFIX = 'data/';
/**
 * Key suffix for storing npm package bundles.
 */
exports.PACKAGE_KEY_SUFFIX = '/package.tgz';
/**
 * Key suffix for storing npm package metadata.
 */
exports.METADATA_KEY_SUFFIX = '/metadata.json';
/**
 * The key suffix for (TypeScript) assembly files
 */
exports.ASSEMBLY_KEY_SUFFIX = '/assembly.json';
/**
 * The key suffix for a TypeScript doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_TYPESCRIPT = docsKeySuffix(language_1.DocumentationLanguage.TYPESCRIPT);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_PYTHON = docsKeySuffix(language_1.DocumentationLanguage.PYTHON);
/**
 * The key suffix matching any documentation artifact.
 */
exports.DOCS_KEY_SUFFIX_ANY = docsKeySuffix('*');
/**
 * Return the S3 object key prefix for a specific package name and optionally a
 * version. Note that the prefix does not end with a "/" so you will likely want
 * to add that if you want to match a specific set of objects.
 */
function getObjectKeyPrefix(packageName, packageVersion) {
    let key = `${exports.STORAGE_KEY_PREFIX}${packageName}`;
    if (packageVersion) {
        key += `/v${packageVersion}`;
    }
    return key;
}
exports.getObjectKeyPrefix = getObjectKeyPrefix;
/**
 * Resolves the set of S3 object keys use for a specific package/version.
 */
function getObjectKeys(packageName, packageVersion) {
    const prefix = getObjectKeyPrefix(packageName, packageVersion);
    return {
        assemblyKey: `${prefix}${exports.ASSEMBLY_KEY_SUFFIX}`,
        packageKey: `${prefix}${exports.PACKAGE_KEY_SUFFIX}`,
        metadataKey: `${prefix}${exports.METADATA_KEY_SUFFIX}`,
    };
}
exports.getObjectKeys = getObjectKeys;
/**
 * The key suffix for documentation artifacts by language and submodule.
 */
function docsKeySuffix(lang, submodule) {
    return `/docs-${submodule ? `${submodule}-` : ''}${lang}.md`;
}
exports.docsKeySuffix = docsKeySuffix;
/**
 * Key suffix for beacon files when a particular feature is not supported for
 * the particular package (i.e: Python docs for a package that does not have a
 * Python target configured).
 */
exports.NOT_SUPPORTED_SUFFIX = '.not-supported';
/**
 * The key for the catalog document.
 */
exports.CATALOG_KEY = 'catalog.json';
/**
 * A regular expression that can be used to parse out a storage key.
 */
exports.STORAGE_KEY_FORMAT_REGEX = new RegExp(`^${exports.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`);
// Capture groups:                                                        ┗━━━━━━━━1━━━━━━━━┛  ┗━━2━━┛
//# sourceMappingURL=data:application/json;base64,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