"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                    ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to Python, then
 * uploads the resulting `.jsii.python` object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const language = env_lambda_shared_1.requireEnv('TARGET_LANGUAGE');
        const created = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        if (language !== 'typescript' && assembly.targets[language] == null) {
            console.error(`Package ${assembly.name}@${assembly.version} does not support ${language}, skipping!`);
            console.log(`Assembly targets: ${JSON.stringify(assembly.targets, null, 2)}`);
            for (const submodule of [undefined, ...submodules]) {
                const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(language), submodule)) + constants.NOT_SUPPORTED_SUFFIX;
                const response = await uploadFile(context, event.bucket, key, event.assembly.versionId);
                created.push({ bucket: event.bucket, key, versionId: response.VersionId });
            }
            return created;
        }
        async function generateDocs(lang) {
            const uploads = new Map();
            const docs = await docgen.Documentation.forPackage(packageFqn, { language: docgen.Language.fromString(lang) });
            function renderAndDispatch(submodule) {
                console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                const page = docs.render({ submodule, linkFormatter: linkFormatter(docs) }).render();
                const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language_1.DocumentationLanguage.fromString(lang), submodule));
                console.log(`Uploading ${key}`);
                const upload = uploadFile(context, event.bucket, key, event.assembly.versionId, page);
                uploads.set(key, upload);
            }
            renderAndDispatch();
            for (const submodule of submodules) {
                renderAndDispatch(submodule);
            }
            for (const [key, upload] of uploads.entries()) {
                const response = await upload;
                created.push({ bucket: event.bucket, key, versionId: response.VersionId });
                console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
            }
        }
        await generateDocs(language);
        return created;
    });
}
exports.handler = handler;
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(context, bucket, key, sourceVersionId, body) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: 'public',
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
        },
    }).promise();
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(docs) {
    function _formatter(type) {
        const packageName = type.source.assembly.name;
        const packageVersion = type.source.assembly.version;
        // the webapp sanitizes anchors - so we need to as well when
        // linking to them.
        const hash = sanitize(type.fqn);
        if (docs.assembly.name === packageName) {
            // link to the same package - just add the hash
            return `#${hash}`;
        }
        // cross link to another package
        return `/packages/${packageName}/v/${packageVersion}?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}#${hash}`;
    }
    return _formatter;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
//# sourceMappingURL=data:application/json;base64,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