"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmJs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../deep-link");
const metric_utils_1 = require("../metric-utils");
const runbook_url_1 = require("../runbook-url");
const canary_1 = require("./npmjs/canary");
const constants_lambda_shared_1 = require("./npmjs/constants.lambda-shared");
const npm_js_follower_1 = require("./npmjs/npm-js-follower");
const stage_and_notify_1 = require("./npmjs/stage-and-notify");
/**
 * The periodicity at which the NpmJs follower will run. This MUST be a valid
 * CloudWatch Metric grain, as this will also be the period of the CloudWatch
 * alarm that montiors the health of the follower.
 */
const FOLLOWER_RUN_RATE = core_1.Duration.minutes(5);
/**
 * (experimental) A package source that gets package data from the npmjs.com package registry.
 *
 * @experimental
 */
class NpmJs {
    /**
     * @experimental
     */
    constructor(props = {}) {
        this.props = props;
    }
    /**
     * (experimental) Binds the package source to a scope and target queue.
     *
     * @experimental
     */
    bind(scope, { baseUrl, denyList, ingestion, licenseList, monitoring, queue, repository }) {
        var _b, _c, _d;
        repository === null || repository === void 0 ? void 0 : repository.addExternalConnection('public:npmjs');
        const bucket = this.props.stagingBucket || new aws_s3_1.Bucket(scope, 'NpmJs/StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [{ prefix: "staged/" /* STAGED_KEY_PREFIX */, expiration: core_1.Duration.days(30) }],
        });
        bucket.grantRead(ingestion);
        const stager = new stage_and_notify_1.StageAndNotify(scope, 'NpmJs-StageAndNotify', {
            deadLetterQueue: new aws_sqs_1.Queue(scope, 'StagerDLQ', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                retentionPeriod: core_1.Duration.days(14),
                visibilityTimeout: core_1.Duration.minutes(15),
            }),
            description: `[${scope.node.path}/NpmJS-StageAndNotify] Stages tarballs to S3 and notifies ConstructHub`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                QUEUE_URL: queue.queueUrl,
            },
            memorySize: 10024,
            retryAttempts: 2,
            timeout: core_1.Duration.minutes(5),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(stager);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(stager);
        queue.grantSendMessages(stager);
        stager.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(stager.deadLetterQueue, { batchSize: 1, enabled: false }));
        const follower = new npm_js_follower_1.NpmJsFollower(scope, 'NpmJs', {
            description: `[${scope.node.path}/NpmJs] Periodically query npmjs.com index for new packages`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                FUNCTION_NAME: stager.functionName,
            },
            memorySize: 10024,
            reservedConcurrentExecutions: 1,
            timeout: FOLLOWER_RUN_RATE,
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(follower, constants_lambda_shared_1.MARKER_FILE_NAME);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(follower);
        licenseList.grantRead(follower);
        stager.grantInvoke(follower);
        const rule = new aws_events_1.Rule(scope, 'NpmJs/Schedule', {
            description: `${scope.node.path}/NpmJs/Schedule`,
            schedule: aws_events_1.Schedule.rate(FOLLOWER_RUN_RATE),
            targets: [new aws_events_targets_1.LambdaFunction(follower)],
        });
        this.registerAlarms(scope, follower, stager, monitoring, rule);
        return {
            name: follower.node.path,
            links: [
                { name: 'NpmJs Follower', url: deep_link_1.lambdaFunctionUrl(follower), primary: true },
                { name: 'Marker Object', url: deep_link_1.s3ObjectUrl(bucket, constants_lambda_shared_1.MARKER_FILE_NAME) },
                { name: 'Stager', url: deep_link_1.lambdaFunctionUrl(stager) },
                { name: 'Stager DLQ', url: deep_link_1.sqsQueueUrl(stager.deadLetterQueue) },
            ],
            dashboardWidgets: [
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Follower Health',
                        left: [
                            metric_utils_1.fillMetric(follower.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(follower.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            this.metricRemainingTime({ label: 'Remaining Time' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Stager Health',
                        left: [
                            metric_utils_1.fillMetric(stager.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(stager.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            stager.metricDuration({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ], [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Follower',
                        left: [
                            metric_utils_1.fillMetric(this.metricChangeCount({ label: 'Change Count' }), 0),
                            metric_utils_1.fillMetric(this.metricUnprocessableEntity({ label: 'Unprocessable' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            metric_utils_1.fillMetric(this.metricNpmJsChangeAge({ label: 'Lag to npmjs.com' }), 'REPEAT'),
                            metric_utils_1.fillMetric(this.metricPackageVersionAge({ label: 'Package Version Age' }), 'REPEAT'),
                        ],
                        rightYAxis: { label: 'Milliseconds', min: 0, showUnits: false },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Changes',
                        left: [
                            metric_utils_1.fillMetric(this.metricLastSeq({ label: 'Last Sequence Number' }), 'REPEAT'),
                        ],
                        period: core_1.Duration.minutes(5),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Stager Dead-Letter Queue',
                        left: [
                            metric_utils_1.fillMetric(stager.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }), 0),
                            metric_utils_1.fillMetric(stager.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            stager.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    ...(((_b = this.props.enableCanary) !== null && _b !== void 0 ? _b : true)
                        ? this.registerCanary(follower, (_c = this.props.canaryPackage) !== null && _c !== void 0 ? _c : 'construct-hub-probe', (_d = this.props.canarySla) !== null && _d !== void 0 ? _d : core_1.Duration.minutes(5), bucket, baseUrl, monitoring)
                        : []),
                ],
            ],
        };
    }
    /**
     * (experimental) The average time it took to process a changes batch.
     *
     * @experimental
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of changes that were processed.
     *
     * @experimental
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The last sequence number that was processed.
     *
     * This metric can be used to
     * discover when a sequence reset has happened in the CouchDB instance.
     *
     * @experimental
     */
    metricLastSeq(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "LastSeq" /* LAST_SEQ */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * @experimental
     */
    metricNpmJsChangeAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "NpmJsChangeAge" /* NPMJS_CHANGE_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The age of the oldest package version that was processed.
     *
     * @experimental
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of package versions that were inspected.
     *
     * @experimental
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of package versions that were deemed relevant.
     *
     * @experimental
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The amount of time that was remaining when the lambda returned in order to avoid hitting a timeout.
     *
     * @experimental
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The amount of changes that were not processed due to having an invalid format.
     *
     * @experimental
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    registerAlarms(scope, follower, stager, monitoring, schedule) {
        const failureAlarm = follower.metricErrors().createAlarm(scope, 'NpmJs/Follower/Failures', {
            alarmName: `${scope.node.path}/NpmJs/Follower/Failures`,
            alarmDescription: [
                'The NpmJs follower function failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 3,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Follower Failures', failureAlarm);
        const notRunningAlarm = follower.metricInvocations({ period: FOLLOWER_RUN_RATE })
            .createAlarm(scope, 'NpmJs/Follower/NotRunning', {
            alarmName: `${scope.node.path}/NpmJs/Follower/NotRunning`,
            alarmDescription: [
                'The NpmJs follower function is not running!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Follower Not Running', notRunningAlarm);
        // The period for this alarm needs to match the scheduling interval of the
        // follower, otherwise the metric will be too sparse to properly detect
        // problems.
        const noChangeAlarm = this.metricChangeCount({ period: FOLLOWER_RUN_RATE })
            .createAlarm(scope, 'NpmJs/Follower/NoChanges', {
            alarmName: `${scope.node.path}/NpmJs/Follower/NoChanges`,
            alarmDescription: [
                'The NpmJs follower function is no discovering any changes from CouchDB!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // If the metric is not emitted, it can be assumed to be zero.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        monitoring.addLowSeverityAlarm('Np npmjs.com changes discovered', noChangeAlarm);
        const dlqNotEmptyAlarm = new aws_cloudwatch_1.MathExpression({
            expression: 'mVisible + mHidden',
            usingMetrics: {
                mVisible: stager.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                mHidden: stager.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(scope, `${scope.node.path}/NpmJs/Stager/DLQNotEmpty`, {
            alarmName: `${scope.node.path}/NpmJs/Stager/DLQNotEmpty`,
            alarmDescription: [
                'The NpmJS package stager is failing - its dead letter queue is not empty',
                '',
                `Link to the lambda function: ${deep_link_1.lambdaFunctionUrl(stager)}`,
                `Link to the dead letter queue: ${deep_link_1.sqsQueueUrl(stager.deadLetterQueue)}`,
                '',
                `Runbook: ${runbook_url_1.RUNBOOK_URL}`,
            ].join('/n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Stager DLQ Not Empty', dlqNotEmptyAlarm);
        // Finally - the "not running" alarm depends on the schedule (it won't run until the schedule
        // exists!), and the schedule depends on the failure alarm existing (we don't want it to run
        // before we can know it is failing). This means the returned `IDependable` effectively ensures
        // all alarms have been provisionned already! Isn't it nice!
        notRunningAlarm.node.addDependency(schedule);
        schedule.node.addDependency(failureAlarm);
    }
    registerCanary(scope, packageName, visibilitySla, bucket, constructHubBaseUrl, monitoring) {
        const canary = new canary_1.NpmJsPackageCanary(scope, 'Canary', { bucket, constructHubBaseUrl, packageName });
        const alarm = new aws_cloudwatch_1.MathExpression({
            expression: 'MAX([mDwell, mTTC])',
            period: core_1.Duration.minutes(1),
            usingMetrics: {
                mDwell: canary.metricDwellTime(),
                mTTC: canary.metricTimeToCatalog(),
            },
        }).createAlarm(canary, 'Alarm', {
            alarmName: `${canary.node.path}/SLA-Breached`,
            alarmDescription: [
                `New versions of ${packageName} have been published over ${visibilitySla.toHumanString()} ago and are still not visible in construct hub`,
                `Runbook: ${runbook_url_1.RUNBOOK_URL}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 2,
            // If there is no data, the canary might not be running, so... *Chuckles* we're in danger!
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
            threshold: visibilitySla.toSeconds(),
        });
        monitoring.addHighSeverityAlarm('New version visibility SLA breached', alarm);
        return [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Package Canary',
                left: [
                    canary.metricDwellTime({ label: 'Dwell Time' }),
                    canary.metricTimeToCatalog({ label: 'Time to Catalog' }),
                ],
                leftAnnotations: [{
                        color: '#ff0000',
                        label: `SLA (${visibilitySla.toHumanString()})`,
                        value: visibilitySla.toSeconds(),
                    }],
                leftYAxis: { min: 0 },
                right: [
                    canary.metricTrackedVersionCount({ label: 'Tracked Version Count' }),
                ],
                rightYAxis: { min: 0 },
            }),
        ];
    }
}
exports.NpmJs = NpmJs;
_a = JSII_RTTI_SYMBOL_1;
NpmJs[_a] = { fqn: "construct-hub.sources.NpmJs", version: "0.3.145" };
//# sourceMappingURL=data:application/json;base64,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