"use strict";
/*
 * IMPORTANT: This file is used both by bundled lambda functions and by
 * constructs. This means it should not directly or transitively import anything
 * not part of the package's `dependencies`, such as the `aws-sdk`.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.STORAGE_KEY_FORMAT_REGEX = exports.CORRUPT_ASSEMBLY_REPORT_PATTERN = exports.MISSING_DOCUMENTATION_REPORT_PATTERN = exports.UNINSTALLABLE_PACKAGES_REPORT = exports.corruptAssemblyReport = exports.missingDocumentationReport = exports.VERSION_TRACKER_KEY = exports.CATALOG_KEY = exports.UNPROCESSABLE_PACKAGE_ERROR_NAME = exports.UNINSTALLABLE_PACKAGE_SUFFIX = exports.CORRUPT_ASSEMBLY_SUFFIX = exports.NOT_SUPPORTED_SUFFIX = exports.notSupportedKeySuffix = exports.corruptAssemblyKeySuffix = exports.docsKeySuffix = exports.getObjectKeys = exports.getObjectKeyPrefix = exports.DOCS_KEY_SUFFIX_ANY = exports.DOCS_KEY_SUFFIX_CSHARP = exports.DOCS_KEY_SUFFIX_JAVA = exports.DOCS_KEY_SUFFIX_PYTHON = exports.DOCS_KEY_SUFFIX_TYPESCRIPT = exports.ASSEMBLY_KEY_SUFFIX = exports.METADATA_KEY_SUFFIX = exports.PACKAGE_KEY_SUFFIX = exports.STORAGE_KEY_PREFIX = void 0;
const language_1 = require("./language");
/**
 * Key prefix for the package data storage.
 */
exports.STORAGE_KEY_PREFIX = 'data/';
/**
 * Key suffix for storing npm package bundles.
 */
exports.PACKAGE_KEY_SUFFIX = '/package.tgz';
/**
 * Key suffix for storing npm package metadata.
 */
exports.METADATA_KEY_SUFFIX = '/metadata.json';
/**
 * The key suffix for (TypeScript) assembly files
 */
exports.ASSEMBLY_KEY_SUFFIX = '/assembly.json';
/**
 * The key suffix for a TypeScript doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_TYPESCRIPT = docsKeySuffix(language_1.DocumentationLanguage.TYPESCRIPT);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_PYTHON = docsKeySuffix(language_1.DocumentationLanguage.PYTHON);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_JAVA = docsKeySuffix(language_1.DocumentationLanguage.JAVA);
/**
 * The key suffix for a Python doc artifact (root module).
 */
exports.DOCS_KEY_SUFFIX_CSHARP = docsKeySuffix(language_1.DocumentationLanguage.CSHARP);
/**
 * The key suffix matching any documentation artifact.
 */
exports.DOCS_KEY_SUFFIX_ANY = docsKeySuffix('*');
/**
 * Return the S3 object key prefix for a specific package name and optionally a
 * version. Note that the prefix does not end with a "/" so you will likely want
 * to add that if you want to match a specific set of objects.
 */
function getObjectKeyPrefix(packageName, packageVersion) {
    let key = `${exports.STORAGE_KEY_PREFIX}${packageName}`;
    if (packageVersion) {
        key += `/v${packageVersion}`;
    }
    return key;
}
exports.getObjectKeyPrefix = getObjectKeyPrefix;
/**
 * Resolves the set of S3 object keys use for a specific package/version.
 */
function getObjectKeys(packageName, packageVersion) {
    const prefix = getObjectKeyPrefix(packageName, packageVersion);
    return {
        assemblyKey: `${prefix}${exports.ASSEMBLY_KEY_SUFFIX}`,
        packageKey: `${prefix}${exports.PACKAGE_KEY_SUFFIX}`,
        metadataKey: `${prefix}${exports.METADATA_KEY_SUFFIX}`,
    };
}
exports.getObjectKeys = getObjectKeys;
/**
 * The key suffix for documentation artifacts by language and submodule.
 */
function docsKeySuffix(lang, submodule) {
    return `/docs-${submodule ? `${submodule}-` : ''}${lang}.md`;
}
exports.docsKeySuffix = docsKeySuffix;
/**
 * The key suffix for a corrupted assembly marker by language and submodule.
 */
function corruptAssemblyKeySuffix(lang, submodule) {
    return `${docsKeySuffix(lang, submodule)}${exports.CORRUPT_ASSEMBLY_SUFFIX}`;
}
exports.corruptAssemblyKeySuffix = corruptAssemblyKeySuffix;
/**
 * The key suffix for a not supported marker by language and submodule.
 */
function notSupportedKeySuffix(lang, submodule) {
    return `${docsKeySuffix(lang, submodule)}${exports.NOT_SUPPORTED_SUFFIX}`;
}
exports.notSupportedKeySuffix = notSupportedKeySuffix;
/**
 * Key suffix for beacon files when a particular feature is not supported for
 * the particular package (i.e: Python docs for a package that does not have a
 * Python target configured).
 */
exports.NOT_SUPPORTED_SUFFIX = '.not-supported';
/**
 * Key suffix for beacon files marking that a language specific assembly is corrupt
 * and we cannot generate docs from it.
 */
exports.CORRUPT_ASSEMBLY_SUFFIX = '.corruptassembly';
/**
 * Key suffix for a beacon file when a package cannot be installed.
 */
exports.UNINSTALLABLE_PACKAGE_SUFFIX = '/uninstallable';
/**
 * Name of the error denoting an unprocessable package that should be diverted away from the DLQ.
 */
exports.UNPROCESSABLE_PACKAGE_ERROR_NAME = 'UnprocessablePackageError';
/**
 * The key for the catalog document.
 */
exports.CATALOG_KEY = 'catalog.json';
/**
 * The key for the version tracking document.
 */
exports.VERSION_TRACKER_KEY = 'all-versions.json';
/**
 * The key for missing documentation report.
 *
 * @param language the language for which missing documentation is requested.
 */
function missingDocumentationReport(language) {
    return `missing-objects/${language.name}-documentation.json`;
}
exports.missingDocumentationReport = missingDocumentationReport;
/**
 * The key for corrupt assembly report.
 *
 * @param language the language for which the report is requested.
 */
function corruptAssemblyReport(language) {
    return `corruptassembly-objects/${language.name}.json`;
}
exports.corruptAssemblyReport = corruptAssemblyReport;
/**
 * The key for uninstallable packages report.
 */
exports.UNINSTALLABLE_PACKAGES_REPORT = 'uninstallable-objects/data.json';
/**
 * The key pattern for objects containing missing documentation lists.
 */
exports.MISSING_DOCUMENTATION_REPORT_PATTERN = 'missing-objects/*-documentation.json';
/**
 * The key pattern for objects containing unprocessable assembly lists.
 */
exports.CORRUPT_ASSEMBLY_REPORT_PATTERN = 'corruptassembly-objects/*.json';
/**
 * A regular expression that can be used to parse out a storage key.
 */
exports.STORAGE_KEY_FORMAT_REGEX = new RegExp(`^${exports.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)/.*$`);
// Capture groups:                                                        ┗━━━━━━━━1━━━━━━━━┛  ┗━━2━━┛
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY29uc3RhbnRzLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL2JhY2tlbmQvc2hhcmVkL2NvbnN0YW50cy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7R0FJRzs7O0FBRUgseUNBQW1EO0FBRW5EOztHQUVHO0FBQ1UsUUFBQSxrQkFBa0IsR0FBRyxPQUFPLENBQUM7QUFFMUM7O0dBRUc7QUFDVSxRQUFBLGtCQUFrQixHQUFHLGNBQWMsQ0FBQztBQUVqRDs7R0FFRztBQUNVLFFBQUEsbUJBQW1CLEdBQUcsZ0JBQWdCLENBQUM7QUFFcEQ7O0dBRUc7QUFDVSxRQUFBLG1CQUFtQixHQUFHLGdCQUFnQixDQUFDO0FBRXBEOztHQUVHO0FBQ1UsUUFBQSwwQkFBMEIsR0FBRyxhQUFhLENBQUMsZ0NBQXFCLENBQUMsVUFBVSxDQUFDLENBQUM7QUFFMUY7O0dBRUc7QUFDVSxRQUFBLHNCQUFzQixHQUFHLGFBQWEsQ0FBQyxnQ0FBcUIsQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUVsRjs7R0FFRztBQUNVLFFBQUEsb0JBQW9CLEdBQUcsYUFBYSxDQUFDLGdDQUFxQixDQUFDLElBQUksQ0FBQyxDQUFDO0FBRTlFOztHQUVHO0FBQ1UsUUFBQSxzQkFBc0IsR0FBRyxhQUFhLENBQUMsZ0NBQXFCLENBQUMsTUFBTSxDQUFDLENBQUM7QUFFbEY7O0dBRUc7QUFDVSxRQUFBLG1CQUFtQixHQUFHLGFBQWEsQ0FBQyxHQUFHLENBQUMsQ0FBQztBQUV0RDs7OztHQUlHO0FBQ0gsU0FBZ0Isa0JBQWtCLENBQUMsV0FBbUIsRUFBRSxjQUF1QjtJQUM3RSxJQUFJLEdBQUcsR0FBRyxHQUFHLDBCQUFrQixHQUFHLFdBQVcsRUFBRSxDQUFDO0lBQ2hELElBQUksY0FBYyxFQUFFO1FBQ2xCLEdBQUcsSUFBSSxLQUFLLGNBQWMsRUFBRSxDQUFDO0tBQzlCO0lBRUQsT0FBTyxHQUFHLENBQUM7QUFDYixDQUFDO0FBUEQsZ0RBT0M7QUFFRDs7R0FFRztBQUNILFNBQWdCLGFBQWEsQ0FBQyxXQUFtQixFQUFFLGNBQXNCO0lBQ3ZFLE1BQU0sTUFBTSxHQUFHLGtCQUFrQixDQUFDLFdBQVcsRUFBRSxjQUFjLENBQUMsQ0FBQztJQUMvRCxPQUFPO1FBQ0wsV0FBVyxFQUFFLEdBQUcsTUFBTSxHQUFHLDJCQUFtQixFQUFFO1FBQzlDLFVBQVUsRUFBRSxHQUFHLE1BQU0sR0FBRywwQkFBa0IsRUFBRTtRQUM1QyxXQUFXLEVBQUUsR0FBRyxNQUFNLEdBQUcsMkJBQW1CLEVBQUU7S0FDL0MsQ0FBQztBQUNKLENBQUM7QUFQRCxzQ0FPQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IsYUFBYSxDQUFDLElBQWtDLEVBQUUsU0FBa0I7SUFDbEYsT0FBTyxTQUFTLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxTQUFTLEdBQUcsQ0FBQyxDQUFDLENBQUMsRUFBRSxHQUFHLElBQUksS0FBSyxDQUFDO0FBQy9ELENBQUM7QUFGRCxzQ0FFQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0Isd0JBQXdCLENBQUMsSUFBa0MsRUFBRSxTQUFrQjtJQUM3RixPQUFPLEdBQUcsYUFBYSxDQUFDLElBQUksRUFBRSxTQUFTLENBQUMsR0FBRywrQkFBdUIsRUFBRSxDQUFDO0FBQ3ZFLENBQUM7QUFGRCw0REFFQztBQUVEOztHQUVHO0FBQ0gsU0FBZ0IscUJBQXFCLENBQUMsSUFBa0MsRUFBRSxTQUFrQjtJQUMxRixPQUFPLEdBQUcsYUFBYSxDQUFDLElBQUksRUFBRSxTQUFTLENBQUMsR0FBRyw0QkFBb0IsRUFBRSxDQUFDO0FBQ3BFLENBQUM7QUFGRCxzREFFQztBQUVEOzs7O0dBSUc7QUFDVSxRQUFBLG9CQUFvQixHQUFHLGdCQUFnQixDQUFDO0FBRXJEOzs7R0FHRztBQUNVLFFBQUEsdUJBQXVCLEdBQUcsa0JBQWtCLENBQUM7QUFFMUQ7O0dBRUc7QUFDVSxRQUFBLDRCQUE0QixHQUFHLGdCQUFnQixDQUFDO0FBRTdEOztHQUVHO0FBQ1UsUUFBQSxnQ0FBZ0MsR0FBRywyQkFBMkIsQ0FBQztBQUU1RTs7R0FFRztBQUNVLFFBQUEsV0FBVyxHQUFHLGNBQWMsQ0FBQztBQUUxQzs7R0FFRztBQUNVLFFBQUEsbUJBQW1CLEdBQUcsbUJBQW1CLENBQUM7QUFFdkQ7Ozs7R0FJRztBQUNILFNBQWdCLDBCQUEwQixDQUFDLFFBQStCO0lBQ3hFLE9BQU8sbUJBQW1CLFFBQVEsQ0FBQyxJQUFJLHFCQUFxQixDQUFDO0FBQy9ELENBQUM7QUFGRCxnRUFFQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFnQixxQkFBcUIsQ0FBQyxRQUErQjtJQUNuRSxPQUFPLDJCQUEyQixRQUFRLENBQUMsSUFBSSxPQUFPLENBQUM7QUFDekQsQ0FBQztBQUZELHNEQUVDO0FBRUQ7O0dBRUc7QUFDVSxRQUFBLDZCQUE2QixHQUFHLGlDQUFpQyxDQUFDO0FBRS9FOztHQUVHO0FBQ1UsUUFBQSxvQ0FBb0MsR0FBRyxzQ0FBc0MsQ0FBQztBQUUzRjs7R0FFRztBQUNVLFFBQUEsK0JBQStCLEdBQUcsZ0NBQWdDLENBQUM7QUFFaEY7O0dBRUc7QUFDVSxRQUFBLHdCQUF3QixHQUFHLElBQUksTUFBTSxDQUFDLElBQUksMEJBQWtCLGtDQUFrQyxDQUFDLENBQUM7QUFDN0csc0dBQXNHIiwic291cmNlc0NvbnRlbnQiOlsiLypcbiAqIElNUE9SVEFOVDogVGhpcyBmaWxlIGlzIHVzZWQgYm90aCBieSBidW5kbGVkIGxhbWJkYSBmdW5jdGlvbnMgYW5kIGJ5XG4gKiBjb25zdHJ1Y3RzLiBUaGlzIG1lYW5zIGl0IHNob3VsZCBub3QgZGlyZWN0bHkgb3IgdHJhbnNpdGl2ZWx5IGltcG9ydCBhbnl0aGluZ1xuICogbm90IHBhcnQgb2YgdGhlIHBhY2thZ2UncyBgZGVwZW5kZW5jaWVzYCwgc3VjaCBhcyB0aGUgYGF3cy1zZGtgLlxuICovXG5cbmltcG9ydCB7IERvY3VtZW50YXRpb25MYW5ndWFnZSB9IGZyb20gJy4vbGFuZ3VhZ2UnO1xuXG4vKipcbiAqIEtleSBwcmVmaXggZm9yIHRoZSBwYWNrYWdlIGRhdGEgc3RvcmFnZS5cbiAqL1xuZXhwb3J0IGNvbnN0IFNUT1JBR0VfS0VZX1BSRUZJWCA9ICdkYXRhLyc7XG5cbi8qKlxuICogS2V5IHN1ZmZpeCBmb3Igc3RvcmluZyBucG0gcGFja2FnZSBidW5kbGVzLlxuICovXG5leHBvcnQgY29uc3QgUEFDS0FHRV9LRVlfU1VGRklYID0gJy9wYWNrYWdlLnRneic7XG5cbi8qKlxuICogS2V5IHN1ZmZpeCBmb3Igc3RvcmluZyBucG0gcGFja2FnZSBtZXRhZGF0YS5cbiAqL1xuZXhwb3J0IGNvbnN0IE1FVEFEQVRBX0tFWV9TVUZGSVggPSAnL21ldGFkYXRhLmpzb24nO1xuXG4vKipcbiAqIFRoZSBrZXkgc3VmZml4IGZvciAoVHlwZVNjcmlwdCkgYXNzZW1ibHkgZmlsZXNcbiAqL1xuZXhwb3J0IGNvbnN0IEFTU0VNQkxZX0tFWV9TVUZGSVggPSAnL2Fzc2VtYmx5Lmpzb24nO1xuXG4vKipcbiAqIFRoZSBrZXkgc3VmZml4IGZvciBhIFR5cGVTY3JpcHQgZG9jIGFydGlmYWN0IChyb290IG1vZHVsZSkuXG4gKi9cbmV4cG9ydCBjb25zdCBET0NTX0tFWV9TVUZGSVhfVFlQRVNDUklQVCA9IGRvY3NLZXlTdWZmaXgoRG9jdW1lbnRhdGlvbkxhbmd1YWdlLlRZUEVTQ1JJUFQpO1xuXG4vKipcbiAqIFRoZSBrZXkgc3VmZml4IGZvciBhIFB5dGhvbiBkb2MgYXJ0aWZhY3QgKHJvb3QgbW9kdWxlKS5cbiAqL1xuZXhwb3J0IGNvbnN0IERPQ1NfS0VZX1NVRkZJWF9QWVRIT04gPSBkb2NzS2V5U3VmZml4KERvY3VtZW50YXRpb25MYW5ndWFnZS5QWVRIT04pO1xuXG4vKipcbiAqIFRoZSBrZXkgc3VmZml4IGZvciBhIFB5dGhvbiBkb2MgYXJ0aWZhY3QgKHJvb3QgbW9kdWxlKS5cbiAqL1xuZXhwb3J0IGNvbnN0IERPQ1NfS0VZX1NVRkZJWF9KQVZBID0gZG9jc0tleVN1ZmZpeChEb2N1bWVudGF0aW9uTGFuZ3VhZ2UuSkFWQSk7XG5cbi8qKlxuICogVGhlIGtleSBzdWZmaXggZm9yIGEgUHl0aG9uIGRvYyBhcnRpZmFjdCAocm9vdCBtb2R1bGUpLlxuICovXG5leHBvcnQgY29uc3QgRE9DU19LRVlfU1VGRklYX0NTSEFSUCA9IGRvY3NLZXlTdWZmaXgoRG9jdW1lbnRhdGlvbkxhbmd1YWdlLkNTSEFSUCk7XG5cbi8qKlxuICogVGhlIGtleSBzdWZmaXggbWF0Y2hpbmcgYW55IGRvY3VtZW50YXRpb24gYXJ0aWZhY3QuXG4gKi9cbmV4cG9ydCBjb25zdCBET0NTX0tFWV9TVUZGSVhfQU5ZID0gZG9jc0tleVN1ZmZpeCgnKicpO1xuXG4vKipcbiAqIFJldHVybiB0aGUgUzMgb2JqZWN0IGtleSBwcmVmaXggZm9yIGEgc3BlY2lmaWMgcGFja2FnZSBuYW1lIGFuZCBvcHRpb25hbGx5IGFcbiAqIHZlcnNpb24uIE5vdGUgdGhhdCB0aGUgcHJlZml4IGRvZXMgbm90IGVuZCB3aXRoIGEgXCIvXCIgc28geW91IHdpbGwgbGlrZWx5IHdhbnRcbiAqIHRvIGFkZCB0aGF0IGlmIHlvdSB3YW50IHRvIG1hdGNoIGEgc3BlY2lmaWMgc2V0IG9mIG9iamVjdHMuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBnZXRPYmplY3RLZXlQcmVmaXgocGFja2FnZU5hbWU6IHN0cmluZywgcGFja2FnZVZlcnNpb24/OiBzdHJpbmcpIHtcbiAgbGV0IGtleSA9IGAke1NUT1JBR0VfS0VZX1BSRUZJWH0ke3BhY2thZ2VOYW1lfWA7XG4gIGlmIChwYWNrYWdlVmVyc2lvbikge1xuICAgIGtleSArPSBgL3Yke3BhY2thZ2VWZXJzaW9ufWA7XG4gIH1cblxuICByZXR1cm4ga2V5O1xufVxuXG4vKipcbiAqIFJlc29sdmVzIHRoZSBzZXQgb2YgUzMgb2JqZWN0IGtleXMgdXNlIGZvciBhIHNwZWNpZmljIHBhY2thZ2UvdmVyc2lvbi5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldE9iamVjdEtleXMocGFja2FnZU5hbWU6IHN0cmluZywgcGFja2FnZVZlcnNpb246IHN0cmluZykge1xuICBjb25zdCBwcmVmaXggPSBnZXRPYmplY3RLZXlQcmVmaXgocGFja2FnZU5hbWUsIHBhY2thZ2VWZXJzaW9uKTtcbiAgcmV0dXJuIHtcbiAgICBhc3NlbWJseUtleTogYCR7cHJlZml4fSR7QVNTRU1CTFlfS0VZX1NVRkZJWH1gLFxuICAgIHBhY2thZ2VLZXk6IGAke3ByZWZpeH0ke1BBQ0tBR0VfS0VZX1NVRkZJWH1gLFxuICAgIG1ldGFkYXRhS2V5OiBgJHtwcmVmaXh9JHtNRVRBREFUQV9LRVlfU1VGRklYfWAsXG4gIH07XG59XG5cbi8qKlxuICogVGhlIGtleSBzdWZmaXggZm9yIGRvY3VtZW50YXRpb24gYXJ0aWZhY3RzIGJ5IGxhbmd1YWdlIGFuZCBzdWJtb2R1bGUuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBkb2NzS2V5U3VmZml4KGxhbmc/OiBEb2N1bWVudGF0aW9uTGFuZ3VhZ2UgfCAnKicsIHN1Ym1vZHVsZT86IHN0cmluZykge1xuICByZXR1cm4gYC9kb2NzLSR7c3VibW9kdWxlID8gYCR7c3VibW9kdWxlfS1gIDogJyd9JHtsYW5nfS5tZGA7XG59XG5cbi8qKlxuICogVGhlIGtleSBzdWZmaXggZm9yIGEgY29ycnVwdGVkIGFzc2VtYmx5IG1hcmtlciBieSBsYW5ndWFnZSBhbmQgc3VibW9kdWxlLlxuICovXG5leHBvcnQgZnVuY3Rpb24gY29ycnVwdEFzc2VtYmx5S2V5U3VmZml4KGxhbmc/OiBEb2N1bWVudGF0aW9uTGFuZ3VhZ2UgfCAnKicsIHN1Ym1vZHVsZT86IHN0cmluZykge1xuICByZXR1cm4gYCR7ZG9jc0tleVN1ZmZpeChsYW5nLCBzdWJtb2R1bGUpfSR7Q09SUlVQVF9BU1NFTUJMWV9TVUZGSVh9YDtcbn1cblxuLyoqXG4gKiBUaGUga2V5IHN1ZmZpeCBmb3IgYSBub3Qgc3VwcG9ydGVkIG1hcmtlciBieSBsYW5ndWFnZSBhbmQgc3VibW9kdWxlLlxuICovXG5leHBvcnQgZnVuY3Rpb24gbm90U3VwcG9ydGVkS2V5U3VmZml4KGxhbmc/OiBEb2N1bWVudGF0aW9uTGFuZ3VhZ2UgfCAnKicsIHN1Ym1vZHVsZT86IHN0cmluZykge1xuICByZXR1cm4gYCR7ZG9jc0tleVN1ZmZpeChsYW5nLCBzdWJtb2R1bGUpfSR7Tk9UX1NVUFBPUlRFRF9TVUZGSVh9YDtcbn1cblxuLyoqXG4gKiBLZXkgc3VmZml4IGZvciBiZWFjb24gZmlsZXMgd2hlbiBhIHBhcnRpY3VsYXIgZmVhdHVyZSBpcyBub3Qgc3VwcG9ydGVkIGZvclxuICogdGhlIHBhcnRpY3VsYXIgcGFja2FnZSAoaS5lOiBQeXRob24gZG9jcyBmb3IgYSBwYWNrYWdlIHRoYXQgZG9lcyBub3QgaGF2ZSBhXG4gKiBQeXRob24gdGFyZ2V0IGNvbmZpZ3VyZWQpLlxuICovXG5leHBvcnQgY29uc3QgTk9UX1NVUFBPUlRFRF9TVUZGSVggPSAnLm5vdC1zdXBwb3J0ZWQnO1xuXG4vKipcbiAqIEtleSBzdWZmaXggZm9yIGJlYWNvbiBmaWxlcyBtYXJraW5nIHRoYXQgYSBsYW5ndWFnZSBzcGVjaWZpYyBhc3NlbWJseSBpcyBjb3JydXB0XG4gKiBhbmQgd2UgY2Fubm90IGdlbmVyYXRlIGRvY3MgZnJvbSBpdC5cbiAqL1xuZXhwb3J0IGNvbnN0IENPUlJVUFRfQVNTRU1CTFlfU1VGRklYID0gJy5jb3JydXB0YXNzZW1ibHknO1xuXG4vKipcbiAqIEtleSBzdWZmaXggZm9yIGEgYmVhY29uIGZpbGUgd2hlbiBhIHBhY2thZ2UgY2Fubm90IGJlIGluc3RhbGxlZC5cbiAqL1xuZXhwb3J0IGNvbnN0IFVOSU5TVEFMTEFCTEVfUEFDS0FHRV9TVUZGSVggPSAnL3VuaW5zdGFsbGFibGUnO1xuXG4vKipcbiAqIE5hbWUgb2YgdGhlIGVycm9yIGRlbm90aW5nIGFuIHVucHJvY2Vzc2FibGUgcGFja2FnZSB0aGF0IHNob3VsZCBiZSBkaXZlcnRlZCBhd2F5IGZyb20gdGhlIERMUS5cbiAqL1xuZXhwb3J0IGNvbnN0IFVOUFJPQ0VTU0FCTEVfUEFDS0FHRV9FUlJPUl9OQU1FID0gJ1VucHJvY2Vzc2FibGVQYWNrYWdlRXJyb3InO1xuXG4vKipcbiAqIFRoZSBrZXkgZm9yIHRoZSBjYXRhbG9nIGRvY3VtZW50LlxuICovXG5leHBvcnQgY29uc3QgQ0FUQUxPR19LRVkgPSAnY2F0YWxvZy5qc29uJztcblxuLyoqXG4gKiBUaGUga2V5IGZvciB0aGUgdmVyc2lvbiB0cmFja2luZyBkb2N1bWVudC5cbiAqL1xuZXhwb3J0IGNvbnN0IFZFUlNJT05fVFJBQ0tFUl9LRVkgPSAnYWxsLXZlcnNpb25zLmpzb24nO1xuXG4vKipcbiAqIFRoZSBrZXkgZm9yIG1pc3NpbmcgZG9jdW1lbnRhdGlvbiByZXBvcnQuXG4gKlxuICogQHBhcmFtIGxhbmd1YWdlIHRoZSBsYW5ndWFnZSBmb3Igd2hpY2ggbWlzc2luZyBkb2N1bWVudGF0aW9uIGlzIHJlcXVlc3RlZC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIG1pc3NpbmdEb2N1bWVudGF0aW9uUmVwb3J0KGxhbmd1YWdlOiBEb2N1bWVudGF0aW9uTGFuZ3VhZ2UpOiBzdHJpbmcge1xuICByZXR1cm4gYG1pc3Npbmctb2JqZWN0cy8ke2xhbmd1YWdlLm5hbWV9LWRvY3VtZW50YXRpb24uanNvbmA7XG59XG5cbi8qKlxuICogVGhlIGtleSBmb3IgY29ycnVwdCBhc3NlbWJseSByZXBvcnQuXG4gKlxuICogQHBhcmFtIGxhbmd1YWdlIHRoZSBsYW5ndWFnZSBmb3Igd2hpY2ggdGhlIHJlcG9ydCBpcyByZXF1ZXN0ZWQuXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBjb3JydXB0QXNzZW1ibHlSZXBvcnQobGFuZ3VhZ2U6IERvY3VtZW50YXRpb25MYW5ndWFnZSk6IHN0cmluZyB7XG4gIHJldHVybiBgY29ycnVwdGFzc2VtYmx5LW9iamVjdHMvJHtsYW5ndWFnZS5uYW1lfS5qc29uYDtcbn1cblxuLyoqXG4gKiBUaGUga2V5IGZvciB1bmluc3RhbGxhYmxlIHBhY2thZ2VzIHJlcG9ydC5cbiAqL1xuZXhwb3J0IGNvbnN0IFVOSU5TVEFMTEFCTEVfUEFDS0FHRVNfUkVQT1JUID0gJ3VuaW5zdGFsbGFibGUtb2JqZWN0cy9kYXRhLmpzb24nO1xuXG4vKipcbiAqIFRoZSBrZXkgcGF0dGVybiBmb3Igb2JqZWN0cyBjb250YWluaW5nIG1pc3NpbmcgZG9jdW1lbnRhdGlvbiBsaXN0cy5cbiAqL1xuZXhwb3J0IGNvbnN0IE1JU1NJTkdfRE9DVU1FTlRBVElPTl9SRVBPUlRfUEFUVEVSTiA9ICdtaXNzaW5nLW9iamVjdHMvKi1kb2N1bWVudGF0aW9uLmpzb24nO1xuXG4vKipcbiAqIFRoZSBrZXkgcGF0dGVybiBmb3Igb2JqZWN0cyBjb250YWluaW5nIHVucHJvY2Vzc2FibGUgYXNzZW1ibHkgbGlzdHMuXG4gKi9cbmV4cG9ydCBjb25zdCBDT1JSVVBUX0FTU0VNQkxZX1JFUE9SVF9QQVRURVJOID0gJ2NvcnJ1cHRhc3NlbWJseS1vYmplY3RzLyouanNvbic7XG5cbi8qKlxuICogQSByZWd1bGFyIGV4cHJlc3Npb24gdGhhdCBjYW4gYmUgdXNlZCB0byBwYXJzZSBvdXQgYSBzdG9yYWdlIGtleS5cbiAqL1xuZXhwb3J0IGNvbnN0IFNUT1JBR0VfS0VZX0ZPUk1BVF9SRUdFWCA9IG5ldyBSZWdFeHAoYF4ke1NUT1JBR0VfS0VZX1BSRUZJWH0oKD86QFteL10rLyk/W14vXSspL3YoW14vXSspLy4qJGApO1xuLy8gQ2FwdHVyZSBncm91cHM6ICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICDilJfilIHilIHilIHilIHilIHilIHilIHilIEx4pSB4pSB4pSB4pSB4pSB4pSB4pSB4pSB4pSbICDilJfilIHilIEy4pSB4pSB4pSbXG4iXX0=