"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../backend/shared/integrity.lambda-shared");
const tarball_lambda_shared_1 = require("../../backend/shared/tarball.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const DETAIL_TYPE = 'CodeArtifact Package Version State Change';
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    metrics.setNamespace(constants_lambda_shared_1.METRICS_NAMESPACE);
    metrics.setDimensions({
        [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: event.detail.domainOwner,
        [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: event.detail.domainName,
        [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: event.detail.repositoryName,
    });
    const packageName = event.detail.packageNamespace
        ? `@${event.detail.packageNamespace}/${event.detail.packageName}`
        : event.detail.packageName;
    if (event.detail.operationType === 'Deleted') {
        console.log(`Operation type is "Deleted" for ${packageName}@${event.detail.packageVersion}`);
        metrics.putMetric("DeletedCount" /* DELETED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
        return;
    }
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const denied = denyList.lookup(packageName, event.detail.packageVersion);
    if (denied) {
        console.log(`Package ${packageName}@${event.detail.packageVersion} denied: ${JSON.stringify(denied, null, 2)}`);
        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
        return;
    }
    const { asset, packageVersionRevision } = await aws.codeArtifact().getPackageVersionAsset({
        asset: 'package.tgz',
        domainOwner: event.detail.domainOwner,
        domain: event.detail.domainName,
        repository: event.detail.repositoryName,
        format: event.detail.packageFormat,
        namespace: event.detail.packageNamespace,
        package: event.detail.packageName,
        packageVersion: event.detail.packageVersion,
    }).promise();
    const tarball = Buffer.from(asset);
    const { assemblyJson, packageJson } = await tarball_lambda_shared_1.extractObjects(tarball, {
        assemblyJson: { path: 'package/.jsii' },
        packageJson: { path: 'package/package.json', required: true },
    });
    metrics.putMetric("NotJsiiEnabledCount" /* NOT_JSII_ENABLED_COUNT */, assemblyJson ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
    if (assemblyJson == null) {
        console.log(`Package "${packageName}@${event.detail.packageVersion}" does not contain a .jsii assembly`);
        return;
    }
    const metadata = JSON.parse(packageJson.toString('utf-8'));
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const eligibleLicense = licenseList.lookup((_a = metadata.license) !== null && _a !== void 0 ? _a : 'UNLICENSED');
    metrics.putMetric("IneligibleLicenseCount" /* INELIGIBLE_LICENSE */, eligibleLicense ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
    if (!eligibleLicense) {
        console.log(`Package "${packageName}@${event.detail.packageVersion}" does not use allow-listed license: ${(_b = metadata.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
        return;
    }
    const stagingKey = `${packageName}/${event.detail.packageVersion}/${packageVersionRevision}/package.tgz`;
    await aws.s3().putObject({
        Bucket: stagingBucket,
        Key: stagingKey,
        Body: asset,
        ContentType: 'application/octet-stream',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
        },
    }).promise();
    const message = integrity_lambda_shared_1.integrity({
        tarballUri: `s3://${stagingBucket}/${stagingKey}`,
        metadata: { resources: event.resources.join(', ') },
        time: event.time,
    }, tarball);
    return aws.sqs().sendMessage({
        MessageAttributes: {
            AWS_REQUEST_ID: { DataType: 'String', StringValue: context.awsRequestId },
            LOG_GROUP_NAME: { DataType: 'String', StringValue: context.logGroupName },
            LOG_STREAM_NAME: { DataType: 'String', StringValue: context.logStreamName },
        },
        MessageBody: JSON.stringify(message),
        MessageDeduplicationId: event.detail.eventDeduplicationId,
        QueueUrl: queueUrl,
    }).promise();
});
//# sourceMappingURL=data:application/json;base64,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