"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const index_1 = require("../lib/index");
const sfn = require("@aws-cdk/aws-stepfunctions");
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
const aws_s3_1 = require("@aws-cdk/aws-s3");
function deployNewStateMachine(stack) {
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        }
    };
    return new index_1.S3ToStepFunction(stack, 'test-s3-step-function', props);
}
test('snapshot test S3ToStepFunction default params', () => {
    const stack = new cdk.Stack();
    deployNewStateMachine(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('check deployCloudTrail = false', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        deployCloudTrail: false
    };
    const construct = new index_1.S3ToStepFunction(stack, 'test-s3-step-function', props);
    expect(construct.cloudtrail === undefined);
});
test('override eventRuleProps', () => {
    const stack = new cdk.Stack();
    const mybucket = new aws_s3_1.Bucket(stack, 'mybucket');
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        existingBucketObj: mybucket,
        eventRuleProps: {
            eventPattern: {
                source: ['aws.s3'],
                detailType: ['AWS API Call via CloudTrail'],
                detail: {
                    eventSource: [
                        "s3.amazonaws.com"
                    ],
                    eventName: [
                        "GetObject"
                    ],
                    requestParameters: {
                        bucketName: [
                            mybucket.bucketName
                        ]
                    }
                }
            }
        }
    };
    new index_1.S3ToStepFunction(stack, 'test-s3-step-function', props);
    expect(stack).toHaveResource('AWS::Events::Rule', {
        EventPattern: {
            "source": [
                "aws.s3"
            ],
            "detail-type": [
                "AWS API Call via CloudTrail"
            ],
            "detail": {
                eventSource: [
                    "s3.amazonaws.com"
                ],
                eventName: [
                    "GetObject"
                ],
                requestParameters: {
                    bucketName: [
                        {
                            Ref: "mybucket160F8132"
                        }
                    ]
                }
            }
        },
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "tests3stepfunctiontesteventsrulestepfunctionstackStateMachine5A6C0DFF"
                },
                Id: "Target0",
                RoleArn: {
                    "Fn::GetAtt": [
                        "tests3stepfunctiontesteventsrulestepfunctionstackEventsRuleRoleF447A174",
                        "Arn"
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStateMachine(stack);
    expect(construct.cloudtrail !== null);
    expect(construct.stateMachine !== null);
    expect(construct.s3Bucket !== null);
    expect(construct.cloudwatchAlarms !== null);
    expect(construct.stateMachineLogGroup !== null);
    expect(construct.s3LoggingBucket !== null);
    expect(construct.cloudtrail !== null);
    expect(construct.cloudtrailBucket !== null);
    expect(construct.cloudtrailLoggingBucket !== null);
});
//# sourceMappingURL=data:application/json;base64,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