# AUTOGENERATED! DO NOT EDIT! File to edit: 04_solar.nsrdb.ipynb (unless otherwise specified).

__all__ = ['NSRDB_DataQuery', 'get_nsrdb_download_links', 'download_nsrdb_data']

# Cell
from .._core import get_request, check_api_key, _API_KEY

class NSRDB_DataQuery:
    """Returns information on the closest NSRDB datasets for a location
    including a set of links that can be used to download the data.
    """

    QUERY_URL = "/api/solar/nsrdb_data_query.json"

    def __init__(self,
                 api_key=None,
                 wkt=None,
                 address=None,
                 lat=None,
                 lon=None,
                 dataset_type=None,
                 show_empty=False,
                ):

        if api_key is None:
            api_key = check_api_key()

        self._params = {"api_key": api_key}

        # if well-known text is not provided look for address or lat/lon
        if not wkt:
            if not address:
                self._params.update({"lat" : lat, "lon" : lon})
            else:
                self._params.update({"address" : address})
        elif wkt:
            self._params.update({"wkt" : wkt})
        else:
            raise AttributeError("Need to specify a location. Use lat/lon ot wkt or address.")

        if dataset_type:
            self._params.update({"type" : dataset_type})

        if show_empty:
            self._params.update({"show_empty" : show_empty})

        r = get_request(self.QUERY_URL, self._params)

        # complete response as a dictionary
        self.response = r.json()

        self.outputs = self.response["outputs"]


# Cell
from itertools import chain


def get_nsrdb_download_links(
    year,
    interval=None,
    api_key=None,
    wkt=None,
    address=None,
    lat=None,
    lon=None,
    dataset_type=None,
    show_empty=False
):
    """Get NSRDB dowload links from data query for the specified location"""

    if api_key is None:
        api_key = check_api_key()

    nsrdb_data_query = NSRDB_DataQuery(
        api_key=api_key,
        wkt=wkt,
        address=address,
        lat=lat,
        lon=lon,
        dataset_type=dataset_type,
        show_empty=show_empty,
    )

    outputs = nsrdb_data_query.outputs

    available_years = []
    available_intervals = []
    links = []

    # find available years
    for i in range(len(outputs)):
        available_years.append(outputs[i]["availableYears"])

    # if year specified not available, raise Exception
    if year not in list(chain.from_iterable(available_years)):
        raise Exception("No data for the specified year.")

    # if interval is provided, check its availability
    if interval:
        # find avaialble intervals
        for i in range(len(outputs)):
            available_intervals.append(outputs[i]["availableIntervals"])

        # if time interval not available, raise Exception
        if interval not in list(chain.from_iterable(available_intervals)):
            raise Exception("No data for the specified time interval.")


    for i in range(len(outputs)):
        list_of_links = outputs[i]["links"] # this returns a list of dicts

        for j in range(len(list_of_links)):

            # the str call is to cover 'tmy-*' style named years
            if str(year) in str(list_of_links[j]["year"]):

                # if interval is provided, give those links only
                if interval:
                    if list_of_links[j]["interval"] == interval:
                        links.append(list_of_links[j]["link"])
                else:
                    links.append(list_of_links[j]["link"])


    return links

# Cell
import pandas as pd
import warnings

def download_nsrdb_data(
    link,
    email,
    api_key=None,
):
    """Download NSRDB data from the provided link and returns a pandas DataFrame."""

    if not isinstance(link, str):
        raise ValueError(f"Requires a str type. You provided {type(link)} type")

    if api_key is None:
        api_key = check_api_key()

    link = link.replace("yourapikey", api_key)
    link = link.replace("youremail", email)

    # get and split the parameters
    p = link.split("?")[-1].split("&")

    for i in p:
        i = i.split("=")
        if i[0] == "interval":
            interval = i[1]
        if i[0] == "names":
            year = i[1]  # this will also catch "tmy-*" names in years & and lead to errors in date_range

    df = pd.read_csv(link, skiprows=2)

    try:
        # if this doesn't raise valuerror, then we can go ahead and set the new index
        int(year)
        df = df.set_index(pd.date_range(f"1/1/{year}", freq=interval+'Min', periods=525600/int(interval)))

    except ValueError:
        warnings.warn("Could not set the index to datetime; please do it manually", UserWarning)

    return df