"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DjangoEcs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const patterns = require("@aws-cdk/aws-ecs-patterns");
const logs = require("@aws-cdk/aws-logs");
const s3 = require("@aws-cdk/aws-s3");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const database_1 = require("./common/database");
const elasticache_1 = require("./common/elasticache");
const vpc_1 = require("./common/vpc");
const beat_1 = require("./ecs/celery/beat");
const worker_1 = require("./ecs/celery/worker");
const tasks_1 = require("./ecs/tasks");
/**
 * Configures a Django project using ECS Fargate.
 *
 * @stability stable
 */
class DjangoEcs extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b;
        super(scope, id);
        /**
         * VPC must have public, private and isolated subnets
         *
         * If you don't provide a VPC, a new VPC will be created
         */
        if (!props.vpc) {
            const applicationVpc = new vpc_1.ApplicationVpc(scope, 'AppVpc');
            this.vpc = applicationVpc.vpc;
        }
        else {
            const vpc = props.vpc;
            this.vpc = vpc;
        }
        /**
         * static files bucket name is derived from the Construct id if not provided
         */
        const staticFilesBucket = new s3.Bucket(scope, 'StaticBucket', {
            bucketName: props === null || props === void 0 ? void 0 : props.bucketName,
        });
        this.staticFileBucket = staticFilesBucket;
        /**
         * ECS cluster
         */
        this.cluster = new ecs.Cluster(scope, 'EcsCluster', { vpc: this.vpc });
        /**
         * task definition construct
         */
        const taskDefinition = new ecs.TaskDefinition(scope, 'TaskDefinition', {
            compatibility: ecs.Compatibility.FARGATE,
            cpu: '256',
            memoryMiB: '512',
        });
        /**
         * Secret used for RDS postgres password
         */
        this.secret = new secretsmanager.Secret(scope, 'dbSecret', {
            secretName: 'dbSecret',
            description: 'secret for rds',
        });
        /**
         * Container image used in web API, celery worker and management task containers
         */
        this.image = new ecs.AssetImage(props.imageDirectory);
        /**
         * RDS managed database using PostgreSQL
         */
        const database = new database_1.RdsPostgresInstance(scope, 'RdsPostgresInstance', {
            vpc: this.vpc,
            secret: this.secret,
        });
        /**
         * A security group in the VPC for our application (ECS Fargate services and tasks)
         * Allow the application services to access the RDS security group
         */
        const appSecurityGroup = new ec2.SecurityGroup(scope, 'appSecurityGroup', {
            vpc: this.vpc,
        });
        /**
         * ElastiCache Redis Cluster for caching, celery message brokering
         */
        const elastiCacheRedis = new elasticache_1.ElastiCacheCluster(scope, 'ElastiCacheCluster', {
            vpc: this.vpc,
            appSecurityGroup,
        });
        const environment = {
            AWS_STORAGE_BUCKET_NAME: staticFilesBucket.bucketName,
            POSTGRES_SERVICE_HOST: database.rdsPostgresInstance.dbInstanceEndpointAddress,
            POSTGRES_PASSWORD: this.secret.secretValue.toString(),
            DEBUG: '0',
            DJANGO_SETTINGS_MODULE: 'backend.settings.production',
            REDIS_SERVICE_HOST: elastiCacheRedis.elastiCacheCluster.attrRedisEndpointAddress,
        };
        taskDefinition.addContainer('backendContainer', {
            image: this.image,
            environment,
            command: props.webCommand,
            portMappings: [{
                    containerPort: 8000,
                    hostPort: 8000,
                }],
            logging: ecs.LogDriver.awsLogs({
                logRetention: logs.RetentionDays.ONE_DAY,
                streamPrefix: 'BackendContainer',
            }),
        });
        new tasks_1.managementCommandTask(scope, 'migrate', {
            image: this.image,
            command: ['python3', 'manage.py', 'migrate', '--no-input'],
            appSecurityGroup,
            environment,
        });
        new tasks_1.managementCommandTask(scope, 'collectstatic', {
            image: this.image,
            command: ['python3', 'manage.py', 'collectstatic', '--no-input'],
            appSecurityGroup,
            environment,
        });
        /**
         * Use celery beat if it is configured in props
         */
        if ((_b = props.useCeleryBeat) !== null && _b !== void 0 ? _b : false) {
            new beat_1.CeleryBeat(scope, 'CeleryBeat', {
                image: this.image,
                command: [
                    'celery',
                    '--app=backend.celery_app:app',
                    'beat',
                    '--loglevel=INFO',
                    '--pidfile=/code/celerybeat.pid',
                ],
                environment,
                cluster: this.cluster,
                securityGroups: [appSecurityGroup],
            });
        }
        ;
        /**
         * Celery worker
         *
         * TODO: refactor to support defining multiple queues
         */
        new worker_1.CeleryWorker(scope, 'CeleryWorkerDefaultQueue', {
            image: this.image,
            command: [
                'celery',
                '-A',
                'backend.celery_app:app',
                'worker',
                '-l',
                'info',
                '-Q',
                'celery',
                '-n',
                'worker-celery@%h',
            ],
            environment,
            cluster: this.cluster,
            securityGroups: [appSecurityGroup],
        });
        /**
         * ECS load-balanced fargate service
         */
        const albfs = new patterns.ApplicationLoadBalancedFargateService(scope, 'AlbFargateService', {
            cluster: this.cluster,
            taskDefinition,
            securityGroups: [appSecurityGroup],
            desiredCount: 1,
            assignPublicIp: true,
        });
        const albLogsBucket = new s3.Bucket(scope, `${id}-alb-logs`);
        albfs.loadBalancer.logAccessLogs(albLogsBucket);
        /**
         * Health check for the application load balancer
         */
        albfs.targetGroup.configureHealthCheck({
            path: '/api/health-check/',
        });
        /**
         * Allows the app security group to communicate with the RDS security group
         */
        database.rdsSecurityGroup.addIngressRule(appSecurityGroup, ec2.Port.tcp(5432));
        /**
         * Grant the task defintion read-write access to static files bucket
         */
        staticFilesBucket.grantReadWrite(albfs.taskDefinition.taskRole);
        new cdk.CfnOutput(this, 'bucketName', { value: staticFilesBucket.bucketName });
        new cdk.CfnOutput(this, 'apiUrl', { value: albfs.loadBalancer.loadBalancerFullName });
    }
}
exports.DjangoEcs = DjangoEcs;
_a = JSII_RTTI_SYMBOL_1;
DjangoEcs[_a] = { fqn: "django-cdk.DjangoEcs", version: "0.0.13" };
//# sourceMappingURL=data:application/json;base64,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