"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DjangoEks = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const ec2 = require("@aws-cdk/aws-ec2");
const ecrAssets = require("@aws-cdk/aws-ecr-assets");
const eks = require("@aws-cdk/aws-eks");
// import * as logs from '@aws-cdk/aws-logs';
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const secretsmanager = require("@aws-cdk/aws-secretsmanager");
const cdk = require("@aws-cdk/core");
const database_1 = require("./common/database");
const vpc_1 = require("./common/vpc");
const awslbc_1 = require("./eks/awslbc");
// import { ElastiCacheCluster } from './elasticache';
const irsa_1 = require("./eks/irsa");
const ingress_1 = require("./eks/resources/ingress");
const migrate_1 = require("./eks/resources/migrate");
const web_1 = require("./eks/resources/web");
/**
 * Configures a Django project using EKS.
 *
 * @stability stable
 */
class DjangoEks extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        /**
         * VPC must have public, private and isolated subnets
         *
         * If you don't provide a VPC, a new VPC will be created
         */
        if (!props.vpc) {
            const applicationVpc = new vpc_1.ApplicationVpc(scope, 'AppVpc');
            this.vpc = applicationVpc.vpc;
        }
        else {
            const vpc = props.vpc;
            this.vpc = vpc;
        }
        /**
         * static files bucket name is derived from the Construct id if not provided
         */
        const staticFilesBucket = new s3.Bucket(scope, 'StaticBucket', {
            bucketName: props.bucketName,
            autoDeleteObjects: true,
            removalPolicy: cdk.RemovalPolicy.DESTROY,
        });
        this.staticFileBucket = staticFilesBucket;
        // allow all account users to assume this role in order to admin the cluster
        const mastersRole = new iam.Role(this, 'AdminRole', {
            assumedBy: new iam.AccountRootPrincipal(),
        });
        /**
         * EKS cluster
         */
        this.cluster = new eks.Cluster(this, 'MyEksCluster', {
            version: eks.KubernetesVersion.V1_19,
            vpc: this.vpc,
            mastersRole,
            defaultCapacity: 2,
        });
        /**
         * Namespace for application
         */
        const appNamespace = this.cluster.addManifest('app-namespace', {
            apiVersion: 'v1',
            kind: 'Namespace',
            metadata: {
                name: 'app',
            },
        });
        const DB_SECRET_NAME = 'dbSecret';
        /**
         * Secret used for RDS postgres password
         */
        this.secret = new secretsmanager.Secret(scope, 'dbSecret', {
            secretName: DB_SECRET_NAME,
            description: 'secret for rds',
            generateSecretString: {
                secretStringTemplate: JSON.stringify({ username: 'postgres' }),
                generateStringKey: 'password',
                excludePunctuation: true,
                includeSpace: false,
            },
        });
        /**
         * Creates an IAM role with a trust relationship that is scoped to the cluster's OIDC provider
         * https://docs.aws.amazon.com/eks/latest/userguide/iam-roles-for-service-accounts-technical-overview.html
         *
         * The role (podRole) will be given access the Secrets Manager Secret, the S3 bucket and
         * any other resources that are needed by pods that run the Django application
         */
        const irsa = new irsa_1.Irsa(scope, 'Irsa', {
            cluster: this.cluster,
        });
        /**
         * Make sure that the namespace has been deployed
         */
        irsa.node.addDependency(appNamespace);
        /**
         * Give the IRSA podRole read access to the bucket and read/write access to the S3 bucket
         */
        this.secret.grantRead(irsa.podRole);
        this.staticFileBucket.grantReadWrite(irsa.podRole);
        /**
         * RDS instance
         */
        const database = new database_1.RdsPostgresInstance(scope, 'RdsPostgresInstance', {
            vpc: this.vpc,
            secret: this.secret,
        });
        /**
         * Security Group for worker nodes
         */
        const appSecurityGroup = new ec2.SecurityGroup(scope, 'appSecurityGroup', {
            vpc: this.vpc,
        });
        /**
         * cluster.defaultCapactiy is the autoScalingGroup (the cluster's default node group)
         * Here the appSecurityGroup created above is added to that ASG
         *
         * TODO: use a non-default node-group
         */
        (_b = this.cluster.defaultCapacity) === null || _b === void 0 ? void 0 : _b.addSecurityGroup(appSecurityGroup);
        /**
         * Allow th ASG to accesss the database
         */
        database.rdsSecurityGroup.addIngressRule(appSecurityGroup, ec2.Port.tcp(5432));
        /**
         * Installation of AWS Load Balancer Controller
         * https://kubernetes-sigs.github.io/aws-load-balancer-controller/v2.2/deploy/installation/
         */
        new awslbc_1.AwsLoadBalancerController(this, 'AwsLoadBalancerController', {
            cluster: this.cluster,
        });
        /**
         * Backend docker image using DockerImageAssets
         * This image will be pushed to the ECR Registry created by `cdk bootstrap`
         */
        const backendImage = new ecrAssets.DockerImageAsset(scope, 'backendImage', {
            directory: props.imageDirectory,
        });
        /**
         * Common environment variables for Jobs and Deployments
         */
        const env = [
            {
                name: 'DEBUG',
                value: '0',
            },
            {
                // this is used in the application to fetch the secret value
                name: 'DB_SECRET_NAME',
                value: DB_SECRET_NAME,
            },
            {
                name: 'POSTGRES_SERVICE_HOST',
                value: database.rdsPostgresInstance.dbInstanceEndpointAddress,
            },
            {
                name: 'DJANGO_SETTINGS_MODULE',
                value: 'backend.settings.production',
            },
        ];
        // Django K8s resources
        /**
         * Kubernetes Job that runs database migrations
         */
        new migrate_1.MigrateJob(scope, 'django-migrate-job', {
            cluster: this.cluster,
            backendImage,
            namespace: 'app',
            env,
        });
        // web service and deployment
        const webResources = new web_1.WebResources(scope, 'web-resources', {
            env,
            cluster: this.cluster,
            webCommand: (_c = props.webCommand) !== null && _c !== void 0 ? _c : ['./scripts/start_prod.sh'],
            backendImage,
            namespace: 'app',
        });
        // webResources.node.addDependency(appNamespace);
        this.cluster.addManifest('web-deployment', webResources.deploymentManifest);
        this.cluster.addManifest('web-service', webResources.serviceManifest);
        /**
         * Add deployment and service manifests for web to the cluster
         */
        this.cluster.addManifest('app-ingresss', ingress_1.appIngress);
        // ingress.node.addDependency(webService);
        /**
         * Get the ALB address using KubernetesObjectValue as a String
         */
        // https://github.com/aws/aws-cdk/issues/14933
        // const albAddress = new eks.KubernetesObjectValue(scope, 'AlbAddress', {
        //   cluster: this.cluster,
        //   objectType: 'ingress',
        //   objectNamespace: 'app',
        //   objectName: 'app-ingress',
        //   jsonPath: '.items[0].status.loadBalancer.ingress[0].hostname',
        // });
        /**
         * Route53 A Record pointing to ALB that is created by AWS Application Load Balancer Controller
         *
         * TODO: fix this, since KubernetesObjectValue is not giving the ALB Public DNS URL
         */
        const alb = elbv2.ApplicationLoadBalancer.fromLookup(this, 'appAlb', {
            loadBalancerTags: {
                Environment: 'test',
            },
        });
        /**
         * Output the Load Balancer URL as a CfnOutput
         */
        new cdk.CfnOutput(this, 'apiUrl', { value: alb.loadBalancerDnsName });
    }
}
exports.DjangoEks = DjangoEks;
_a = JSII_RTTI_SYMBOL_1;
DjangoEks[_a] = { fqn: "django-cdk.DjangoEks", version: "0.0.13" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGphbmdvLWVrcy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uL3NyYy9kamFuZ28tZWtzLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7Ozs7O0FBQUEsd0NBQXdDO0FBQ3hDLHFEQUFxRDtBQUNyRCx3Q0FBd0M7QUFDeEMsNkNBQTZDO0FBQzdDLDZEQUE2RDtBQUM3RCx3Q0FBd0M7QUFDeEMsc0NBQXNDO0FBQ3RDLDhEQUE4RDtBQUM5RCxxQ0FBcUM7QUFDckMsZ0RBQXdEO0FBQ3hELHNDQUE4QztBQUM5Qyx5Q0FBeUQ7QUFDekQsc0RBQXNEO0FBQ3RELHFDQUFrQztBQUNsQyxxREFBcUQ7QUFDckQscURBQXFEO0FBQ3JELDZDQUFtRDs7Ozs7O0FBZ0RuRCxNQUFhLFNBQVUsU0FBUSxHQUFHLENBQUMsU0FBUzs7OztJQVExQyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQXFCOztRQUNqRSxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsQ0FBQyxDQUFDO1FBRWpCOzs7O1dBSUc7UUFDSCxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsRUFBRTtZQUNkLE1BQU0sY0FBYyxHQUFHLElBQUksb0JBQWMsQ0FBQyxLQUFLLEVBQUUsUUFBUSxDQUFDLENBQUM7WUFDM0QsSUFBSSxDQUFDLEdBQUcsR0FBRyxjQUFjLENBQUMsR0FBRyxDQUFDO1NBQy9CO2FBQU07WUFDTCxNQUFNLEdBQUcsR0FBRyxLQUFLLENBQUMsR0FBRyxDQUFDO1lBQ3RCLElBQUksQ0FBQyxHQUFHLEdBQUcsR0FBRyxDQUFDO1NBQ2hCO1FBRUQ7O1dBRUc7UUFDSCxNQUFNLGlCQUFpQixHQUFHLElBQUksRUFBRSxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQzdELFVBQVUsRUFBRSxLQUFLLENBQUMsVUFBVTtZQUM1QixpQkFBaUIsRUFBRSxJQUFJO1lBQ3ZCLGFBQWEsRUFBRSxHQUFHLENBQUMsYUFBYSxDQUFDLE9BQU87U0FDekMsQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLGdCQUFnQixHQUFHLGlCQUFpQixDQUFDO1FBRTFDLDRFQUE0RTtRQUM1RSxNQUFNLFdBQVcsR0FBRyxJQUFJLEdBQUcsQ0FBQyxJQUFJLENBQUMsSUFBSSxFQUFFLFdBQVcsRUFBRTtZQUNsRCxTQUFTLEVBQUUsSUFBSSxHQUFHLENBQUMsb0JBQW9CLEVBQUU7U0FDMUMsQ0FBQyxDQUFDO1FBRUg7O1dBRUc7UUFDSCxJQUFJLENBQUMsT0FBTyxHQUFHLElBQUksR0FBRyxDQUFDLE9BQU8sQ0FBQyxJQUFJLEVBQUUsY0FBYyxFQUFFO1lBQ25ELE9BQU8sRUFBRSxHQUFHLENBQUMsaUJBQWlCLENBQUMsS0FBSztZQUNwQyxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7WUFDYixXQUFXO1lBQ1gsZUFBZSxFQUFFLENBQUM7U0FDbkIsQ0FBQyxDQUFDO1FBRUg7O1dBRUc7UUFDSCxNQUFNLFlBQVksR0FBRyxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxlQUFlLEVBQUU7WUFDN0QsVUFBVSxFQUFFLElBQUk7WUFDaEIsSUFBSSxFQUFFLFdBQVc7WUFDakIsUUFBUSxFQUFFO2dCQUNSLElBQUksRUFBRSxLQUFLO2FBQ1o7U0FDRixDQUFDLENBQUM7UUFFSCxNQUFNLGNBQWMsR0FBRyxVQUFVLENBQUM7UUFDbEM7O1dBRUc7UUFDSCxJQUFJLENBQUMsTUFBTSxHQUFHLElBQUksY0FBYyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsVUFBVSxFQUFFO1lBQ3pELFVBQVUsRUFBRSxjQUFjO1lBQzFCLFdBQVcsRUFBRSxnQkFBZ0I7WUFDN0Isb0JBQW9CLEVBQUU7Z0JBQ3BCLG9CQUFvQixFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxRQUFRLEVBQUUsVUFBVSxFQUFFLENBQUM7Z0JBQzlELGlCQUFpQixFQUFFLFVBQVU7Z0JBQzdCLGtCQUFrQixFQUFFLElBQUk7Z0JBQ3hCLFlBQVksRUFBRSxLQUFLO2FBQ3BCO1NBQ0YsQ0FBQyxDQUFDO1FBRUg7Ozs7OztXQU1HO1FBQ0gsTUFBTSxJQUFJLEdBQUcsSUFBSSxXQUFJLENBQUMsS0FBSyxFQUFFLE1BQU0sRUFBRTtZQUNuQyxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87U0FDdEIsQ0FBQyxDQUFDO1FBRUg7O1dBRUc7UUFDSCxJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxZQUFZLENBQUMsQ0FBQztRQUV0Qzs7V0FFRztRQUNILElBQUksQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUNwQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxPQUFPLENBQUMsQ0FBQztRQUVuRDs7V0FFRztRQUNILE1BQU0sUUFBUSxHQUFHLElBQUksOEJBQW1CLENBQUMsS0FBSyxFQUFFLHFCQUFxQixFQUFFO1lBQ3JFLEdBQUcsRUFBRSxJQUFJLENBQUMsR0FBRztZQUNiLE1BQU0sRUFBRSxJQUFJLENBQUMsTUFBTTtTQUNwQixDQUFDLENBQUM7UUFFSDs7V0FFRztRQUNILE1BQU0sZ0JBQWdCLEdBQUcsSUFBSSxHQUFHLENBQUMsYUFBYSxDQUFDLEtBQUssRUFBRSxrQkFBa0IsRUFBRTtZQUN4RSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUc7U0FDZCxDQUFDLENBQUM7UUFFSDs7Ozs7V0FLRztRQUNILE1BQUEsSUFBSSxDQUFDLE9BQU8sQ0FBQyxlQUFlLDBDQUFFLGdCQUFnQixDQUFDLGdCQUFnQixFQUFFO1FBRWpFOztXQUVHO1FBQ0gsUUFBUSxDQUFDLGdCQUFnQixDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBRS9FOzs7V0FHRztRQUNILElBQUksa0NBQXlCLENBQUMsSUFBSSxFQUFFLDJCQUEyQixFQUFFO1lBQy9ELE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztTQUN0QixDQUFDLENBQUM7UUFFSDs7O1dBR0c7UUFDSCxNQUFNLFlBQVksR0FBRyxJQUFJLFNBQVMsQ0FBQyxnQkFBZ0IsQ0FBQyxLQUFLLEVBQUUsY0FBYyxFQUFFO1lBQ3pFLFNBQVMsRUFBRSxLQUFLLENBQUMsY0FBYztTQUNoQyxDQUFDLENBQUM7UUFFSDs7V0FFRztRQUNILE1BQU0sR0FBRyxHQUFHO1lBQ1Y7Z0JBQ0UsSUFBSSxFQUFFLE9BQU87Z0JBQ2IsS0FBSyxFQUFFLEdBQUc7YUFDWDtZQUNEO2dCQUNFLDREQUE0RDtnQkFDNUQsSUFBSSxFQUFFLGdCQUFnQjtnQkFDdEIsS0FBSyxFQUFFLGNBQWM7YUFFdEI7WUFDRDtnQkFDRSxJQUFJLEVBQUUsdUJBQXVCO2dCQUM3QixLQUFLLEVBQUUsUUFBUSxDQUFDLG1CQUFtQixDQUFDLHlCQUF5QjthQUM5RDtZQUNEO2dCQUNFLElBQUksRUFBRSx3QkFBd0I7Z0JBQzlCLEtBQUssRUFBRSw2QkFBNkI7YUFDckM7U0FDRixDQUFDO1FBRUYsdUJBQXVCO1FBRXZCOztXQUVHO1FBQ0gsSUFBSSxvQkFBVSxDQUFDLEtBQUssRUFBRSxvQkFBb0IsRUFBRTtZQUMxQyxPQUFPLEVBQUUsSUFBSSxDQUFDLE9BQU87WUFDckIsWUFBWTtZQUNaLFNBQVMsRUFBRSxLQUFLO1lBQ2hCLEdBQUc7U0FDSixDQUFDLENBQUM7UUFFSCw2QkFBNkI7UUFDN0IsTUFBTSxZQUFZLEdBQUcsSUFBSSxrQkFBWSxDQUFDLEtBQUssRUFBRSxlQUFlLEVBQUU7WUFDNUQsR0FBRztZQUNILE9BQU8sRUFBRSxJQUFJLENBQUMsT0FBTztZQUNyQixVQUFVLFFBQUUsS0FBSyxDQUFDLFVBQVUsbUNBQUksQ0FBQyx5QkFBeUIsQ0FBQztZQUMzRCxZQUFZO1lBQ1osU0FBUyxFQUFFLEtBQUs7U0FDakIsQ0FBQyxDQUFDO1FBRUgsaURBQWlEO1FBQ2pELElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLGdCQUFnQixFQUFFLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDO1FBQzVFLElBQUksQ0FBQyxPQUFPLENBQUMsV0FBVyxDQUFDLGFBQWEsRUFBRSxZQUFZLENBQUMsZUFBZSxDQUFDLENBQUM7UUFFdEU7O1dBRUc7UUFDSCxJQUFJLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxjQUFjLEVBQUUsb0JBQVUsQ0FBQyxDQUFDO1FBQ3JELDBDQUEwQztRQUUxQzs7V0FFRztRQUNILDhDQUE4QztRQUM5QywwRUFBMEU7UUFDMUUsMkJBQTJCO1FBQzNCLDJCQUEyQjtRQUMzQiw0QkFBNEI7UUFDNUIsK0JBQStCO1FBQy9CLG1FQUFtRTtRQUNuRSxNQUFNO1FBRU47Ozs7V0FJRztRQUVILE1BQU0sR0FBRyxHQUFHLEtBQUssQ0FBQyx1QkFBdUIsQ0FBQyxVQUFVLENBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRTtZQUNuRSxnQkFBZ0IsRUFBRTtnQkFDaEIsV0FBVyxFQUFFLE1BQU07YUFDcEI7U0FDRixDQUFDLENBQUM7UUFHSDs7V0FFRztRQUNILElBQUksR0FBRyxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUUsUUFBUSxFQUFFLEVBQUUsS0FBSyxFQUFFLEdBQUcsQ0FBQyxtQkFBbUIsRUFBRSxDQUFDLENBQUM7SUFFeEUsQ0FBQzs7QUFsT0gsOEJBbU9DIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZWMyIGZyb20gJ0Bhd3MtY2RrL2F3cy1lYzInO1xuaW1wb3J0ICogYXMgZWNyQXNzZXRzIGZyb20gJ0Bhd3MtY2RrL2F3cy1lY3ItYXNzZXRzJztcbmltcG9ydCAqIGFzIGVrcyBmcm9tICdAYXdzLWNkay9hd3MtZWtzJztcbi8vIGltcG9ydCAqIGFzIGxvZ3MgZnJvbSAnQGF3cy1jZGsvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgZWxidjIgZnJvbSAnQGF3cy1jZGsvYXdzLWVsYXN0aWNsb2FkYmFsYW5jaW5ndjInO1xuaW1wb3J0ICogYXMgaWFtIGZyb20gJ0Bhd3MtY2RrL2F3cy1pYW0nO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCAqIGFzIHNlY3JldHNtYW5hZ2VyIGZyb20gJ0Bhd3MtY2RrL2F3cy1zZWNyZXRzbWFuYWdlcic7XG5pbXBvcnQgKiBhcyBjZGsgZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5pbXBvcnQgeyBSZHNQb3N0Z3Jlc0luc3RhbmNlIH0gZnJvbSAnLi9jb21tb24vZGF0YWJhc2UnO1xuaW1wb3J0IHsgQXBwbGljYXRpb25WcGMgfSBmcm9tICcuL2NvbW1vbi92cGMnO1xuaW1wb3J0IHsgQXdzTG9hZEJhbGFuY2VyQ29udHJvbGxlciB9IGZyb20gJy4vZWtzL2F3c2xiYyc7XG4vLyBpbXBvcnQgeyBFbGFzdGlDYWNoZUNsdXN0ZXIgfSBmcm9tICcuL2VsYXN0aWNhY2hlJztcbmltcG9ydCB7IElyc2EgfSBmcm9tICcuL2Vrcy9pcnNhJztcbmltcG9ydCB7IGFwcEluZ3Jlc3MgfSBmcm9tICcuL2Vrcy9yZXNvdXJjZXMvaW5ncmVzcyc7XG5pbXBvcnQgeyBNaWdyYXRlSm9iIH0gZnJvbSAnLi9la3MvcmVzb3VyY2VzL21pZ3JhdGUnO1xuaW1wb3J0IHsgV2ViUmVzb3VyY2VzIH0gZnJvbSAnLi9la3MvcmVzb3VyY2VzL3dlYic7XG5cblxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuZXhwb3J0IGludGVyZmFjZSBEamFuZ29Fa3NQcm9wcyB7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBidWNrZXROYW1lID8gOiBzdHJpbmc7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHZwYyA/IDogZWMyLklWcGM7XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBpbWFnZURpcmVjdG9yeTogc3RyaW5nO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgXG4gIHJlYWRvbmx5IHdlYkNvbW1hbmQgPyA6IHN0cmluZ1tdO1xuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbiAgcmVhZG9ubHkgdXNlQ2VsZXJ5QmVhdCA/IDogYm9vbGVhbjtcblxufVxuXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICBcbmV4cG9ydCBjbGFzcyBEamFuZ29Fa3MgZXh0ZW5kcyBjZGsuQ29uc3RydWN0IHtcblxuICBwdWJsaWMgc3RhdGljRmlsZUJ1Y2tldDogczMuQnVja2V0O1xuICBwdWJsaWMgdnBjOiBlYzIuSVZwYztcbiAgcHVibGljIGNsdXN0ZXI6IGVrcy5DbHVzdGVyO1xuICBwcml2YXRlIHNlY3JldDogc2VjcmV0c21hbmFnZXIuSVNlY3JldDtcblxuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogRGphbmdvRWtzUHJvcHMpIHtcbiAgICBzdXBlcihzY29wZSwgaWQpO1xuXG4gICAgLyoqXG4gICAgICogVlBDIG11c3QgaGF2ZSBwdWJsaWMsIHByaXZhdGUgYW5kIGlzb2xhdGVkIHN1Ym5ldHNcbiAgICAgKlxuICAgICAqIElmIHlvdSBkb24ndCBwcm92aWRlIGEgVlBDLCBhIG5ldyBWUEMgd2lsbCBiZSBjcmVhdGVkXG4gICAgICovXG4gICAgaWYgKCFwcm9wcy52cGMpIHtcbiAgICAgIGNvbnN0IGFwcGxpY2F0aW9uVnBjID0gbmV3IEFwcGxpY2F0aW9uVnBjKHNjb3BlLCAnQXBwVnBjJyk7XG4gICAgICB0aGlzLnZwYyA9IGFwcGxpY2F0aW9uVnBjLnZwYztcbiAgICB9IGVsc2Uge1xuICAgICAgY29uc3QgdnBjID0gcHJvcHMudnBjO1xuICAgICAgdGhpcy52cGMgPSB2cGM7XG4gICAgfVxuXG4gICAgLyoqXG4gICAgICogc3RhdGljIGZpbGVzIGJ1Y2tldCBuYW1lIGlzIGRlcml2ZWQgZnJvbSB0aGUgQ29uc3RydWN0IGlkIGlmIG5vdCBwcm92aWRlZFxuICAgICAqL1xuICAgIGNvbnN0IHN0YXRpY0ZpbGVzQnVja2V0ID0gbmV3IHMzLkJ1Y2tldChzY29wZSwgJ1N0YXRpY0J1Y2tldCcsIHtcbiAgICAgIGJ1Y2tldE5hbWU6IHByb3BzLmJ1Y2tldE5hbWUsXG4gICAgICBhdXRvRGVsZXRlT2JqZWN0czogdHJ1ZSxcbiAgICAgIHJlbW92YWxQb2xpY3k6IGNkay5SZW1vdmFsUG9saWN5LkRFU1RST1ksXG4gICAgfSk7XG4gICAgdGhpcy5zdGF0aWNGaWxlQnVja2V0ID0gc3RhdGljRmlsZXNCdWNrZXQ7XG5cbiAgICAvLyBhbGxvdyBhbGwgYWNjb3VudCB1c2VycyB0byBhc3N1bWUgdGhpcyByb2xlIGluIG9yZGVyIHRvIGFkbWluIHRoZSBjbHVzdGVyXG4gICAgY29uc3QgbWFzdGVyc1JvbGUgPSBuZXcgaWFtLlJvbGUodGhpcywgJ0FkbWluUm9sZScsIHtcbiAgICAgIGFzc3VtZWRCeTogbmV3IGlhbS5BY2NvdW50Um9vdFByaW5jaXBhbCgpLFxuICAgIH0pO1xuXG4gICAgLyoqXG4gICAgICogRUtTIGNsdXN0ZXJcbiAgICAgKi9cbiAgICB0aGlzLmNsdXN0ZXIgPSBuZXcgZWtzLkNsdXN0ZXIodGhpcywgJ015RWtzQ2x1c3RlcicsIHtcbiAgICAgIHZlcnNpb246IGVrcy5LdWJlcm5ldGVzVmVyc2lvbi5WMV8xOSxcbiAgICAgIHZwYzogdGhpcy52cGMsXG4gICAgICBtYXN0ZXJzUm9sZSxcbiAgICAgIGRlZmF1bHRDYXBhY2l0eTogMixcbiAgICB9KTtcblxuICAgIC8qKlxuICAgICAqIE5hbWVzcGFjZSBmb3IgYXBwbGljYXRpb25cbiAgICAgKi9cbiAgICBjb25zdCBhcHBOYW1lc3BhY2UgPSB0aGlzLmNsdXN0ZXIuYWRkTWFuaWZlc3QoJ2FwcC1uYW1lc3BhY2UnLCB7XG4gICAgICBhcGlWZXJzaW9uOiAndjEnLFxuICAgICAga2luZDogJ05hbWVzcGFjZScsXG4gICAgICBtZXRhZGF0YToge1xuICAgICAgICBuYW1lOiAnYXBwJyxcbiAgICAgIH0sXG4gICAgfSk7XG5cbiAgICBjb25zdCBEQl9TRUNSRVRfTkFNRSA9ICdkYlNlY3JldCc7XG4gICAgLyoqXG4gICAgICogU2VjcmV0IHVzZWQgZm9yIFJEUyBwb3N0Z3JlcyBwYXNzd29yZFxuICAgICAqL1xuICAgIHRoaXMuc2VjcmV0ID0gbmV3IHNlY3JldHNtYW5hZ2VyLlNlY3JldChzY29wZSwgJ2RiU2VjcmV0Jywge1xuICAgICAgc2VjcmV0TmFtZTogREJfU0VDUkVUX05BTUUsXG4gICAgICBkZXNjcmlwdGlvbjogJ3NlY3JldCBmb3IgcmRzJyxcbiAgICAgIGdlbmVyYXRlU2VjcmV0U3RyaW5nOiB7XG4gICAgICAgIHNlY3JldFN0cmluZ1RlbXBsYXRlOiBKU09OLnN0cmluZ2lmeSh7IHVzZXJuYW1lOiAncG9zdGdyZXMnIH0pLFxuICAgICAgICBnZW5lcmF0ZVN0cmluZ0tleTogJ3Bhc3N3b3JkJyxcbiAgICAgICAgZXhjbHVkZVB1bmN0dWF0aW9uOiB0cnVlLFxuICAgICAgICBpbmNsdWRlU3BhY2U6IGZhbHNlLFxuICAgICAgfSxcbiAgICB9KTtcblxuICAgIC8qKlxuICAgICAqIENyZWF0ZXMgYW4gSUFNIHJvbGUgd2l0aCBhIHRydXN0IHJlbGF0aW9uc2hpcCB0aGF0IGlzIHNjb3BlZCB0byB0aGUgY2x1c3RlcidzIE9JREMgcHJvdmlkZXJcbiAgICAgKiBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWtzL2xhdGVzdC91c2VyZ3VpZGUvaWFtLXJvbGVzLWZvci1zZXJ2aWNlLWFjY291bnRzLXRlY2huaWNhbC1vdmVydmlldy5odG1sXG4gICAgICpcbiAgICAgKiBUaGUgcm9sZSAocG9kUm9sZSkgd2lsbCBiZSBnaXZlbiBhY2Nlc3MgdGhlIFNlY3JldHMgTWFuYWdlciBTZWNyZXQsIHRoZSBTMyBidWNrZXQgYW5kXG4gICAgICogYW55IG90aGVyIHJlc291cmNlcyB0aGF0IGFyZSBuZWVkZWQgYnkgcG9kcyB0aGF0IHJ1biB0aGUgRGphbmdvIGFwcGxpY2F0aW9uXG4gICAgICovXG4gICAgY29uc3QgaXJzYSA9IG5ldyBJcnNhKHNjb3BlLCAnSXJzYScsIHtcbiAgICAgIGNsdXN0ZXI6IHRoaXMuY2x1c3RlcixcbiAgICB9KTtcblxuICAgIC8qKlxuICAgICAqIE1ha2Ugc3VyZSB0aGF0IHRoZSBuYW1lc3BhY2UgaGFzIGJlZW4gZGVwbG95ZWRcbiAgICAgKi9cbiAgICBpcnNhLm5vZGUuYWRkRGVwZW5kZW5jeShhcHBOYW1lc3BhY2UpO1xuXG4gICAgLyoqXG4gICAgICogR2l2ZSB0aGUgSVJTQSBwb2RSb2xlIHJlYWQgYWNjZXNzIHRvIHRoZSBidWNrZXQgYW5kIHJlYWQvd3JpdGUgYWNjZXNzIHRvIHRoZSBTMyBidWNrZXRcbiAgICAgKi9cbiAgICB0aGlzLnNlY3JldC5ncmFudFJlYWQoaXJzYS5wb2RSb2xlKTtcbiAgICB0aGlzLnN0YXRpY0ZpbGVCdWNrZXQuZ3JhbnRSZWFkV3JpdGUoaXJzYS5wb2RSb2xlKTtcblxuICAgIC8qKlxuICAgICAqIFJEUyBpbnN0YW5jZVxuICAgICAqL1xuICAgIGNvbnN0IGRhdGFiYXNlID0gbmV3IFJkc1Bvc3RncmVzSW5zdGFuY2Uoc2NvcGUsICdSZHNQb3N0Z3Jlc0luc3RhbmNlJywge1xuICAgICAgdnBjOiB0aGlzLnZwYyxcbiAgICAgIHNlY3JldDogdGhpcy5zZWNyZXQsXG4gICAgfSk7XG5cbiAgICAvKipcbiAgICAgKiBTZWN1cml0eSBHcm91cCBmb3Igd29ya2VyIG5vZGVzXG4gICAgICovXG4gICAgY29uc3QgYXBwU2VjdXJpdHlHcm91cCA9IG5ldyBlYzIuU2VjdXJpdHlHcm91cChzY29wZSwgJ2FwcFNlY3VyaXR5R3JvdXAnLCB7XG4gICAgICB2cGM6IHRoaXMudnBjLFxuICAgIH0pO1xuXG4gICAgLyoqXG4gICAgICogY2x1c3Rlci5kZWZhdWx0Q2FwYWN0aXkgaXMgdGhlIGF1dG9TY2FsaW5nR3JvdXAgKHRoZSBjbHVzdGVyJ3MgZGVmYXVsdCBub2RlIGdyb3VwKVxuICAgICAqIEhlcmUgdGhlIGFwcFNlY3VyaXR5R3JvdXAgY3JlYXRlZCBhYm92ZSBpcyBhZGRlZCB0byB0aGF0IEFTR1xuICAgICAqXG4gICAgICogVE9ETzogdXNlIGEgbm9uLWRlZmF1bHQgbm9kZS1ncm91cFxuICAgICAqL1xuICAgIHRoaXMuY2x1c3Rlci5kZWZhdWx0Q2FwYWNpdHk/LmFkZFNlY3VyaXR5R3JvdXAoYXBwU2VjdXJpdHlHcm91cCk7XG5cbiAgICAvKipcbiAgICAgKiBBbGxvdyB0aCBBU0cgdG8gYWNjZXNzcyB0aGUgZGF0YWJhc2VcbiAgICAgKi9cbiAgICBkYXRhYmFzZS5yZHNTZWN1cml0eUdyb3VwLmFkZEluZ3Jlc3NSdWxlKGFwcFNlY3VyaXR5R3JvdXAsIGVjMi5Qb3J0LnRjcCg1NDMyKSk7XG5cbiAgICAvKipcbiAgICAgKiBJbnN0YWxsYXRpb24gb2YgQVdTIExvYWQgQmFsYW5jZXIgQ29udHJvbGxlclxuICAgICAqIGh0dHBzOi8va3ViZXJuZXRlcy1zaWdzLmdpdGh1Yi5pby9hd3MtbG9hZC1iYWxhbmNlci1jb250cm9sbGVyL3YyLjIvZGVwbG95L2luc3RhbGxhdGlvbi9cbiAgICAgKi9cbiAgICBuZXcgQXdzTG9hZEJhbGFuY2VyQ29udHJvbGxlcih0aGlzLCAnQXdzTG9hZEJhbGFuY2VyQ29udHJvbGxlcicsIHtcbiAgICAgIGNsdXN0ZXI6IHRoaXMuY2x1c3RlcixcbiAgICB9KTtcblxuICAgIC8qKlxuICAgICAqIEJhY2tlbmQgZG9ja2VyIGltYWdlIHVzaW5nIERvY2tlckltYWdlQXNzZXRzXG4gICAgICogVGhpcyBpbWFnZSB3aWxsIGJlIHB1c2hlZCB0byB0aGUgRUNSIFJlZ2lzdHJ5IGNyZWF0ZWQgYnkgYGNkayBib290c3RyYXBgXG4gICAgICovXG4gICAgY29uc3QgYmFja2VuZEltYWdlID0gbmV3IGVjckFzc2V0cy5Eb2NrZXJJbWFnZUFzc2V0KHNjb3BlLCAnYmFja2VuZEltYWdlJywge1xuICAgICAgZGlyZWN0b3J5OiBwcm9wcy5pbWFnZURpcmVjdG9yeSxcbiAgICB9KTtcblxuICAgIC8qKlxuICAgICAqIENvbW1vbiBlbnZpcm9ubWVudCB2YXJpYWJsZXMgZm9yIEpvYnMgYW5kIERlcGxveW1lbnRzXG4gICAgICovXG4gICAgY29uc3QgZW52ID0gW1xuICAgICAge1xuICAgICAgICBuYW1lOiAnREVCVUcnLFxuICAgICAgICB2YWx1ZTogJzAnLFxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgLy8gdGhpcyBpcyB1c2VkIGluIHRoZSBhcHBsaWNhdGlvbiB0byBmZXRjaCB0aGUgc2VjcmV0IHZhbHVlXG4gICAgICAgIG5hbWU6ICdEQl9TRUNSRVRfTkFNRScsXG4gICAgICAgIHZhbHVlOiBEQl9TRUNSRVRfTkFNRSxcblxuICAgICAgfSxcbiAgICAgIHtcbiAgICAgICAgbmFtZTogJ1BPU1RHUkVTX1NFUlZJQ0VfSE9TVCcsXG4gICAgICAgIHZhbHVlOiBkYXRhYmFzZS5yZHNQb3N0Z3Jlc0luc3RhbmNlLmRiSW5zdGFuY2VFbmRwb2ludEFkZHJlc3MsXG4gICAgICB9LFxuICAgICAge1xuICAgICAgICBuYW1lOiAnREpBTkdPX1NFVFRJTkdTX01PRFVMRScsXG4gICAgICAgIHZhbHVlOiAnYmFja2VuZC5zZXR0aW5ncy5wcm9kdWN0aW9uJyxcbiAgICAgIH0sXG4gICAgXTtcblxuICAgIC8vIERqYW5nbyBLOHMgcmVzb3VyY2VzXG5cbiAgICAvKipcbiAgICAgKiBLdWJlcm5ldGVzIEpvYiB0aGF0IHJ1bnMgZGF0YWJhc2UgbWlncmF0aW9uc1xuICAgICAqL1xuICAgIG5ldyBNaWdyYXRlSm9iKHNjb3BlLCAnZGphbmdvLW1pZ3JhdGUtam9iJywge1xuICAgICAgY2x1c3RlcjogdGhpcy5jbHVzdGVyLFxuICAgICAgYmFja2VuZEltYWdlLFxuICAgICAgbmFtZXNwYWNlOiAnYXBwJyxcbiAgICAgIGVudixcbiAgICB9KTtcblxuICAgIC8vIHdlYiBzZXJ2aWNlIGFuZCBkZXBsb3ltZW50XG4gICAgY29uc3Qgd2ViUmVzb3VyY2VzID0gbmV3IFdlYlJlc291cmNlcyhzY29wZSwgJ3dlYi1yZXNvdXJjZXMnLCB7XG4gICAgICBlbnYsXG4gICAgICBjbHVzdGVyOiB0aGlzLmNsdXN0ZXIsXG4gICAgICB3ZWJDb21tYW5kOiBwcm9wcy53ZWJDb21tYW5kID8/IFsnLi9zY3JpcHRzL3N0YXJ0X3Byb2Quc2gnXSxcbiAgICAgIGJhY2tlbmRJbWFnZSxcbiAgICAgIG5hbWVzcGFjZTogJ2FwcCcsXG4gICAgfSk7XG5cbiAgICAvLyB3ZWJSZXNvdXJjZXMubm9kZS5hZGREZXBlbmRlbmN5KGFwcE5hbWVzcGFjZSk7XG4gICAgdGhpcy5jbHVzdGVyLmFkZE1hbmlmZXN0KCd3ZWItZGVwbG95bWVudCcsIHdlYlJlc291cmNlcy5kZXBsb3ltZW50TWFuaWZlc3QpO1xuICAgIHRoaXMuY2x1c3Rlci5hZGRNYW5pZmVzdCgnd2ViLXNlcnZpY2UnLCB3ZWJSZXNvdXJjZXMuc2VydmljZU1hbmlmZXN0KTtcblxuICAgIC8qKlxuICAgICAqIEFkZCBkZXBsb3ltZW50IGFuZCBzZXJ2aWNlIG1hbmlmZXN0cyBmb3Igd2ViIHRvIHRoZSBjbHVzdGVyXG4gICAgICovXG4gICAgdGhpcy5jbHVzdGVyLmFkZE1hbmlmZXN0KCdhcHAtaW5ncmVzc3MnLCBhcHBJbmdyZXNzKTtcbiAgICAvLyBpbmdyZXNzLm5vZGUuYWRkRGVwZW5kZW5jeSh3ZWJTZXJ2aWNlKTtcblxuICAgIC8qKlxuICAgICAqIEdldCB0aGUgQUxCIGFkZHJlc3MgdXNpbmcgS3ViZXJuZXRlc09iamVjdFZhbHVlIGFzIGEgU3RyaW5nXG4gICAgICovXG4gICAgLy8gaHR0cHM6Ly9naXRodWIuY29tL2F3cy9hd3MtY2RrL2lzc3Vlcy8xNDkzM1xuICAgIC8vIGNvbnN0IGFsYkFkZHJlc3MgPSBuZXcgZWtzLkt1YmVybmV0ZXNPYmplY3RWYWx1ZShzY29wZSwgJ0FsYkFkZHJlc3MnLCB7XG4gICAgLy8gICBjbHVzdGVyOiB0aGlzLmNsdXN0ZXIsXG4gICAgLy8gICBvYmplY3RUeXBlOiAnaW5ncmVzcycsXG4gICAgLy8gICBvYmplY3ROYW1lc3BhY2U6ICdhcHAnLFxuICAgIC8vICAgb2JqZWN0TmFtZTogJ2FwcC1pbmdyZXNzJyxcbiAgICAvLyAgIGpzb25QYXRoOiAnLml0ZW1zWzBdLnN0YXR1cy5sb2FkQmFsYW5jZXIuaW5ncmVzc1swXS5ob3N0bmFtZScsXG4gICAgLy8gfSk7XG5cbiAgICAvKipcbiAgICAgKiBSb3V0ZTUzIEEgUmVjb3JkIHBvaW50aW5nIHRvIEFMQiB0aGF0IGlzIGNyZWF0ZWQgYnkgQVdTIEFwcGxpY2F0aW9uIExvYWQgQmFsYW5jZXIgQ29udHJvbGxlclxuICAgICAqXG4gICAgICogVE9ETzogZml4IHRoaXMsIHNpbmNlIEt1YmVybmV0ZXNPYmplY3RWYWx1ZSBpcyBub3QgZ2l2aW5nIHRoZSBBTEIgUHVibGljIEROUyBVUkxcbiAgICAgKi9cblxuICAgIGNvbnN0IGFsYiA9IGVsYnYyLkFwcGxpY2F0aW9uTG9hZEJhbGFuY2VyLmZyb21Mb29rdXAodGhpcywgJ2FwcEFsYicsIHtcbiAgICAgIGxvYWRCYWxhbmNlclRhZ3M6IHtcbiAgICAgICAgRW52aXJvbm1lbnQ6ICd0ZXN0JyxcbiAgICAgIH0sXG4gICAgfSk7XG5cblxuICAgIC8qKlxuICAgICAqIE91dHB1dCB0aGUgTG9hZCBCYWxhbmNlciBVUkwgYXMgYSBDZm5PdXRwdXRcbiAgICAgKi9cbiAgICBuZXcgY2RrLkNmbk91dHB1dCh0aGlzLCAnYXBpVXJsJywgeyB2YWx1ZTogYWxiLmxvYWRCYWxhbmNlckRuc05hbWUgfSk7XG5cbiAgfVxufSJdfQ==