"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsLoadBalancerController = void 0;
const eks = require("@aws-cdk/aws-eks");
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
// eslint-disable-next-line
const request = require('sync-request');
// eslint-disable-next-line
const yaml = require('js-yaml');
class AwsLoadBalancerController extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Adopted from comments in this issue: https://github.com/aws/aws-cdk/issues/8836
        const albServiceAccount = props.cluster.addServiceAccount('aws-alb-ingress-controller-sa', {
            name: 'aws-load-balancer-controller',
            namespace: 'kube-system',
        });
        const awsAlbControllerPolicyUrl = 'https://raw.githubusercontent.com/kubernetes-sigs/aws-load-balancer-controller/v2.2.0/docs/install/iam_policy.json';
        const policyJson = request('GET', awsAlbControllerPolicyUrl).getBody('utf8');
        (JSON.parse(policyJson)).Statement.forEach(statement => {
            albServiceAccount.addToPrincipalPolicy(iam.PolicyStatement.fromJson(statement));
        });
        // AWS Load Balancer Controller Helm Chart
        // https://kubernetes-sigs.github.io/aws-load-balancer-controller/v2.2/deploy/installation/#summary
        // AWS Application Load Balancer Controller Helm Chart CRDs
        // These must be installed before installing the Helm Chart because CDK
        // installs the Helm chart with `helm upgrade` which does not
        // automatically install CRDs
        // This should be equivalent to Step 2:
        // kubectl apply -k "github.com/aws/eks-charts/stable/aws-load-balancer-controller//crds?ref=master"
        const awslbcCrdsUrl = 'https://raw.githubusercontent.com/aws/eks-charts/master/stable/aws-load-balancer-controller/crds/crds.yaml';
        const awslbcCrdsYaml = request('GET', awslbcCrdsUrl).getBody('utf8');
        const awslbcCrdsObjects = yaml.loadAll(awslbcCrdsYaml);
        new eks.KubernetesManifest(this, 'alb-custom-resource-definition', {
            cluster: props.cluster,
            manifest: awslbcCrdsObjects,
            overwrite: true,
            prune: true,
        });
        // The following is equivalent to:
        // helm repo add eks https://aws.github.io/eks-charts
        // helm install aws-load-balancer-controller eks/aws-load-balancer-controller -n kube-system --set clusterName=<cluster-name> --set serviceAccount.create=false --set serviceAccount.name=aws-load-balancer-controller
        new eks.HelmChart(scope, 'alb-ingress-controller', {
            cluster: props.cluster,
            wait: true,
            chart: 'aws-load-balancer-controller',
            repository: 'https://aws.github.io/eks-charts',
            // Note: the chart version 1.2.0 will install version 2.2.0 of the Helm Chart
            // https://github.com/aws/eks-charts/blob/master/stable/aws-load-balancer-controller/Chart.yaml
            version: '1.2.0',
            namespace: 'kube-system',
            values: {
                clusterName: props.cluster.clusterName,
                serviceAccount: {
                    create: false,
                    name: 'aws-load-balancer-controller',
                },
            },
        });
    }
}
exports.AwsLoadBalancerController = AwsLoadBalancerController;
//# sourceMappingURL=data:application/json;base64,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