"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Irsa = void 0;
const iam = require("@aws-cdk/aws-iam");
const cdk = require("@aws-cdk/core");
/**
 * Pod Role setup with IRSA (IAM Roles for Service Accounts)
 * https://docs.aws.amazon.com/eks/latest/userguide/iam-roles-for-service-accounts-technical-overview.html
 */
class Irsa extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const POD_SERVICE_ACCOUNT_NAME = 'pod-service-account';
        const oidcProviderId = props.cluster.openIdConnectProvider.openIdConnectProviderIssuer;
        /**
         * The Principal that will assume the podRole
         */
        const federatedPrincipal = new iam.FederatedPrincipal(props.cluster.openIdConnectProvider.openIdConnectProviderArn, {
            StringEquals: new cdk.CfnJson(scope, 'FederatedPrincipalCondition', {
                value: {
                    [`${oidcProviderId}:aud`]: 'sts.amazonaws.com',
                    [`${oidcProviderId}:sub`]: `system:serviceaccount:app:${POD_SERVICE_ACCOUNT_NAME}`,
                },
            }),
        }, 'sts:AssumeRoleWithWebIdentity');
        /**
         * Setup a new IAM Role that will be used by Pods to access Secrets Manager, S3, etc.
         */
        this.podRole = new iam.Role(scope, 'podRole', {
            assumedBy: federatedPrincipal,
        });
        const podServiceAccount = {
            apiVersion: 'v1',
            kind: 'ServiceAccount',
            metadata: {
                name: POD_SERVICE_ACCOUNT_NAME,
                namespace: 'app',
                annotations: {
                    'eks.amazonaws.com/role-arn': this.podRole.roleArn,
                },
            },
        };
        /**
         * Apply the service account manfiest that will be used by pods running the application
         */
        props.cluster.addManifest('pod-service-account', podServiceAccount);
    }
}
exports.Irsa = Irsa;
//# sourceMappingURL=data:application/json;base64,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