"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.BaseFargateServices = exports.LoadBalancerAccessibility = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const constructs_1 = require("constructs");
const cdknag = require("./cdknag");
const common_functions_1 = require("./common/common-functions");
/**
 * The load balancer accessibility.
 */
var LoadBalancerAccessibility;
(function (LoadBalancerAccessibility) {
    /**
     * register to external load balancer only
     */
    LoadBalancerAccessibility["EXTERNAL_ONLY"] = "EXTERNAL_ONLY";
    /**
     * register to internal load balancer only
     */
    LoadBalancerAccessibility["INTERNAL_ONLY"] = "INTERNAL_ONLY";
})(LoadBalancerAccessibility = exports.LoadBalancerAccessibility || (exports.LoadBalancerAccessibility = {}));
class BaseFargateServices extends constructs_1.Construct {
    constructor(scope, id, props) {
        var _b, _c, _d;
        super(scope, id);
        this.zoneName = '';
        this.hasExternalLoadBalancer = false;
        this.hasInternalLoadBalancer = false;
        this.vpcSubnets = { subnetType: aws_cdk_lib_1.aws_ec2.SubnetType.PRIVATE_WITH_NAT };
        /**
         * determine if vpcSubnets are all public ones
         */
        this.isPublicSubnets = false;
        this.vpc = (_b = props.vpc) !== null && _b !== void 0 ? _b : common_functions_1.getOrCreateVpc(this),
            this.service = [];
        if (props.vpcSubnets) {
            this.vpcSubnets = props.vpcSubnets;
            this.validateSubnets(this.vpc, this.vpcSubnets);
        }
        // determine whether we need the external LB
        props.tasks.forEach(t => {
            // determine the accessibility
            if (t.accessibility != LoadBalancerAccessibility.INTERNAL_ONLY) {
                this.hasExternalLoadBalancer = true;
            }
            if (t.accessibility != LoadBalancerAccessibility.EXTERNAL_ONLY) {
                this.hasInternalLoadBalancer = true;
            }
        });
        const cluster = new aws_cdk_lib_1.aws_ecs.Cluster(this, 'Cluster', {
            vpc: this.vpc,
            enableFargateCapacityProviders: true,
            containerInsights: true,
            executeCommandConfiguration: {
                logging: aws_cdk_lib_1.aws_ecs.ExecuteCommandLogging.DEFAULT,
            },
        });
        const spotOnlyStrategy = [
            {
                capacityProvider: 'FARGATE_SPOT',
                base: 0,
                weight: 1,
            },
            {
                capacityProvider: 'FARGATE',
                base: 0,
                weight: 0,
            },
        ];
        props.tasks.forEach(t => {
            var _b, _c, _d;
            const defaultContainerName = (_b = t.task.defaultContainer) === null || _b === void 0 ? void 0 : _b.containerName;
            const svc = new aws_cdk_lib_1.aws_ecs.FargateService(this, `${defaultContainerName}Service`, {
                taskDefinition: t.task,
                cluster,
                capacityProviderStrategies: (_c = t.capacityProviderStrategy) !== null && _c !== void 0 ? _c : (props.spot ? spotOnlyStrategy : undefined),
                desiredCount: t.desiredCount,
                enableExecuteCommand: (_d = props.enableExecuteCommand) !== null && _d !== void 0 ? _d : false,
                vpcSubnets: this.vpcSubnets,
                assignPublicIp: this.isPublicSubnets,
            });
            this.service.push(svc);
        });
        // Route53
        this.zoneName = (_d = (_c = props.route53Ops) === null || _c === void 0 ? void 0 : _c.zoneName) !== null && _d !== void 0 ? _d : 'svc.local';
        // ensure the dependency
        const cp = this.node.tryFindChild('Cluster');
        this.service.forEach(s => {
            s.node.addDependency(cp);
        });
        // add solution ID for the stack
        if (!aws_cdk_lib_1.Stack.of(this).templateOptions.description) {
            aws_cdk_lib_1.Stack.of(this).templateOptions.description = '(SO8030) - AWS CDK stack with serverless-container-constructs';
        }
        props.tasks.forEach(t => {
            var _b, _c;
            let cfnPolicy = (_b = t.task.executionRole) === null || _b === void 0 ? void 0 : _b.node.tryFindChild('DefaultPolicy');
            cdknag.Suppress.iamPolicy(cfnPolicy, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'ecr:GetAuthorizationToken requires wildcard resource',
                },
            ]);
            cfnPolicy = (_c = t.task.taskRole) === null || _c === void 0 ? void 0 : _c.node.tryFindChild('DefaultPolicy');
            cdknag.Suppress.iamPolicy(cfnPolicy, [
                {
                    id: 'AwsSolutions-IAM5',
                    reason: 'task role with ECS exec support requires wildcard resource for ssmmessages. see https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-exec.html',
                },
            ]);
        });
    }
    validateSubnets(vpc, vpcSubnets) {
        const subnets = vpc.selectSubnets(vpcSubnets);
        // get all subnets in the VPC
        const allsubnetIds = vpc.publicSubnets.concat(vpc.privateSubnets).concat(vpc.isolatedSubnets).map(x => x.subnetId);
        // validate the given subnets
        subnets.subnetIds.forEach(s => {
            if (!allsubnetIds.includes(s)) {
                throw new Error(`${s} does not exist in the VPC`);
            }
            if (vpc.isolatedSubnets.map(i => i.subnetId).includes(s)) {
                throw new Error(`Isolated subnet ${s} is not allowed`);
            }
        });
        const hasPublic = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
        const hasPrivate = subnets.subnetIds.some(s => new Set(vpc.privateSubnets.map(x => x.subnetId)).has(s));
        if (hasPublic && hasPrivate) {
            throw new Error('You should provide either all public or all private subnets, not both.');
        }
        this.isPublicSubnets = subnets.subnetIds.some(s => new Set(vpc.publicSubnets.map(x => x.subnetId)).has(s));
    }
}
exports.BaseFargateServices = BaseFargateServices;
_a = JSII_RTTI_SYMBOL_1;
BaseFargateServices[_a] = { fqn: "serverless-container-constructs.BaseFargateServices", version: "0.1.20" };
//# sourceMappingURL=data:application/json;base64,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