"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const events = require("@aws-cdk/aws-events");
const index_1 = require("../lib/index");
const core_1 = require("@aws-cdk/core");
const sfn = require("@aws-cdk/aws-stepfunctions");
require("@aws-cdk/assert/jest");
const cdk = require("@aws-cdk/core");
function deployNewStateMachine(stack) {
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        eventRuleProps: {
            schedule: events.Schedule.rate(core_1.Duration.minutes(5))
        }
    };
    return new index_1.EventbridgeToStepfunctions(stack, 'test-eventbridge-stepfunctions', props);
}
function deployNewStateMachineAndEventBus(stack) {
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: {}
    };
    return new index_1.EventbridgeToStepfunctions(stack, 'test-eventbridge-stepfunctions-eventbus', props);
}
test('check events rule role policy permissions', () => {
    const stack = new cdk.Stack();
    deployNewStateMachine(stack);
    expect(stack).toHaveResource("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: "states:StartExecution",
                    Effect: "Allow",
                    Resource: {
                        Ref: "testeventbridgestepfunctionsStateMachineDD09BCB6"
                    }
                }
            ],
            Version: "2012-10-17"
        }
    });
});
test('check events rule properties', () => {
    const stack = new cdk.Stack();
    deployNewStateMachine(stack);
    expect(stack).toHaveResource('AWS::Events::Rule', {
        ScheduleExpression: "rate(5 minutes)",
        State: "ENABLED",
        Targets: [
            {
                Arn: {
                    Ref: "testeventbridgestepfunctionsStateMachineDD09BCB6"
                },
                Id: "Target0",
                RoleArn: {
                    "Fn::GetAtt": [
                        "testeventbridgestepfunctionsEventsRuleRoleFFAAD2A8",
                        "Arn"
                    ]
                }
            }
        ]
    });
});
test('check properties', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStateMachine(stack);
    expect(construct.cloudwatchAlarms !== null);
    expect(construct.stateMachine !== null);
    expect(construct.eventsRule !== null);
    expect(construct.stateMachineLogGroup !== null);
});
test('check properties with no CW Alarms', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        eventRuleProps: {
            schedule: events.Schedule.rate(core_1.Duration.minutes(5))
        },
        createCloudWatchAlarms: false
    };
    const construct = new index_1.EventbridgeToStepfunctions(stack, 'test-eventbridge-stepfunctions', props);
    expect(construct.cloudwatchAlarms === null);
    expect(construct.stateMachine !== null);
    expect(construct.eventsRule !== null);
    expect(construct.stateMachineLogGroup !== null);
});
test('check eventbus property, snapshot & eventbus exists', () => {
    const stack = new cdk.Stack();
    const construct = deployNewStateMachineAndEventBus(stack);
    expect(construct.cloudwatchAlarms !== null);
    expect(construct.stateMachine !== null);
    expect(construct.eventsRule !== null);
    expect(construct.stateMachineLogGroup !== null);
    expect(construct.eventBus !== null);
    // Check whether eventbus exists
    expect(stack).toHaveResource('AWS::Events::EventBus');
});
test('check exception while passing existingEventBus & eventBusProps', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        },
        eventBusProps: {},
        existingEventBusInterface: new events.EventBus(stack, `test-existing-new-eventbus`, {})
    };
    const app = () => {
        new index_1.EventbridgeToStepfunctions(stack, 'test-eventbridge-stepfunctions', props);
    };
    expect(app).toThrowError();
});
test('check custom event bus resource with props when deploy:true', () => {
    const stack = new cdk.Stack();
    const startState = new sfn.Pass(stack, 'StartState');
    const props = {
        stateMachineProps: {
            definition: startState
        },
        eventBusProps: {
            eventBusName: 'testcustomeventbus'
        },
        eventRuleProps: {
            eventPattern: {
                source: ['solutionsconstructs']
            }
        }
    };
    new index_1.EventbridgeToStepfunctions(stack, 'test-new-eventbridge-stepfunctions', props);
    expect(stack).toHaveResource('AWS::Events::EventBus', {
        Name: 'testcustomeventbus'
    });
});
//# sourceMappingURL=data:application/json;base64,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