"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const _lambda_architecture_1 = require("../_lambda-architecture");
const catalog_builder_1 = require("../catalog-builder");
const constants_1 = require("../shared/constants");
const transliterator_1 = require("../transliterator");
const needs_catalog_update_1 = require("./needs-catalog-update");
const redrive_state_machine_1 = require("./redrive-state-machine");
/**
 * This retry policy is used for all items in the state machine and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * After 30 attempts, given the parameters, the last attempt will wait just
 * under 16 minutes, which should be enough for currently running Lambda
 * functions to complete (or time out after 15 minutes). The total time spent
 * waiting between retries by this time is just over 3 hours. This is a lot of
 * time, but in extreme burst situations (i.e: reprocessing everything), this
 * is actually a good thing.
 */
const THROTTLE_RETRY_POLICY = { backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 };
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            label: 'Dead-Letter Queue not empty',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).next(new aws_stepfunctions_1.Succeed(this, 'Sent to DLQ'));
        const ignore = new aws_stepfunctions_1.Pass(this, 'Ignore');
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props);
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder.function,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(sendToDeadLetterQueue, { errors: ['Lambda.TooManyRequestsException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' });
        const needsCatalogUpdateFunction = new needs_catalog_update_1.NeedsCatalogUpdate(this, 'NeedsCatalogUpdate', {
            architecture: _lambda_architecture_1.gravitonLambdaIfAvailable(this),
            description: '[ConstructHub/Orchestration/NeedsCatalogUpdate] Determines whether a package version requires a catalog update',
            environment: { CATALOG_BUCKET_NAME: props.bucket.bucketName, CATALOG_OBJECT_KEY: constants_1.CATALOG_KEY },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(1),
        });
        props.bucket.grantRead(needsCatalogUpdateFunction);
        // Check whether the catalog needs updating. If so, trigger addToCatalog.
        const addToCatalogIfNeeded = new tasks.LambdaInvoke(this, 'Check whether catalog needs udpating', {
            lambdaFunction: needsCatalogUpdateFunction,
            payloadResponseOnly: true,
            resultPath: '$.catalogNeedsUpdating',
        })
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(sendToDeadLetterQueue, { errors: ['Lambda.TooManyRequestsException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' })
            .next(new aws_stepfunctions_1.Choice(this, 'Is catalog update needed?')
            .when(aws_stepfunctions_1.Condition.booleanEquals('$.catalogNeedsUpdating', true), addToCatalog)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.ecsCluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            containerInsights: true,
            enableFargateCapacityProviders: true,
            vpc: props.vpc,
        });
        this.transliterator = new transliterator_1.Transliterator(this, 'Transliterator', props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        })
            .next(new aws_stepfunctions_1.Pass(this, 'Prepare doc-gen ECS Command', {
            parameters: { 'command.$': 'States.Array(States.JsonToString($))' },
            resultPath: '$.docGen',
        }))
            .next(this.transliterator.createEcsRunTask(this, 'Generate docs', {
            cluster: this.ecsCluster,
            inputPath: '$.docGen.command',
            resultPath: '$.docGenOutput',
            // Expect this to complete within one hour
            timeout: core_1.Duration.hours(1),
            vpcSubnets: props.vpcSubnets,
            securityGroups: props.vpcSecurityGroups,
        })
            // Do not retry NoSpaceLeftOnDevice errors, these are typically not transient.
            .addRetry({ errors: ['jsii-docgen.NoSpaceLeftOnDevice'], maxAttempts: 0 })
            .addRetry({
            errors: [
                'ECS.AmazonECSException',
                'ECS.InvalidParameterException',
                'jsii-docgen.NpmError.E429',
                'jsii-codgen.NpmError.EPROTO',
            ],
            ...THROTTLE_RETRY_POLICY,
        })
            .addRetry({
            errors: ['jsii-docgen.NpmError.ETARGET'],
            // We'll wait longer between retries. This is to account for CodeArtifact's lag behind npm
            backoffRate: 2,
            interval: core_1.Duration.minutes(5),
            maxAttempts: 3,
        })
            .addRetry({ maxAttempts: 3 })
            .addCatch(ignore, { errors: [constants_1.UNPROCESSABLE_PACKAGE_ERROR_NAME] })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.Timeout'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['ECS.AmazonECSException', 'ECS.InvalidParameterException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' })
            .next(addToCatalogIfNeeded));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // The workflow is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the orchestrator.
        this.regenerateAllDocumentation = new RegenerateAllDocumentation(this, 'RegenerateAllDocumentation', {
            bucket: props.bucket,
            stateMachine: this.stateMachine,
        }).stateMachine;
    }
    metricEcsTaskCount(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'TaskCount',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % CPU utilization from the CPU units utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mCpuUtilized, 0) / FILL(mCpuReserved, REPEAT)',
            usingMetrics: {
                mCpuReserved: this.metricEcsCpuReserved(),
                mCpuUtilized: this.metricEcsCpuUtilized(),
            },
        });
    }
    metricEcsMemoryReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % memory utilization from the RAM utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mMemoryUtilized, 0) / FILL(mMemoryReserved, REPEAT)',
            usingMetrics: {
                mMemoryReserved: this.metricEcsMemoryReserved(),
                mMemoryUtilized: this.metricEcsMemoryUtilized(),
            },
        });
    }
    metricEcsNetworkRxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkRxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsNetworkTxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkTxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
}
exports.Orchestration = Orchestration;
class RegenerateAllDocumentation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const processVersions = new aws_stepfunctions_1.Choice(this, 'Get package versions page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each key prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Start Orchestration Workflow', {
            stateMachine: props.stateMachine,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                bucket: props.bucket.bucketName,
                assembly: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.ASSEMBLY_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                metadata: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.METADATA_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                package: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.PACKAGE_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })));
        processVersions.next(new aws_stepfunctions_1.Choice(this, 'Has more versions?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processVersions)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success')));
        const processPackageVersions = new aws_stepfunctions_1.StateMachine(this, 'PerPackage', {
            definition: processVersions,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This workflow is broken into two sub-workflows because otherwise it hits the 25K events limit
        // of StepFunction executions relatively quickly.
        const processNamespace = new aws_stepfunctions_1.Choice(this, 'Get @scope page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each @scope/pkg', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Process scoped package', {
            stateMachine: processPackageVersions,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })));
        processNamespace.next(new aws_stepfunctions_1.Choice(this, 'Has more packages?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processNamespace)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done')));
        const start = new aws_stepfunctions_1.Choice(this, 'Get prefix page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next prefixes page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First prefix page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] })).afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new aws_stepfunctions_1.Choice(this, 'Is this a @scope/ prefix?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Prefix', `${constants_1.STORAGE_KEY_PREFIX}@*`), processNamespace)
            .otherwise(new tasks.StepFunctionsStartExecution(this, 'Process unscoped package', {
            stateMachine: processPackageVersions,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] }))
            .afterwards()));
        start.next(new aws_stepfunctions_1.Choice(this, 'Has more prefixes?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), start)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition: start,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.hours(4),
            tracingEnabled: true,
        });
        props.bucket.grantRead(processPackageVersions);
        props.bucket.grantRead(this.stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,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