"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
async function handler(event, _context) {
    var _a;
    console.log('Event:', JSON.stringify(event, null, 2));
    const indexedPackages = new Map();
    const packageNames = new Set();
    const packageMajorVersions = new Set();
    const perLanguage = new Map();
    /**
     * Records the status of a particular package, package major version, package
     * version, and package version submodule in the per-language state storage.
     * Whenever a new entry is added, a `MISSING` entry is automatically inserted
     * for the other languages (unless another entry already exists).
     *
     * If a submodule is provided, only that submodule's availability is updated.
     */
    function recordPerLanguage(language, status, pkgName, pkgMajor, pkgVersion, submodule) {
        for (const lang of language_1.DocumentationLanguage.ALL) {
            doRecordPerLanguage(perLanguage, lang, 
            // If the language is NOT the registered one, then we insert "MISSING".
            lang === language ? status : "Missing" /* MISSING */, pkgName, pkgMajor, pkgVersion, submodule);
        }
    }
    const bucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    for await (const key of relevantObjectKeys(bucket)) {
        const [, name, version] = constants.STORAGE_KEY_FORMAT_REGEX.exec(key);
        packageNames.add(name);
        const majorVersion = `${name}@${new semver_1.SemVer(version).major}`;
        packageMajorVersions.add(majorVersion);
        const fullName = `${name}@${version}`;
        // Ensure the package is fully registered for per-language status, even if no doc exists yet.
        for (const language of language_1.DocumentationLanguage.ALL) {
            recordPerLanguage(language, "Missing" /* MISSING */, name, majorVersion, fullName);
        }
        if (!indexedPackages.has(fullName)) {
            indexedPackages.set(fullName, {});
        }
        const status = indexedPackages.get(fullName);
        if (key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            status.metadataPresent = true;
        }
        else if (key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            status.tarballPresent = true;
        }
        else if (key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            status.assemblyPresent = true;
        }
        else {
            let identified = false;
            for (const language of language_1.DocumentationLanguage.ALL) {
                const match = submoduleKeyRegexp(language).exec(key);
                if (match != null) {
                    const [, submodule, isUnsupported] = match;
                    if (status.submodules == null) {
                        status.submodules = new Set();
                    }
                    status.submodules.add(`${fullName}.${submodule}`);
                    recordPerLanguage(language, isUnsupported ? "Unsupported" /* UNSUPPORTED */ : "Supported" /* SUPPORTED */, name, majorVersion, fullName, submodule);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language))) {
                    recordPerLanguage(language, "Supported" /* SUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language) + constants.NOT_SUPPORTED_SUFFIX)) {
                    recordPerLanguage(language, "Supported" /* SUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
            }
            if (!identified) {
                status.unknownObjects = (_a = status.unknownObjects) !== null && _a !== void 0 ? _a : [];
                status.unknownObjects.push(key);
            }
        }
    }
    await aws_embedded_metrics_1.metricScope((metrics) => () => {
        var _a, _b, _c;
        // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
        metrics.setDimensions();
        const missingMetadata = new Array();
        const missingAssembly = new Array();
        const missingTarball = new Array();
        const unknownObjects = new Array();
        const submodules = new Array();
        for (const [name, status] of indexedPackages.entries()) {
            if (!status.metadataPresent) {
                missingMetadata.push(name);
            }
            if (!status.assemblyPresent) {
                missingAssembly.push(name);
            }
            if (!status.tarballPresent) {
                missingTarball.push(name);
            }
            if ((_b = (_a = status.unknownObjects) === null || _a === void 0 ? void 0 : _a.length) !== null && _b !== void 0 ? _b : 0 > 0) {
                unknownObjects.push(...status.unknownObjects);
            }
            for (const submodule of (_c = status.submodules) !== null && _c !== void 0 ? _c : []) {
                submodules.push(submodule);
            }
        }
        metrics.setProperty('detail', { missingMetadata, missingAssembly, missingTarball, unknownObjects });
        metrics.putMetric("MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */, missingMetadata.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */, missingAssembly.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */, missingTarball.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageCount" /* PACKAGE_COUNT */, packageNames.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */, packageMajorVersions.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, indexedPackages.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("SubmoduleCount" /* SUBMODULE_COUNT */, submodules.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */, unknownObjects.length, aws_embedded_metrics_1.Unit.Count);
    })();
    for (const entry of perLanguage.entries()) {
        await aws_embedded_metrics_1.metricScope((metrics) => ([language, data]) => {
            metrics.setDimensions({ [constants_1.LANGUAGE_DIMENSION]: language.toString() });
            for (const forStatus of ["Supported" /* SUPPORTED */, "Unsupported" /* UNSUPPORTED */, "Missing" /* MISSING */]) {
                for (const [key, statuses] of Object.entries(data)) {
                    let filtered = Array.from(statuses.entries()).filter(([, status]) => forStatus === status);
                    let metricName = METRIC_NAME_BY_STATUS_AND_GRAIN[forStatus][key];
                    // List out selected packages for posterity (and troubleshooting)
                    console.log(`${forStatus} ${key} for ${language}: ${filtered.map(([name]) => name).join(', ')}`);
                    metrics.putMetric(metricName, filtered.length, aws_embedded_metrics_1.Unit.Count);
                }
            }
        })(entry);
    }
}
exports.handler = handler;
async function* relevantObjectKeys(bucket) {
    var _a;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        for (const { Key } of (_a = response.Contents) !== null && _a !== void 0 ? _a : []) {
            if (Key == null) {
                continue;
            }
            yield Key;
        }
        request.ContinuationToken = response.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
/**
 * This function obtains a regular expression with a capture group that allows
 * determining the submodule name from a submodule documentation key, and
 * another to determine whether the object is an "unsupported beacon" or not.
 */
function submoduleKeyRegexp(language) {
    // We use a placeholder to be able to insert the capture group once we have
    // fully quoted the key prefix for Regex safety.
    const placeholder = '<SUBMODULENAME>';
    // We obtain the standard key prefix.
    const keyPrefix = constants.docsKeySuffix(language, placeholder);
    // Finally, assemble the regular expression with the capture group.
    return new RegExp(`.*${reQuote(keyPrefix).replace(placeholder, '(.+)')}(${reQuote(constants.NOT_SUPPORTED_SUFFIX)})?$`);
    /**
     * Escapes all "speacial meaning" characters in a string, so it can be used as
     * part of a regular expression.
     */
    function reQuote(str) {
        return str.replace(/([+*.()?$[\]])/g, '\\$1');
    }
}
const METRIC_NAME_BY_STATUS_AND_GRAIN = {
    ["Missing" /* MISSING */]: {
        ["packages" /* PACKAGES */]: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
    },
    ["Unsupported" /* UNSUPPORTED */]: {
        ["packages" /* PACKAGES */]: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
    },
    ["Supported" /* SUPPORTED */]: {
        ["packages" /* PACKAGES */]: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
    },
};
/**
 * Registers the information for the provided language. A "MISSING" status
 * will be ignored if another status was already registered for the same
 * entity. An "UNSUPPORTED" status will be ignored if a "SUPPORTED" status
 * was already registered for the same entity.
 *
 * If a submodule is provided, only that submodule's availability is updated.
 */
function doRecordPerLanguage(perLanguage, language, status, pkgName, pkgMajor, pkgVersion, submodule) {
    if (!perLanguage.has(language)) {
        perLanguage.set(language, {
            ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: new Map(),
            ["packages" /* PACKAGES */]: new Map(),
            ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: new Map(),
            ["package versions" /* PACKAGE_VERSIONS */]: new Map(),
        });
    }
    const data = perLanguage.get(language);
    // If there is a submodule, only update the submodule domain.
    const outputDomains = submodule
        ? [
            [data["package version submodules" /* PACKAGE_VERSION_SUBMODULES */], `${pkgVersion}.${submodule}`],
        ]
        : [
            [data["package major versions" /* PACKAGE_MAJOR_VERSIONS */], pkgMajor],
            [data["package versions" /* PACKAGE_VERSIONS */], pkgVersion],
            [data["packages" /* PACKAGES */], pkgName],
        ];
    for (const [map, name] of outputDomains) {
        switch (status) {
            case "Missing" /* MISSING */:
                // If we already have a status, don't override it with "MISSING".
                if (!map.has(name)) {
                    map.set(name, status);
                }
                break;
            case "Supported" /* SUPPORTED */:
                // If thr package is "supported", this always "wins"
                map.set(name, status);
                break;
            case "Unsupported" /* UNSUPPORTED */:
                // If we already have a status, only override with "UNSUPPORTED" if it was "MISSING".
                if (!map.has(name) || map.get(name) === "Missing" /* MISSING */) {
                    map.set(name, status);
                }
                break;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY2FuYXJ5LmxhbWJkYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9iYWNrZW5kL2ludmVudG9yeS9jYW5hcnkubGFtYmRhLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLCtEQUF3RTtBQUV4RSxtQ0FBZ0M7QUFDaEMsbURBQW1EO0FBQ25ELGlEQUFpRDtBQUNqRCxtRUFBeUQ7QUFDekQsaURBQTJEO0FBQzNELDJDQUFnRjtBQUVoRixvQ0FBYSxDQUFDLFNBQVMsR0FBRyw2QkFBaUIsQ0FBQztBQUVyQyxLQUFLLFVBQVUsT0FBTyxDQUFDLEtBQXFCLEVBQUUsUUFBaUI7O0lBQ3BFLE9BQU8sQ0FBQyxHQUFHLENBQUMsUUFBUSxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsS0FBSyxFQUFFLElBQUksRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBRXRELE1BQU0sZUFBZSxHQUFHLElBQUksR0FBRyxFQUFnQyxDQUFDO0lBQ2hFLE1BQU0sWUFBWSxHQUFHLElBQUksR0FBRyxFQUFVLENBQUM7SUFDdkMsTUFBTSxvQkFBb0IsR0FBRyxJQUFJLEdBQUcsRUFBVSxDQUFDO0lBQy9DLE1BQU0sV0FBVyxHQUFHLElBQUksR0FBRyxFQUEwQyxDQUFDO0lBRXRFOzs7Ozs7O09BT0c7SUFDSCxTQUFTLGlCQUFpQixDQUN4QixRQUErQixFQUMvQixNQUF5QixFQUN6QixPQUFlLEVBQ2YsUUFBZ0IsRUFDaEIsVUFBa0IsRUFDbEIsU0FBa0I7UUFFbEIsS0FBSyxNQUFNLElBQUksSUFBSSxnQ0FBcUIsQ0FBQyxHQUFHLEVBQUU7WUFDNUMsbUJBQW1CLENBQ2pCLFdBQVcsRUFDWCxJQUFJO1lBQ0osdUVBQXVFO1lBQ3ZFLElBQUksS0FBSyxRQUFRLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLHdCQUEwQixFQUN0RCxPQUFPLEVBQ1AsUUFBUSxFQUNSLFVBQVUsRUFDVixTQUFTLENBQ1YsQ0FBQztTQUNIO0lBQ0gsQ0FBQztJQUVELE1BQU0sTUFBTSxHQUFHLDhCQUFVLENBQUMsYUFBYSxDQUFDLENBQUM7SUFDekMsSUFBSSxLQUFLLEVBQUUsTUFBTSxHQUFHLElBQUksa0JBQWtCLENBQUMsTUFBTSxDQUFDLEVBQUU7UUFDbEQsTUFBTSxDQUFDLEVBQUUsSUFBSSxFQUFFLE9BQU8sQ0FBQyxHQUFHLFNBQVMsQ0FBQyx3QkFBd0IsQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFFLENBQUM7UUFFeEUsWUFBWSxDQUFDLEdBQUcsQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUN2QixNQUFNLFlBQVksR0FBRyxHQUFHLElBQUksSUFBSSxJQUFJLGVBQU0sQ0FBQyxPQUFPLENBQUMsQ0FBQyxLQUFLLEVBQUUsQ0FBQztRQUM1RCxvQkFBb0IsQ0FBQyxHQUFHLENBQUMsWUFBWSxDQUFDLENBQUM7UUFFdkMsTUFBTSxRQUFRLEdBQUcsR0FBRyxJQUFJLElBQUksT0FBTyxFQUFFLENBQUM7UUFFdEMsNkZBQTZGO1FBQzdGLEtBQUssTUFBTSxRQUFRLElBQUksZ0NBQXFCLENBQUMsR0FBRyxFQUFFO1lBQ2hELGlCQUFpQixDQUFDLFFBQVEsMkJBQTZCLElBQUksRUFBRSxZQUFZLEVBQUUsUUFBUSxDQUFDLENBQUM7U0FDdEY7UUFFRCxJQUFJLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUNsQyxlQUFlLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRSxFQUFFLENBQUMsQ0FBQztTQUNuQztRQUNELE1BQU0sTUFBTSxHQUFHLGVBQWUsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFFLENBQUM7UUFFOUMsSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFO1lBQy9DLE1BQU0sQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDO1NBQy9CO2FBQU0sSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFO1lBQ3JELE1BQU0sQ0FBQyxjQUFjLEdBQUcsSUFBSSxDQUFDO1NBQzlCO2FBQU0sSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxFQUFFO1lBQ3RELE1BQU0sQ0FBQyxlQUFlLEdBQUcsSUFBSSxDQUFDO1NBQy9CO2FBQU07WUFDTCxJQUFJLFVBQVUsR0FBRyxLQUFLLENBQUM7WUFDdkIsS0FBSyxNQUFNLFFBQVEsSUFBSSxnQ0FBcUIsQ0FBQyxHQUFHLEVBQUU7Z0JBQ2hELE1BQU0sS0FBSyxHQUFHLGtCQUFrQixDQUFDLFFBQVEsQ0FBQyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQztnQkFDckQsSUFBSSxLQUFLLElBQUksSUFBSSxFQUFFO29CQUNqQixNQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsYUFBYSxDQUFDLEdBQUcsS0FBSyxDQUFDO29CQUMzQyxJQUFJLE1BQU0sQ0FBQyxVQUFVLElBQUksSUFBSSxFQUFFO3dCQUM3QixNQUFNLENBQUMsVUFBVSxHQUFHLElBQUksR0FBRyxFQUFFLENBQUM7cUJBQy9CO29CQUNELE1BQU0sQ0FBQyxVQUFVLENBQUMsR0FBRyxDQUFDLEdBQUcsUUFBUSxJQUFJLFNBQVMsRUFBRSxDQUFDLENBQUM7b0JBQ2xELGlCQUFpQixDQUNmLFFBQVEsRUFDUixhQUFhLENBQUMsQ0FBQyxpQ0FBK0IsQ0FBQyw0QkFBNEIsRUFDM0UsSUFBSSxFQUNKLFlBQVksRUFDWixRQUFRLEVBQ1IsU0FBUyxDQUNWLENBQUM7b0JBQ0YsVUFBVSxHQUFHLElBQUksQ0FBQztpQkFDbkI7cUJBQU0sSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLENBQUMsRUFBRTtvQkFDMUQsaUJBQWlCLENBQUMsUUFBUSwrQkFBK0IsSUFBSSxFQUFFLFlBQVksRUFBRSxRQUFRLENBQUMsQ0FBQztvQkFDdkYsVUFBVSxHQUFHLElBQUksQ0FBQztpQkFDbkI7cUJBQU0sSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLFNBQVMsQ0FBQyxhQUFhLENBQUMsUUFBUSxDQUFDLEdBQUcsU0FBUyxDQUFDLG9CQUFvQixDQUFDLEVBQUU7b0JBQzNGLGlCQUFpQixDQUFDLFFBQVEsK0JBQStCLElBQUksRUFBRSxZQUFZLEVBQUUsUUFBUSxDQUFDLENBQUM7b0JBQ3ZGLFVBQVUsR0FBRyxJQUFJLENBQUM7aUJBQ25CO2FBQ0Y7WUFDRCxJQUFJLENBQUMsVUFBVSxFQUFFO2dCQUNmLE1BQU0sQ0FBQyxjQUFjLFNBQUcsTUFBTSxDQUFDLGNBQWMsbUNBQUksRUFBRSxDQUFDO2dCQUNwRCxNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUNqQztTQUNGO0tBQ0Y7SUFFRCxNQUFNLGtDQUFXLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLEdBQUcsRUFBRTs7UUFDbEMsMkhBQTJIO1FBQzNILE9BQU8sQ0FBQyxhQUFhLEVBQUUsQ0FBQztRQUV4QixNQUFNLGVBQWUsR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO1FBQzVDLE1BQU0sZUFBZSxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7UUFDNUMsTUFBTSxjQUFjLEdBQUcsSUFBSSxLQUFLLEVBQVUsQ0FBQztRQUMzQyxNQUFNLGNBQWMsR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO1FBQzNDLE1BQU0sVUFBVSxHQUFHLElBQUksS0FBSyxFQUFVLENBQUM7UUFDdkMsS0FBSyxNQUFNLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLGVBQWUsQ0FBQyxPQUFPLEVBQUUsRUFBRTtZQUN0RCxJQUFJLENBQUMsTUFBTSxDQUFDLGVBQWUsRUFBRTtnQkFDM0IsZUFBZSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQzthQUM1QjtZQUNELElBQUksQ0FBQyxNQUFNLENBQUMsZUFBZSxFQUFFO2dCQUMzQixlQUFlLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxDQUFDO2FBQzVCO1lBQ0QsSUFBSSxDQUFDLE1BQU0sQ0FBQyxjQUFjLEVBQUU7Z0JBQzFCLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUM7YUFDM0I7WUFDRCxnQkFBSSxNQUFNLENBQUMsY0FBYywwQ0FBRSxNQUFNLG1DQUFJLENBQUMsR0FBRyxDQUFDLEVBQUU7Z0JBQzFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsR0FBRyxNQUFNLENBQUMsY0FBZSxDQUFDLENBQUM7YUFDaEQ7WUFFRCxLQUFLLE1BQU0sU0FBUyxVQUFJLE1BQU0sQ0FBQyxVQUFVLG1DQUFJLEVBQUUsRUFBRTtnQkFDL0MsVUFBVSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsQ0FBQzthQUM1QjtTQUNGO1FBRUQsT0FBTyxDQUFDLFdBQVcsQ0FBQyxRQUFRLEVBQUUsRUFBRSxlQUFlLEVBQUUsZUFBZSxFQUFFLGNBQWMsRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDO1FBRXBHLE9BQU8sQ0FBQyxTQUFTLDZEQUFvQyxlQUFlLENBQUMsTUFBTSxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDekYsT0FBTyxDQUFDLFNBQVMsc0RBQW9DLGVBQWUsQ0FBQyxNQUFNLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUN6RixPQUFPLENBQUMsU0FBUywyREFBbUMsY0FBYyxDQUFDLE1BQU0sRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3ZGLE9BQU8sQ0FBQyxTQUFTLHFDQUEyQixZQUFZLENBQUMsSUFBSSxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDM0UsT0FBTyxDQUFDLFNBQVMsdURBQWlDLG9CQUFvQixDQUFDLElBQUksRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO1FBQ3pGLE9BQU8sQ0FBQyxTQUFTLG9EQUFtQyxlQUFlLENBQUMsSUFBSSxFQUFFLDJCQUFJLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDdEYsT0FBTyxDQUFDLFNBQVMseUNBQTZCLFVBQVUsQ0FBQyxNQUFNLEVBQUUsMkJBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUM3RSxPQUFPLENBQUMsU0FBUyxrREFBa0MsY0FBYyxDQUFDLE1BQU0sRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO0lBQ3hGLENBQUMsQ0FBQyxFQUFFLENBQUM7SUFFTCxLQUFLLE1BQU0sS0FBSyxJQUFJLFdBQVcsQ0FBQyxPQUFPLEVBQUUsRUFBRTtRQUN6QyxNQUFNLGtDQUFXLENBQUMsQ0FBQyxPQUFPLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUEyQyxFQUFFLEVBQUU7WUFDNUYsT0FBTyxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsOEJBQWtCLENBQUMsRUFBRSxRQUFRLENBQUMsUUFBUSxFQUFFLEVBQUUsQ0FBQyxDQUFDO1lBRXJFLEtBQUssTUFBTSxTQUFTLElBQUksdUZBQXVGLEVBQUU7Z0JBQy9HLEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxRQUFRLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUNsRCxJQUFJLFFBQVEsR0FBRyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxNQUFNLENBQUMsRUFBRSxFQUFFLENBQUMsU0FBUyxLQUFLLE1BQU0sQ0FBQyxDQUFDO29CQUMzRixJQUFJLFVBQVUsR0FBRywrQkFBK0IsQ0FBQyxTQUE4QixDQUFDLENBQUMsR0FBNEIsQ0FBQyxDQUFDO29CQUMvRyxpRUFBaUU7b0JBQ2pFLE9BQU8sQ0FBQyxHQUFHLENBQUMsR0FBRyxTQUFTLElBQUksR0FBRyxRQUFRLFFBQVEsS0FBSyxRQUFRLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsRUFBRSxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUMsSUFBSSxDQUFDLElBQUksQ0FBQyxFQUFFLENBQUMsQ0FBQztvQkFDakcsT0FBTyxDQUFDLFNBQVMsQ0FBQyxVQUFVLEVBQUUsUUFBUSxDQUFDLE1BQU0sRUFBRSwyQkFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDO2lCQUM1RDthQUNGO1FBQ0gsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLENBQUM7S0FDWDtBQUNILENBQUM7QUF6SkQsMEJBeUpDO0FBRUQsS0FBSyxTQUFTLENBQUMsQ0FBQyxrQkFBa0IsQ0FBQyxNQUFjOztJQUMvQyxNQUFNLE9BQU8sR0FBZ0M7UUFDM0MsTUFBTSxFQUFFLE1BQU07UUFDZCxNQUFNLEVBQUUsU0FBUyxDQUFDLGtCQUFrQjtLQUNyQyxDQUFDO0lBQ0YsR0FBRztRQUNELE1BQU0sUUFBUSxHQUFHLE1BQU0sR0FBRyxDQUFDLEVBQUUsRUFBRSxDQUFDLGFBQWEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLEVBQUUsQ0FBQztRQUNqRSxLQUFLLE1BQU0sRUFBRSxHQUFHLEVBQUUsVUFBSSxRQUFRLENBQUMsUUFBUSxtQ0FBSSxFQUFFLEVBQUU7WUFDN0MsSUFBSSxHQUFHLElBQUksSUFBSSxFQUFFO2dCQUFFLFNBQVM7YUFBRTtZQUM5QixNQUFNLEdBQUcsQ0FBQztTQUNYO1FBQ0QsT0FBTyxDQUFDLGlCQUFpQixHQUFHLFFBQVEsQ0FBQyxxQkFBcUIsQ0FBQztLQUM1RCxRQUFRLE9BQU8sQ0FBQyxpQkFBaUIsSUFBSSxJQUFJLEVBQUU7QUFDOUMsQ0FBQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFTLGtCQUFrQixDQUFDLFFBQStCO0lBQ3pELDJFQUEyRTtJQUMzRSxnREFBZ0Q7SUFDaEQsTUFBTSxXQUFXLEdBQUcsaUJBQWlCLENBQUM7SUFFdEMscUNBQXFDO0lBQ3JDLE1BQU0sU0FBUyxHQUFHLFNBQVMsQ0FBQyxhQUFhLENBQUMsUUFBUSxFQUFFLFdBQVcsQ0FBQyxDQUFDO0lBRWpFLG1FQUFtRTtJQUNuRSxPQUFPLElBQUksTUFBTSxDQUFDLEtBQUssT0FBTyxDQUFDLFNBQVMsQ0FBQyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEVBQUUsTUFBTSxDQUFDLElBQUksT0FBTyxDQUFDLFNBQVMsQ0FBQyxvQkFBb0IsQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUV4SDs7O09BR0c7SUFDSCxTQUFTLE9BQU8sQ0FBQyxHQUFXO1FBQzFCLE9BQU8sR0FBRyxDQUFDLE9BQU8sQ0FBQyxpQkFBaUIsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUNoRCxDQUFDO0FBQ0gsQ0FBQztBQXlCRCxNQUFNLCtCQUErQixHQUEwRjtJQUM3SCx5QkFBMkIsRUFBRTtRQUMzQiwyQkFBZ0IsMkRBQTBDO1FBQzFELHVEQUE4Qiw4REFBd0M7UUFDdEUsMkNBQXdCLGtFQUEwQztRQUNsRSwrREFBa0MsK0RBQTRDO0tBQy9FO0lBQ0QsaUNBQStCLEVBQUU7UUFDL0IsMkJBQWdCLG1FQUE4QztRQUM5RCx1REFBOEIsc0VBQTRDO1FBQzFFLDJDQUF3QiwwRUFBOEM7UUFDdEUsK0RBQWtDLHVFQUFnRDtLQUNuRjtJQUNELDZCQUE2QixFQUFFO1FBQzdCLDJCQUFnQiwrREFBNEM7UUFDNUQsdURBQThCLGtFQUEwQztRQUN4RSwyQ0FBd0Isc0VBQTRDO1FBQ3BFLCtEQUFrQyxtRUFBOEM7S0FDakY7Q0FDRixDQUFDO0FBR0Y7Ozs7Ozs7R0FPRztBQUNILFNBQVMsbUJBQW1CLENBQzFCLFdBQXdELEVBQ3hELFFBQStCLEVBQy9CLE1BQXlCLEVBQ3pCLE9BQWUsRUFDZixRQUFnQixFQUNoQixVQUFrQixFQUNsQixTQUFrQjtJQUVsQixJQUFJLENBQUMsV0FBVyxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsRUFBRTtRQUM5QixXQUFXLENBQUMsR0FBRyxDQUFDLFFBQVEsRUFBRTtZQUN4Qix1REFBOEIsRUFBRSxJQUFJLEdBQUcsRUFBRTtZQUN6QywyQkFBZ0IsRUFBRSxJQUFJLEdBQUcsRUFBRTtZQUMzQiwrREFBa0MsRUFBRSxJQUFJLEdBQUcsRUFBRTtZQUM3QywyQ0FBd0IsRUFBRSxJQUFJLEdBQUcsRUFBRTtTQUNwQyxDQUFDLENBQUM7S0FDSjtJQUNELE1BQU0sSUFBSSxHQUFHLFdBQVcsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFFLENBQUM7SUFFeEMsNkRBQTZEO0lBQzdELE1BQU0sYUFBYSxHQUNqQixTQUFTO1FBQ1AsQ0FBQyxDQUFDO1lBQ0EsQ0FBQyxJQUFJLCtEQUFrQyxFQUFFLEdBQUcsVUFBVSxJQUFJLFNBQVMsRUFBRSxDQUFDO1NBQ3ZFO1FBQ0QsQ0FBQyxDQUFDO1lBQ0EsQ0FBQyxJQUFJLHVEQUE4QixFQUFFLFFBQVEsQ0FBQztZQUM5QyxDQUFDLElBQUksMkNBQXdCLEVBQUUsVUFBVSxDQUFDO1lBQzFDLENBQUMsSUFBSSwyQkFBZ0IsRUFBRSxPQUFPLENBQUM7U0FDaEMsQ0FBQztJQUNOLEtBQUssTUFBTSxDQUFDLEdBQUcsRUFBRSxJQUFJLENBQUMsSUFBSSxhQUFhLEVBQUU7UUFDdkMsUUFBUSxNQUFNLEVBQUU7WUFDZDtnQkFDRSxpRUFBaUU7Z0JBQ2pFLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxFQUFFO29CQUNsQixHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztpQkFDdkI7Z0JBQ0QsTUFBTTtZQUNSO2dCQUNFLG9EQUFvRDtnQkFDcEQsR0FBRyxDQUFDLEdBQUcsQ0FBQyxJQUFJLEVBQUUsTUFBTSxDQUFDLENBQUM7Z0JBQ3RCLE1BQU07WUFDUjtnQkFDRSxxRkFBcUY7Z0JBQ3JGLElBQUksQ0FBQyxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxHQUFHLENBQUMsSUFBSSxDQUFDLDRCQUE4QixFQUFFO29CQUNqRSxHQUFHLENBQUMsR0FBRyxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsQ0FBQztpQkFDdkI7Z0JBQ0QsTUFBTTtTQUNUO0tBQ0Y7QUFDSCxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgbWV0cmljU2NvcGUsIENvbmZpZ3VyYXRpb24sIFVuaXQgfSBmcm9tICdhd3MtZW1iZWRkZWQtbWV0cmljcyc7XG5pbXBvcnQgdHlwZSB7IENvbnRleHQsIFNjaGVkdWxlZEV2ZW50IH0gZnJvbSAnYXdzLWxhbWJkYSc7XG5pbXBvcnQgeyBTZW1WZXIgfSBmcm9tICdzZW12ZXInO1xuaW1wb3J0ICogYXMgYXdzIGZyb20gJy4uL3NoYXJlZC9hd3MubGFtYmRhLXNoYXJlZCc7XG5pbXBvcnQgKiBhcyBjb25zdGFudHMgZnJvbSAnLi4vc2hhcmVkL2NvbnN0YW50cyc7XG5pbXBvcnQgeyByZXF1aXJlRW52IH0gZnJvbSAnLi4vc2hhcmVkL2Vudi5sYW1iZGEtc2hhcmVkJztcbmltcG9ydCB7IERvY3VtZW50YXRpb25MYW5ndWFnZSB9IGZyb20gJy4uL3NoYXJlZC9sYW5ndWFnZSc7XG5pbXBvcnQgeyBNRVRSSUNTX05BTUVTUEFDRSwgTWV0cmljTmFtZSwgTEFOR1VBR0VfRElNRU5TSU9OIH0gZnJvbSAnLi9jb25zdGFudHMnO1xuXG5Db25maWd1cmF0aW9uLm5hbWVzcGFjZSA9IE1FVFJJQ1NfTkFNRVNQQUNFO1xuXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gaGFuZGxlcihldmVudDogU2NoZWR1bGVkRXZlbnQsIF9jb250ZXh0OiBDb250ZXh0KSB7XG4gIGNvbnNvbGUubG9nKCdFdmVudDonLCBKU09OLnN0cmluZ2lmeShldmVudCwgbnVsbCwgMikpO1xuXG4gIGNvbnN0IGluZGV4ZWRQYWNrYWdlcyA9IG5ldyBNYXA8c3RyaW5nLCBJbmRleGVkUGFja2FnZVN0YXR1cz4oKTtcbiAgY29uc3QgcGFja2FnZU5hbWVzID0gbmV3IFNldDxzdHJpbmc+KCk7XG4gIGNvbnN0IHBhY2thZ2VNYWpvclZlcnNpb25zID0gbmV3IFNldDxzdHJpbmc+KCk7XG4gIGNvbnN0IHBlckxhbmd1YWdlID0gbmV3IE1hcDxEb2N1bWVudGF0aW9uTGFuZ3VhZ2UsIFBlckxhbmd1YWdlRGF0YT4oKTtcblxuICAvKipcbiAgICogUmVjb3JkcyB0aGUgc3RhdHVzIG9mIGEgcGFydGljdWxhciBwYWNrYWdlLCBwYWNrYWdlIG1ham9yIHZlcnNpb24sIHBhY2thZ2VcbiAgICogdmVyc2lvbiwgYW5kIHBhY2thZ2UgdmVyc2lvbiBzdWJtb2R1bGUgaW4gdGhlIHBlci1sYW5ndWFnZSBzdGF0ZSBzdG9yYWdlLlxuICAgKiBXaGVuZXZlciBhIG5ldyBlbnRyeSBpcyBhZGRlZCwgYSBgTUlTU0lOR2AgZW50cnkgaXMgYXV0b21hdGljYWxseSBpbnNlcnRlZFxuICAgKiBmb3IgdGhlIG90aGVyIGxhbmd1YWdlcyAodW5sZXNzIGFub3RoZXIgZW50cnkgYWxyZWFkeSBleGlzdHMpLlxuICAgKlxuICAgKiBJZiBhIHN1Ym1vZHVsZSBpcyBwcm92aWRlZCwgb25seSB0aGF0IHN1Ym1vZHVsZSdzIGF2YWlsYWJpbGl0eSBpcyB1cGRhdGVkLlxuICAgKi9cbiAgZnVuY3Rpb24gcmVjb3JkUGVyTGFuZ3VhZ2UoXG4gICAgbGFuZ3VhZ2U6IERvY3VtZW50YXRpb25MYW5ndWFnZSxcbiAgICBzdGF0dXM6IFBlckxhbmd1YWdlU3RhdHVzLFxuICAgIHBrZ05hbWU6IHN0cmluZyxcbiAgICBwa2dNYWpvcjogc3RyaW5nLFxuICAgIHBrZ1ZlcnNpb246IHN0cmluZyxcbiAgICBzdWJtb2R1bGU/OiBzdHJpbmcsXG4gICkge1xuICAgIGZvciAoY29uc3QgbGFuZyBvZiBEb2N1bWVudGF0aW9uTGFuZ3VhZ2UuQUxMKSB7XG4gICAgICBkb1JlY29yZFBlckxhbmd1YWdlKFxuICAgICAgICBwZXJMYW5ndWFnZSxcbiAgICAgICAgbGFuZyxcbiAgICAgICAgLy8gSWYgdGhlIGxhbmd1YWdlIGlzIE5PVCB0aGUgcmVnaXN0ZXJlZCBvbmUsIHRoZW4gd2UgaW5zZXJ0IFwiTUlTU0lOR1wiLlxuICAgICAgICBsYW5nID09PSBsYW5ndWFnZSA/IHN0YXR1cyA6IFBlckxhbmd1YWdlU3RhdHVzLk1JU1NJTkcsXG4gICAgICAgIHBrZ05hbWUsXG4gICAgICAgIHBrZ01ham9yLFxuICAgICAgICBwa2dWZXJzaW9uLFxuICAgICAgICBzdWJtb2R1bGUsXG4gICAgICApO1xuICAgIH1cbiAgfVxuXG4gIGNvbnN0IGJ1Y2tldCA9IHJlcXVpcmVFbnYoJ0JVQ0tFVF9OQU1FJyk7XG4gIGZvciBhd2FpdCAoY29uc3Qga2V5IG9mIHJlbGV2YW50T2JqZWN0S2V5cyhidWNrZXQpKSB7XG4gICAgY29uc3QgWywgbmFtZSwgdmVyc2lvbl0gPSBjb25zdGFudHMuU1RPUkFHRV9LRVlfRk9STUFUX1JFR0VYLmV4ZWMoa2V5KSE7XG5cbiAgICBwYWNrYWdlTmFtZXMuYWRkKG5hbWUpO1xuICAgIGNvbnN0IG1ham9yVmVyc2lvbiA9IGAke25hbWV9QCR7bmV3IFNlbVZlcih2ZXJzaW9uKS5tYWpvcn1gO1xuICAgIHBhY2thZ2VNYWpvclZlcnNpb25zLmFkZChtYWpvclZlcnNpb24pO1xuXG4gICAgY29uc3QgZnVsbE5hbWUgPSBgJHtuYW1lfUAke3ZlcnNpb259YDtcblxuICAgIC8vIEVuc3VyZSB0aGUgcGFja2FnZSBpcyBmdWxseSByZWdpc3RlcmVkIGZvciBwZXItbGFuZ3VhZ2Ugc3RhdHVzLCBldmVuIGlmIG5vIGRvYyBleGlzdHMgeWV0LlxuICAgIGZvciAoY29uc3QgbGFuZ3VhZ2Ugb2YgRG9jdW1lbnRhdGlvbkxhbmd1YWdlLkFMTCkge1xuICAgICAgcmVjb3JkUGVyTGFuZ3VhZ2UobGFuZ3VhZ2UsIFBlckxhbmd1YWdlU3RhdHVzLk1JU1NJTkcsIG5hbWUsIG1ham9yVmVyc2lvbiwgZnVsbE5hbWUpO1xuICAgIH1cblxuICAgIGlmICghaW5kZXhlZFBhY2thZ2VzLmhhcyhmdWxsTmFtZSkpIHtcbiAgICAgIGluZGV4ZWRQYWNrYWdlcy5zZXQoZnVsbE5hbWUsIHt9KTtcbiAgICB9XG4gICAgY29uc3Qgc3RhdHVzID0gaW5kZXhlZFBhY2thZ2VzLmdldChmdWxsTmFtZSkhO1xuXG4gICAgaWYgKGtleS5lbmRzV2l0aChjb25zdGFudHMuTUVUQURBVEFfS0VZX1NVRkZJWCkpIHtcbiAgICAgIHN0YXR1cy5tZXRhZGF0YVByZXNlbnQgPSB0cnVlO1xuICAgIH0gZWxzZSBpZiAoa2V5LmVuZHNXaXRoKGNvbnN0YW50cy5QQUNLQUdFX0tFWV9TVUZGSVgpKSB7XG4gICAgICBzdGF0dXMudGFyYmFsbFByZXNlbnQgPSB0cnVlO1xuICAgIH0gZWxzZSBpZiAoa2V5LmVuZHNXaXRoKGNvbnN0YW50cy5BU1NFTUJMWV9LRVlfU1VGRklYKSkge1xuICAgICAgc3RhdHVzLmFzc2VtYmx5UHJlc2VudCA9IHRydWU7XG4gICAgfSBlbHNlIHtcbiAgICAgIGxldCBpZGVudGlmaWVkID0gZmFsc2U7XG4gICAgICBmb3IgKGNvbnN0IGxhbmd1YWdlIG9mIERvY3VtZW50YXRpb25MYW5ndWFnZS5BTEwpIHtcbiAgICAgICAgY29uc3QgbWF0Y2ggPSBzdWJtb2R1bGVLZXlSZWdleHAobGFuZ3VhZ2UpLmV4ZWMoa2V5KTtcbiAgICAgICAgaWYgKG1hdGNoICE9IG51bGwpIHtcbiAgICAgICAgICBjb25zdCBbLCBzdWJtb2R1bGUsIGlzVW5zdXBwb3J0ZWRdID0gbWF0Y2g7XG4gICAgICAgICAgaWYgKHN0YXR1cy5zdWJtb2R1bGVzID09IG51bGwpIHtcbiAgICAgICAgICAgIHN0YXR1cy5zdWJtb2R1bGVzID0gbmV3IFNldCgpO1xuICAgICAgICAgIH1cbiAgICAgICAgICBzdGF0dXMuc3VibW9kdWxlcy5hZGQoYCR7ZnVsbE5hbWV9LiR7c3VibW9kdWxlfWApO1xuICAgICAgICAgIHJlY29yZFBlckxhbmd1YWdlKFxuICAgICAgICAgICAgbGFuZ3VhZ2UsXG4gICAgICAgICAgICBpc1Vuc3VwcG9ydGVkID8gUGVyTGFuZ3VhZ2VTdGF0dXMuVU5TVVBQT1JURUQgOiBQZXJMYW5ndWFnZVN0YXR1cy5TVVBQT1JURUQsXG4gICAgICAgICAgICBuYW1lLFxuICAgICAgICAgICAgbWFqb3JWZXJzaW9uLFxuICAgICAgICAgICAgZnVsbE5hbWUsXG4gICAgICAgICAgICBzdWJtb2R1bGUsXG4gICAgICAgICAgKTtcbiAgICAgICAgICBpZGVudGlmaWVkID0gdHJ1ZTtcbiAgICAgICAgfSBlbHNlIGlmIChrZXkuZW5kc1dpdGgoY29uc3RhbnRzLmRvY3NLZXlTdWZmaXgobGFuZ3VhZ2UpKSkge1xuICAgICAgICAgIHJlY29yZFBlckxhbmd1YWdlKGxhbmd1YWdlLCBQZXJMYW5ndWFnZVN0YXR1cy5TVVBQT1JURUQsIG5hbWUsIG1ham9yVmVyc2lvbiwgZnVsbE5hbWUpO1xuICAgICAgICAgIGlkZW50aWZpZWQgPSB0cnVlO1xuICAgICAgICB9IGVsc2UgaWYgKGtleS5lbmRzV2l0aChjb25zdGFudHMuZG9jc0tleVN1ZmZpeChsYW5ndWFnZSkgKyBjb25zdGFudHMuTk9UX1NVUFBPUlRFRF9TVUZGSVgpKSB7XG4gICAgICAgICAgcmVjb3JkUGVyTGFuZ3VhZ2UobGFuZ3VhZ2UsIFBlckxhbmd1YWdlU3RhdHVzLlNVUFBPUlRFRCwgbmFtZSwgbWFqb3JWZXJzaW9uLCBmdWxsTmFtZSk7XG4gICAgICAgICAgaWRlbnRpZmllZCA9IHRydWU7XG4gICAgICAgIH1cbiAgICAgIH1cbiAgICAgIGlmICghaWRlbnRpZmllZCkge1xuICAgICAgICBzdGF0dXMudW5rbm93bk9iamVjdHMgPSBzdGF0dXMudW5rbm93bk9iamVjdHMgPz8gW107XG4gICAgICAgIHN0YXR1cy51bmtub3duT2JqZWN0cy5wdXNoKGtleSk7XG4gICAgICB9XG4gICAgfVxuICB9XG5cbiAgYXdhaXQgbWV0cmljU2NvcGUoKG1ldHJpY3MpID0+ICgpID0+IHtcbiAgICAvLyBDbGVhciBvdXQgZGVmYXVsdCBkaW1lbnNpb25zIGFzIHdlIGRvbid0IG5lZWQgdGhvc2UuIFNlZSBodHRwczovL2dpdGh1Yi5jb20vYXdzbGFicy9hd3MtZW1iZWRkZWQtbWV0cmljcy1ub2RlL2lzc3Vlcy83My5cbiAgICBtZXRyaWNzLnNldERpbWVuc2lvbnMoKTtcblxuICAgIGNvbnN0IG1pc3NpbmdNZXRhZGF0YSA9IG5ldyBBcnJheTxzdHJpbmc+KCk7XG4gICAgY29uc3QgbWlzc2luZ0Fzc2VtYmx5ID0gbmV3IEFycmF5PHN0cmluZz4oKTtcbiAgICBjb25zdCBtaXNzaW5nVGFyYmFsbCA9IG5ldyBBcnJheTxzdHJpbmc+KCk7XG4gICAgY29uc3QgdW5rbm93bk9iamVjdHMgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuICAgIGNvbnN0IHN1Ym1vZHVsZXMgPSBuZXcgQXJyYXk8c3RyaW5nPigpO1xuICAgIGZvciAoY29uc3QgW25hbWUsIHN0YXR1c10gb2YgaW5kZXhlZFBhY2thZ2VzLmVudHJpZXMoKSkge1xuICAgICAgaWYgKCFzdGF0dXMubWV0YWRhdGFQcmVzZW50KSB7XG4gICAgICAgIG1pc3NpbmdNZXRhZGF0YS5wdXNoKG5hbWUpO1xuICAgICAgfVxuICAgICAgaWYgKCFzdGF0dXMuYXNzZW1ibHlQcmVzZW50KSB7XG4gICAgICAgIG1pc3NpbmdBc3NlbWJseS5wdXNoKG5hbWUpO1xuICAgICAgfVxuICAgICAgaWYgKCFzdGF0dXMudGFyYmFsbFByZXNlbnQpIHtcbiAgICAgICAgbWlzc2luZ1RhcmJhbGwucHVzaChuYW1lKTtcbiAgICAgIH1cbiAgICAgIGlmIChzdGF0dXMudW5rbm93bk9iamVjdHM/Lmxlbmd0aCA/PyAwID4gMCkge1xuICAgICAgICB1bmtub3duT2JqZWN0cy5wdXNoKC4uLnN0YXR1cy51bmtub3duT2JqZWN0cyEpO1xuICAgICAgfVxuXG4gICAgICBmb3IgKGNvbnN0IHN1Ym1vZHVsZSBvZiBzdGF0dXMuc3VibW9kdWxlcyA/PyBbXSkge1xuICAgICAgICBzdWJtb2R1bGVzLnB1c2goc3VibW9kdWxlKTtcbiAgICAgIH1cbiAgICB9XG5cbiAgICBtZXRyaWNzLnNldFByb3BlcnR5KCdkZXRhaWwnLCB7IG1pc3NpbmdNZXRhZGF0YSwgbWlzc2luZ0Fzc2VtYmx5LCBtaXNzaW5nVGFyYmFsbCwgdW5rbm93bk9iamVjdHMgfSk7XG5cbiAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLk1JU1NJTkdfTUVUQURBVEFfQ09VTlQsIG1pc3NpbmdNZXRhZGF0YS5sZW5ndGgsIFVuaXQuQ291bnQpO1xuICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuTUlTU0lOR19BU1NFTUJMWV9DT1VOVCwgbWlzc2luZ0Fzc2VtYmx5Lmxlbmd0aCwgVW5pdC5Db3VudCk7XG4gICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5NSVNTSU5HX1RBUkJBTExfQ09VTlQsIG1pc3NpbmdUYXJiYWxsLmxlbmd0aCwgVW5pdC5Db3VudCk7XG4gICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5QQUNLQUdFX0NPVU5ULCBwYWNrYWdlTmFtZXMuc2l6ZSwgVW5pdC5Db3VudCk7XG4gICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5QQUNLQUdFX01BSk9SX0NPVU5ULCBwYWNrYWdlTWFqb3JWZXJzaW9ucy5zaXplLCBVbml0LkNvdW50KTtcbiAgICBtZXRyaWNzLnB1dE1ldHJpYyhNZXRyaWNOYW1lLlBBQ0tBR0VfVkVSU0lPTl9DT1VOVCwgaW5kZXhlZFBhY2thZ2VzLnNpemUsIFVuaXQuQ291bnQpO1xuICAgIG1ldHJpY3MucHV0TWV0cmljKE1ldHJpY05hbWUuU1VCTU9EVUxFX0NPVU5ULCBzdWJtb2R1bGVzLmxlbmd0aCwgVW5pdC5Db3VudCk7XG4gICAgbWV0cmljcy5wdXRNZXRyaWMoTWV0cmljTmFtZS5VTktOT1dOX09CSkVDVF9DT1VOVCwgdW5rbm93bk9iamVjdHMubGVuZ3RoLCBVbml0LkNvdW50KTtcbiAgfSkoKTtcblxuICBmb3IgKGNvbnN0IGVudHJ5IG9mIHBlckxhbmd1YWdlLmVudHJpZXMoKSkge1xuICAgIGF3YWl0IG1ldHJpY1Njb3BlKChtZXRyaWNzKSA9PiAoW2xhbmd1YWdlLCBkYXRhXTogW0RvY3VtZW50YXRpb25MYW5ndWFnZSwgUGVyTGFuZ3VhZ2VEYXRhXSkgPT4ge1xuICAgICAgbWV0cmljcy5zZXREaW1lbnNpb25zKHsgW0xBTkdVQUdFX0RJTUVOU0lPTl06IGxhbmd1YWdlLnRvU3RyaW5nKCkgfSk7XG5cbiAgICAgIGZvciAoY29uc3QgZm9yU3RhdHVzIG9mIFtQZXJMYW5ndWFnZVN0YXR1cy5TVVBQT1JURUQsIFBlckxhbmd1YWdlU3RhdHVzLlVOU1VQUE9SVEVELCBQZXJMYW5ndWFnZVN0YXR1cy5NSVNTSU5HXSkge1xuICAgICAgICBmb3IgKGNvbnN0IFtrZXksIHN0YXR1c2VzXSBvZiBPYmplY3QuZW50cmllcyhkYXRhKSkge1xuICAgICAgICAgIGxldCBmaWx0ZXJlZCA9IEFycmF5LmZyb20oc3RhdHVzZXMuZW50cmllcygpKS5maWx0ZXIoKFssIHN0YXR1c10pID0+IGZvclN0YXR1cyA9PT0gc3RhdHVzKTtcbiAgICAgICAgICBsZXQgbWV0cmljTmFtZSA9IE1FVFJJQ19OQU1FX0JZX1NUQVRVU19BTkRfR1JBSU5bZm9yU3RhdHVzIGFzIFBlckxhbmd1YWdlU3RhdHVzXVtrZXkgYXMga2V5b2YgUGVyTGFuZ3VhZ2VEYXRhXTtcbiAgICAgICAgICAvLyBMaXN0IG91dCBzZWxlY3RlZCBwYWNrYWdlcyBmb3IgcG9zdGVyaXR5IChhbmQgdHJvdWJsZXNob290aW5nKVxuICAgICAgICAgIGNvbnNvbGUubG9nKGAke2ZvclN0YXR1c30gJHtrZXl9IGZvciAke2xhbmd1YWdlfTogJHtmaWx0ZXJlZC5tYXAoKFtuYW1lXSkgPT4gbmFtZSkuam9pbignLCAnKX1gKTtcbiAgICAgICAgICBtZXRyaWNzLnB1dE1ldHJpYyhtZXRyaWNOYW1lLCBmaWx0ZXJlZC5sZW5ndGgsIFVuaXQuQ291bnQpO1xuICAgICAgICB9XG4gICAgICB9XG4gICAgfSkoZW50cnkpO1xuICB9XG59XG5cbmFzeW5jIGZ1bmN0aW9uKiByZWxldmFudE9iamVjdEtleXMoYnVja2V0OiBzdHJpbmcpOiBBc3luY0dlbmVyYXRvcjxzdHJpbmcsIHZvaWQsIHZvaWQ+IHtcbiAgY29uc3QgcmVxdWVzdDogQVdTLlMzLkxpc3RPYmplY3RzVjJSZXF1ZXN0ID0ge1xuICAgIEJ1Y2tldDogYnVja2V0LFxuICAgIFByZWZpeDogY29uc3RhbnRzLlNUT1JBR0VfS0VZX1BSRUZJWCxcbiAgfTtcbiAgZG8ge1xuICAgIGNvbnN0IHJlc3BvbnNlID0gYXdhaXQgYXdzLnMzKCkubGlzdE9iamVjdHNWMihyZXF1ZXN0KS5wcm9taXNlKCk7XG4gICAgZm9yIChjb25zdCB7IEtleSB9IG9mIHJlc3BvbnNlLkNvbnRlbnRzID8/IFtdKSB7XG4gICAgICBpZiAoS2V5ID09IG51bGwpIHsgY29udGludWU7IH1cbiAgICAgIHlpZWxkIEtleTtcbiAgICB9XG4gICAgcmVxdWVzdC5Db250aW51YXRpb25Ub2tlbiA9IHJlc3BvbnNlLk5leHRDb250aW51YXRpb25Ub2tlbjtcbiAgfSB3aGlsZSAocmVxdWVzdC5Db250aW51YXRpb25Ub2tlbiAhPSBudWxsKTtcbn1cblxuLyoqXG4gKiBUaGlzIGZ1bmN0aW9uIG9idGFpbnMgYSByZWd1bGFyIGV4cHJlc3Npb24gd2l0aCBhIGNhcHR1cmUgZ3JvdXAgdGhhdCBhbGxvd3NcbiAqIGRldGVybWluaW5nIHRoZSBzdWJtb2R1bGUgbmFtZSBmcm9tIGEgc3VibW9kdWxlIGRvY3VtZW50YXRpb24ga2V5LCBhbmRcbiAqIGFub3RoZXIgdG8gZGV0ZXJtaW5lIHdoZXRoZXIgdGhlIG9iamVjdCBpcyBhbiBcInVuc3VwcG9ydGVkIGJlYWNvblwiIG9yIG5vdC5cbiAqL1xuZnVuY3Rpb24gc3VibW9kdWxlS2V5UmVnZXhwKGxhbmd1YWdlOiBEb2N1bWVudGF0aW9uTGFuZ3VhZ2UpOiBSZWdFeHAge1xuICAvLyBXZSB1c2UgYSBwbGFjZWhvbGRlciB0byBiZSBhYmxlIHRvIGluc2VydCB0aGUgY2FwdHVyZSBncm91cCBvbmNlIHdlIGhhdmVcbiAgLy8gZnVsbHkgcXVvdGVkIHRoZSBrZXkgcHJlZml4IGZvciBSZWdleCBzYWZldHkuXG4gIGNvbnN0IHBsYWNlaG9sZGVyID0gJzxTVUJNT0RVTEVOQU1FPic7XG5cbiAgLy8gV2Ugb2J0YWluIHRoZSBzdGFuZGFyZCBrZXkgcHJlZml4LlxuICBjb25zdCBrZXlQcmVmaXggPSBjb25zdGFudHMuZG9jc0tleVN1ZmZpeChsYW5ndWFnZSwgcGxhY2Vob2xkZXIpO1xuXG4gIC8vIEZpbmFsbHksIGFzc2VtYmxlIHRoZSByZWd1bGFyIGV4cHJlc3Npb24gd2l0aCB0aGUgY2FwdHVyZSBncm91cC5cbiAgcmV0dXJuIG5ldyBSZWdFeHAoYC4qJHtyZVF1b3RlKGtleVByZWZpeCkucmVwbGFjZShwbGFjZWhvbGRlciwgJyguKyknKX0oJHtyZVF1b3RlKGNvbnN0YW50cy5OT1RfU1VQUE9SVEVEX1NVRkZJWCl9KT8kYCk7XG5cbiAgLyoqXG4gICAqIEVzY2FwZXMgYWxsIFwic3BlYWNpYWwgbWVhbmluZ1wiIGNoYXJhY3RlcnMgaW4gYSBzdHJpbmcsIHNvIGl0IGNhbiBiZSB1c2VkIGFzXG4gICAqIHBhcnQgb2YgYSByZWd1bGFyIGV4cHJlc3Npb24uXG4gICAqL1xuICBmdW5jdGlvbiByZVF1b3RlKHN0cjogc3RyaW5nKTogc3RyaW5nIHtcbiAgICByZXR1cm4gc3RyLnJlcGxhY2UoLyhbKyouKCk/JFtcXF1dKS9nLCAnXFxcXCQxJyk7XG4gIH1cbn1cblxuaW50ZXJmYWNlIEluZGV4ZWRQYWNrYWdlU3RhdHVzIHtcbiAgbWV0YWRhdGFQcmVzZW50PzogYm9vbGVhbjtcbiAgYXNzZW1ibHlQcmVzZW50PzogYm9vbGVhbjtcbiAgc3VibW9kdWxlcz86IFNldDxzdHJpbmc+O1xuICB0YXJiYWxsUHJlc2VudD86IGJvb2xlYW47XG4gIHVua25vd25PYmplY3RzPzogc3RyaW5nW107XG59XG5cbmNvbnN0IGVudW0gR3JhaW4ge1xuICBQQUNLQUdFX01BSk9SX1ZFUlNJT05TID0gJ3BhY2thZ2UgbWFqb3IgdmVyc2lvbnMnLFxuICBQQUNLQUdFX1ZFUlNJT05fU1VCTU9EVUxFUyA9ICdwYWNrYWdlIHZlcnNpb24gc3VibW9kdWxlcycsXG4gIFBBQ0tBR0VfVkVSU0lPTlMgPSAncGFja2FnZSB2ZXJzaW9ucycsXG4gIFBBQ0tBR0VTID0gJ3BhY2thZ2VzJyxcbn1cblxudHlwZSBQZXJMYW5ndWFnZURhdGEgPSB7IHJlYWRvbmx5IFtncmFpbiBpbiBHcmFpbl06IE1hcDxzdHJpbmcsIFBlckxhbmd1YWdlU3RhdHVzPiB9O1xuXG5jb25zdCBlbnVtIFBlckxhbmd1YWdlU3RhdHVzIHtcbiAgTUlTU0lORyA9ICdNaXNzaW5nJyxcbiAgVU5TVVBQT1JURUQgPSAnVW5zdXBwb3J0ZWQnLFxuICBTVVBQT1JURUQgPSAnU3VwcG9ydGVkJyxcbn1cblxuY29uc3QgTUVUUklDX05BTUVfQllfU1RBVFVTX0FORF9HUkFJTjogeyByZWFkb25seSBbc3RhdHVzIGluIFBlckxhbmd1YWdlU3RhdHVzXTogeyByZWFkb25seSBbZ3JhaW4gaW4gR3JhaW5dOiBNZXRyaWNOYW1lIH0gfSA9IHtcbiAgW1Blckxhbmd1YWdlU3RhdHVzLk1JU1NJTkddOiB7XG4gICAgW0dyYWluLlBBQ0tBR0VTXTogTWV0cmljTmFtZS5QRVJfTEFOR1VBR0VfTUlTU0lOR19QQUNLQUdFUyxcbiAgICBbR3JhaW4uUEFDS0FHRV9NQUpPUl9WRVJTSU9OU106IE1ldHJpY05hbWUuUEVSX0xBTkdVQUdFX01JU1NJTkdfTUFKT1JTLFxuICAgIFtHcmFpbi5QQUNLQUdFX1ZFUlNJT05TXTogTWV0cmljTmFtZS5QRVJfTEFOR1VBR0VfTUlTU0lOR19WRVJTSU9OUyxcbiAgICBbR3JhaW4uUEFDS0FHRV9WRVJTSU9OX1NVQk1PRFVMRVNdOiBNZXRyaWNOYW1lLlBFUl9MQU5HVUFHRV9NSVNTSU5HX1NVQk1PRFVMRVMsXG4gIH0sXG4gIFtQZXJMYW5ndWFnZVN0YXR1cy5VTlNVUFBPUlRFRF06IHtcbiAgICBbR3JhaW4uUEFDS0FHRVNdOiBNZXRyaWNOYW1lLlBFUl9MQU5HVUFHRV9VTlNVUFBPUlRFRF9QQUNLQUdFUyxcbiAgICBbR3JhaW4uUEFDS0FHRV9NQUpPUl9WRVJTSU9OU106IE1ldHJpY05hbWUuUEVSX0xBTkdVQUdFX1VOU1VQUE9SVEVEX01BSk9SUyxcbiAgICBbR3JhaW4uUEFDS0FHRV9WRVJTSU9OU106IE1ldHJpY05hbWUuUEVSX0xBTkdVQUdFX1VOU1VQUE9SVEVEX1ZFUlNJT05TLFxuICAgIFtHcmFpbi5QQUNLQUdFX1ZFUlNJT05fU1VCTU9EVUxFU106IE1ldHJpY05hbWUuUEVSX0xBTkdVQUdFX1VOU1VQUE9SVEVEX1NVQk1PRFVMRVMsXG4gIH0sXG4gIFtQZXJMYW5ndWFnZVN0YXR1cy5TVVBQT1JURURdOiB7XG4gICAgW0dyYWluLlBBQ0tBR0VTXTogTWV0cmljTmFtZS5QRVJfTEFOR1VBR0VfU1VQUE9SVEVEX1BBQ0tBR0VTLFxuICAgIFtHcmFpbi5QQUNLQUdFX01BSk9SX1ZFUlNJT05TXTogTWV0cmljTmFtZS5QRVJfTEFOR1VBR0VfU1VQUE9SVEVEX01BSk9SUyxcbiAgICBbR3JhaW4uUEFDS0FHRV9WRVJTSU9OU106IE1ldHJpY05hbWUuUEVSX0xBTkdVQUdFX1NVUFBPUlRFRF9WRVJTSU9OUyxcbiAgICBbR3JhaW4uUEFDS0FHRV9WRVJTSU9OX1NVQk1PRFVMRVNdOiBNZXRyaWNOYW1lLlBFUl9MQU5HVUFHRV9TVVBQT1JURURfU1VCTU9EVUxFUyxcbiAgfSxcbn07XG5cblxuLyoqXG4gKiBSZWdpc3RlcnMgdGhlIGluZm9ybWF0aW9uIGZvciB0aGUgcHJvdmlkZWQgbGFuZ3VhZ2UuIEEgXCJNSVNTSU5HXCIgc3RhdHVzXG4gKiB3aWxsIGJlIGlnbm9yZWQgaWYgYW5vdGhlciBzdGF0dXMgd2FzIGFscmVhZHkgcmVnaXN0ZXJlZCBmb3IgdGhlIHNhbWVcbiAqIGVudGl0eS4gQW4gXCJVTlNVUFBPUlRFRFwiIHN0YXR1cyB3aWxsIGJlIGlnbm9yZWQgaWYgYSBcIlNVUFBPUlRFRFwiIHN0YXR1c1xuICogd2FzIGFscmVhZHkgcmVnaXN0ZXJlZCBmb3IgdGhlIHNhbWUgZW50aXR5LlxuICpcbiAqIElmIGEgc3VibW9kdWxlIGlzIHByb3ZpZGVkLCBvbmx5IHRoYXQgc3VibW9kdWxlJ3MgYXZhaWxhYmlsaXR5IGlzIHVwZGF0ZWQuXG4gKi9cbmZ1bmN0aW9uIGRvUmVjb3JkUGVyTGFuZ3VhZ2UoXG4gIHBlckxhbmd1YWdlOiBNYXA8RG9jdW1lbnRhdGlvbkxhbmd1YWdlLCBQZXJMYW5ndWFnZURhdGE+LFxuICBsYW5ndWFnZTogRG9jdW1lbnRhdGlvbkxhbmd1YWdlLFxuICBzdGF0dXM6IFBlckxhbmd1YWdlU3RhdHVzLFxuICBwa2dOYW1lOiBzdHJpbmcsXG4gIHBrZ01ham9yOiBzdHJpbmcsXG4gIHBrZ1ZlcnNpb246IHN0cmluZyxcbiAgc3VibW9kdWxlPzogc3RyaW5nLFxuKSB7XG4gIGlmICghcGVyTGFuZ3VhZ2UuaGFzKGxhbmd1YWdlKSkge1xuICAgIHBlckxhbmd1YWdlLnNldChsYW5ndWFnZSwge1xuICAgICAgW0dyYWluLlBBQ0tBR0VfTUFKT1JfVkVSU0lPTlNdOiBuZXcgTWFwKCksXG4gICAgICBbR3JhaW4uUEFDS0FHRVNdOiBuZXcgTWFwKCksXG4gICAgICBbR3JhaW4uUEFDS0FHRV9WRVJTSU9OX1NVQk1PRFVMRVNdOiBuZXcgTWFwKCksXG4gICAgICBbR3JhaW4uUEFDS0FHRV9WRVJTSU9OU106IG5ldyBNYXAoKSxcbiAgICB9KTtcbiAgfVxuICBjb25zdCBkYXRhID0gcGVyTGFuZ3VhZ2UuZ2V0KGxhbmd1YWdlKSE7XG5cbiAgLy8gSWYgdGhlcmUgaXMgYSBzdWJtb2R1bGUsIG9ubHkgdXBkYXRlIHRoZSBzdWJtb2R1bGUgZG9tYWluLlxuICBjb25zdCBvdXRwdXREb21haW5zOiByZWFkb25seSBbTWFwPHN0cmluZywgUGVyTGFuZ3VhZ2VTdGF0dXM+LCBzdHJpbmddW10gPVxuICAgIHN1Ym1vZHVsZVxuICAgICAgPyBbXG4gICAgICAgIFtkYXRhW0dyYWluLlBBQ0tBR0VfVkVSU0lPTl9TVUJNT0RVTEVTXSwgYCR7cGtnVmVyc2lvbn0uJHtzdWJtb2R1bGV9YF0sXG4gICAgICBdXG4gICAgICA6IFtcbiAgICAgICAgW2RhdGFbR3JhaW4uUEFDS0FHRV9NQUpPUl9WRVJTSU9OU10sIHBrZ01ham9yXSxcbiAgICAgICAgW2RhdGFbR3JhaW4uUEFDS0FHRV9WRVJTSU9OU10sIHBrZ1ZlcnNpb25dLFxuICAgICAgICBbZGF0YVtHcmFpbi5QQUNLQUdFU10sIHBrZ05hbWVdLFxuICAgICAgXTtcbiAgZm9yIChjb25zdCBbbWFwLCBuYW1lXSBvZiBvdXRwdXREb21haW5zKSB7XG4gICAgc3dpdGNoIChzdGF0dXMpIHtcbiAgICAgIGNhc2UgUGVyTGFuZ3VhZ2VTdGF0dXMuTUlTU0lORzpcbiAgICAgICAgLy8gSWYgd2UgYWxyZWFkeSBoYXZlIGEgc3RhdHVzLCBkb24ndCBvdmVycmlkZSBpdCB3aXRoIFwiTUlTU0lOR1wiLlxuICAgICAgICBpZiAoIW1hcC5oYXMobmFtZSkpIHtcbiAgICAgICAgICBtYXAuc2V0KG5hbWUsIHN0YXR1cyk7XG4gICAgICAgIH1cbiAgICAgICAgYnJlYWs7XG4gICAgICBjYXNlIFBlckxhbmd1YWdlU3RhdHVzLlNVUFBPUlRFRDpcbiAgICAgICAgLy8gSWYgdGhyIHBhY2thZ2UgaXMgXCJzdXBwb3J0ZWRcIiwgdGhpcyBhbHdheXMgXCJ3aW5zXCJcbiAgICAgICAgbWFwLnNldChuYW1lLCBzdGF0dXMpO1xuICAgICAgICBicmVhaztcbiAgICAgIGNhc2UgUGVyTGFuZ3VhZ2VTdGF0dXMuVU5TVVBQT1JURUQ6XG4gICAgICAgIC8vIElmIHdlIGFscmVhZHkgaGF2ZSBhIHN0YXR1cywgb25seSBvdmVycmlkZSB3aXRoIFwiVU5TVVBQT1JURURcIiBpZiBpdCB3YXMgXCJNSVNTSU5HXCIuXG4gICAgICAgIGlmICghbWFwLmhhcyhuYW1lKSB8fCBtYXAuZ2V0KG5hbWUpID09PSBQZXJMYW5ndWFnZVN0YXR1cy5NSVNTSU5HKSB7XG4gICAgICAgICAgbWFwLnNldChuYW1lLCBzdGF0dXMpO1xuICAgICAgICB9XG4gICAgICAgIGJyZWFrO1xuICAgIH1cbiAgfVxufVxuIl19