"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmJs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../deep-link");
const metric_utils_1 = require("../metric-utils");
const runbook_url_1 = require("../runbook-url");
const constants_lambda_shared_1 = require("./npmjs/constants.lambda-shared");
const npm_js_follower_1 = require("./npmjs/npm-js-follower");
/**
 * (experimental) A package source that gets package data from the npmjs.com package registry.
 *
 * @experimental
 */
class NpmJs {
    /**
     * @experimental
     */
    constructor(props = {}) {
        this.props = props;
    }
    /**
     * (experimental) Binds the package source to a scope and target queue.
     *
     * @experimental
     */
    bind(scope, { denyList, ingestion, licenseList, monitoring, queue, repository }) {
        repository === null || repository === void 0 ? void 0 : repository.addExternalConnection('public:npmjs');
        const bucket = this.props.stagingBucket || new aws_s3_1.Bucket(scope, 'NpmJs/StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [{ prefix: "staged/" /* STAGED_KEY_PREFIX */, expiration: core_1.Duration.days(30) }],
        });
        bucket.grantRead(ingestion);
        const follower = new npm_js_follower_1.NpmJsFollower(scope, 'NpmJs', {
            description: `[${scope.node.path}/NpmJs] Periodically query npmjs.com index for new packages`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                QUEUE_URL: queue.queueUrl,
            },
            memorySize: 10024,
            reservedConcurrentExecutions: 1,
            timeout: core_1.Duration.minutes(5),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(follower);
        queue.grantSendMessages(follower);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(follower);
        licenseList.grantRead(follower);
        const rule = new aws_events_1.Rule(scope, 'NpmJs/Schedule', {
            description: `${scope.node.path}/NpmJs/Schedule`,
            schedule: aws_events_1.Schedule.rate(core_1.Duration.minutes(5)),
            targets: [new aws_events_targets_1.LambdaFunction(follower)],
        });
        const failureAlarm = follower.metricErrors().createAlarm(scope, 'NpmJs/Follower/Failures', {
            alarmName: `${scope.node.path}/NpmJs/Follower/Failures`,
            alarmDescription: [
                'The NpmJs follower function failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 3,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Follower Failures', failureAlarm);
        const notRunningAlarm = follower.metricInvocations().createAlarm(scope, 'NpmJs/Follower/NotRunning', {
            alarmName: `${scope.node.path}/NpmJs/Follower/NotRunning`,
            alarmDescription: [
                'The NpmJs follower function is not running!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Follower Not Running', notRunningAlarm);
        const noChangeAlarm = this.metricChangeCount().createAlarm(scope, 'NpmJs/Follower/NoChanges', {
            alarmName: `${scope.node.path}/NpmJs/Follower/NoChanges`,
            alarmDescription: [
                'The NpmJs follower function is no discovering any changes from CouchDB!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // If the metric is not emitted, it can be assumed to be zero.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        monitoring.addLowSeverityAlarm('Np npmjs.com changes discovered', noChangeAlarm);
        // Finally - the "not running" alarm depends on the schedule (it won't run until the schedule
        // exists!), and the schedule depends on the failure alarm existing (we don't want it to run
        // before we can know it is failing). This means the returned `IDependable` effectively ensures
        // all alarms have been provisionned already! Isn't it nice!
        notRunningAlarm.node.addDependency(rule);
        rule.node.addDependency(failureAlarm);
        return {
            name: follower.node.path,
            links: [
                { name: 'NpmJs Follower', url: deep_link_1.lambdaFunctionUrl(follower), primary: true },
                { name: 'Marker Object', url: deep_link_1.s3ObjectUrl(bucket, constants_lambda_shared_1.MARKER_FILE_NAME) },
            ],
            dashboardWidgets: [
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            metric_utils_1.fillMetric(follower.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(follower.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            this.metricRemainingTime({ label: 'Remaining Time' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(15),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Follower',
                        left: [
                            this.metricChangeCount({ label: 'Change Count' }),
                            this.metricUnprocessableEntity({ label: 'Unprocessable' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            metric_utils_1.fillMetric(this.metricNpmJsChangeAge({ label: 'Lag to npmjs.com' }), 'REPEAT'),
                            metric_utils_1.fillMetric(this.metricPackageVersionAge({ label: 'Package Version Age' }), 'REPEAT'),
                        ],
                        rightYAxis: { label: 'Milliseconds', min: 0, showUnits: false },
                        period: core_1.Duration.minutes(15),
                    }),
                ], [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Changes',
                        left: [
                            this.metricLastSeq({ label: 'Last Sequence Number' }),
                        ],
                        period: core_1.Duration.minutes(15),
                    }),
                ],
            ],
        };
    }
    /**
     * (experimental) The average time it took to process a changes batch.
     *
     * @experimental
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of changes that were processed.
     *
     * @experimental
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The last sequence number that was processed.
     *
     * This metric can be used to
     * discover when a sequence reset has happened in the CouchDB instance.
     *
     * @experimental
     */
    metricLastSeq(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "LastSeq" /* LAST_SEQ */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * @experimental
     */
    metricNpmJsChangeAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "NpmJsChangeAge" /* NPMJS_CHANGE_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The age of the oldest package version that was processed.
     *
     * @experimental
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of package versions that were inspected.
     *
     * @experimental
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of package versions that were deemed relevant.
     *
     * @experimental
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The amount of time that was remaining when the lambda returned in order to avoid hitting a timeout.
     *
     * @experimental
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The total count of staging failures.
     *
     * @experimental
     */
    metricStagingFailureCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "StagingFailureCount" /* STAGING_FAILURE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The average time it took to stage a package to S3.
     *
     * @experimental
     */
    metricStagingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "StagingTime" /* STAGING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * (experimental) The amount of changes that were not processed due to having an invalid format.
     *
     * @experimental
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
}
exports.NpmJs = NpmJs;
_a = JSII_RTTI_SYMBOL_1;
NpmJs[_a] = { fqn: "construct-hub.sources.NpmJs", version: "0.2.17" };
//# sourceMappingURL=data:application/json;base64,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