"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.throughVpcEndpoint = void 0;
const aws_iam_1 = require("@aws-cdk/aws-iam");
/**
 * Decorates an S3 Bucket so that grants are made including a VPC endpoint
 * policy.
 *
 * This currently only supports the `gratRead` and `grantWrite` APIs.
 *
 * @param bucket      the bucket to be wrapped.
 * @param vpcEndpoint the VPC Endpoint for S3 to be used.
 *
 * @returns a decorated S3 Bucket.
 */
function throughVpcEndpoint(bucket, vpcEndpoint) {
    return new Proxy(bucket, {
        get(target, property, _receiver) {
            switch (property) {
                case 'grantRead':
                    return decoratedGrantRead.bind(target);
                case 'grantWrite':
                    return decoratedGrantWrite.bind(target);
                default:
                    if (typeof property === 'string' && /^grant([A-Z]|$)/.test(property)) {
                        console.warn(`No VPC Endpoint policy grants will be added for ${property} on ${bucket.node.path}`);
                    }
                    return target[property];
            }
        },
        getOwnPropertyDescriptor(target, property) {
            const realDescriptor = Object.getOwnPropertyDescriptor(target, property);
            switch (property) {
                case 'grantRead':
                    return {
                        ...realDescriptor,
                        value: decoratedGrantRead,
                        get: undefined,
                        set: undefined,
                    };
                case 'grantWrite':
                    return {
                        ...realDescriptor,
                        value: decoratedGrantWrite,
                        get: undefined,
                        set: undefined,
                    };
                default:
                    if (typeof property === 'string' && /^grant([A-Z]|$)/.test(property)) {
                        console.warn(`No VPC Endpoint policy grants will be added for ${property} on ${bucket.node.path}`);
                    }
                    return realDescriptor;
            }
        },
    });
    function decoratedGrantRead(identity, objectsKeyPattern = '*') {
        const mainGrant = this.grantRead(identity, objectsKeyPattern);
        if (mainGrant.success) {
            vpcEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                resources: [this.bucketArn, this.arnForObjects(objectsKeyPattern)],
                // Gateway endpoints have this pecular quirk about them that the
                // `principals` are compared strictly using *EXACT MATCH*, meaning you
                // cannot restrict to a particular role, as the actual principal will be
                // an STS assumed-role principal, which cannot be fully predicted. So we
                // would have used a condition to enact this limitation... But
                // unfortunately the `IGrantable` API does not allow us to access the
                // principal ARN for the grantee, so we just skip that... The principal
                // policy will have been configured to limit access already anyway!
                principals: [new aws_iam_1.AnyPrincipal()],
            }));
        }
        return mainGrant;
    }
    function decoratedGrantWrite(identity, objectsKeyPattern = '*') {
        const mainGrant = this.grantWrite(identity, objectsKeyPattern);
        if (mainGrant.success) {
            vpcEndpoint.addToPolicy(new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['s3:Abort*', 's3:DeleteObject*', 's3:PutObject*'],
                resources: [this.bucketArn, this.arnForObjects(objectsKeyPattern)],
                // Gateway endpoints have this pecular quirk about them that the
                // `principals` are compared strictly using *EXACT MATCH*, meaning you
                // cannot restrict to a particular role, as the actual principal will be
                // an STS assumed-role principal, which cannot be fully predicted. So we
                // would have used a condition to enact this limitation... But
                // unfortunately the `IGrantable` API does not allow us to access the
                // principal ARN for the grantee, so we just skip that... The principal
                // policy will have been configured to limit access already anyway!
                principals: [new aws_iam_1.AnyPrincipal()],
            }));
        }
        return mainGrant;
    }
}
exports.throughVpcEndpoint = throughVpcEndpoint;
//# sourceMappingURL=data:application/json;base64,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