"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
// Imports
const core_1 = require("@aws-cdk/core");
const lib_1 = require("../lib");
const lambda = require("@aws-cdk/aws-lambda");
const sqs = require("@aws-cdk/aws-sqs");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
// --------------------------------------------------------------
// Pattern deployment w/ new Lambda function and
// default properties
// --------------------------------------------------------------
test('Pattern deployment w/ new Lambda function and default props', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`)
        }
    };
    new lib_1.SqsToLambda(stack, 'test-sqs-lambda', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Pattern deployment w/ new Lambda function and
// overridden properties
// --------------------------------------------------------------
test('Pattern deployment w/ new Lambda function and overridden props', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`),
            environment: {
                OVERRIDE: "TRUE"
            }
        },
        queueProps: {
            fifo: true
        },
        deployDeadLetterQueue: false,
        maxReceiveCount: 0
    };
    const app = new lib_1.SqsToLambda(stack, 'test-sqs-lambda', props);
    // Assertion 1
    expect(app.lambdaFunction).toHaveProperty('environment.OVERRIDE', 'TRUE');
    // Assertion 2
    expect(app.sqsQueue).toHaveProperty('fifo', true);
});
// --------------------------------------------------------------
// Pattern Deployment w/ Existing Lambda function
// --------------------------------------------------------------
test('Pattern deployment w/ Existing Lambda Function', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const fn = new lambda.Function(stack, 'ExistingLambdaFunction', {
        runtime: lambda.Runtime.NODEJS_10_X,
        handler: 'index.handler',
        code: lambda.Code.asset(`${__dirname}/lambda`)
    });
    const props = {
        existingLambdaObj: fn,
        deployDeadLetterQueue: false,
        maxReceiveCount: 0,
        queueProps: {}
    };
    new lib_1.SqsToLambda(stack, 'test-apigateway-lambda', props);
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
// --------------------------------------------------------------
// Test the getter methods
// --------------------------------------------------------------
test('Test getter methods', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`)
        },
        deployDeadLetterQueue: true,
        maxReceiveCount: 0,
        queueProps: {}
    };
    const app = new lib_1.SqsToLambda(stack, 'test-apigateway-lambda', props);
    // Assertion 1
    expect(app.lambdaFunction !== null);
    // Assertion 2
    expect(app.sqsQueue !== null);
    // Assertion 3
    expect(app.deadLetterQueue !== null);
});
// --------------------------------------------------------------
// Test error handling for existing Lambda function
// --------------------------------------------------------------
test('Test error handling for existing Lambda function', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        existingLambdaObj: undefined,
        deployDeadLetterQueue: false,
        maxReceiveCount: 0,
        queueProps: {}
    };
    // Assertion 1
    expect(() => {
        new lib_1.SqsToLambda(stack, 'test-sqs-lambda', props);
    }).toThrowError();
});
// --------------------------------------------------------------
// Test error handling for new Lambda function
// w/o required properties
// --------------------------------------------------------------
test('Test error handling for new Lambda function w/o required properties', () => {
    // Initial Setup
    const stack = new core_1.Stack();
    const props = {
        deployDeadLetterQueue: false,
        maxReceiveCount: 0,
        queueProps: {}
    };
    // Assertion 1
    expect(() => {
        new lib_1.SqsToLambda(stack, 'test-sqs-lambda', props);
    }).toThrowError();
});
// --------------------------------------------------------------
// Test deployment w/ existing queue
// --------------------------------------------------------------
test('Test deployment w/ existing queue', () => {
    // Stack
    const stack = new core_1.Stack();
    // Helper declaration
    const queue = new sqs.Queue(stack, 'existing-queue-obj', {
        queueName: 'existing-queue-obj'
    });
    new lib_1.SqsToLambda(stack, 'lambda-to-sqs-stack', {
        lambdaFunctionProps: {
            runtime: lambda.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            code: lambda.Code.asset(`${__dirname}/lambda`)
        },
        deployDeadLetterQueue: false,
        existingQueueObj: queue
    });
    // Assertion 1
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGVzdC5zcXMtbGFtYmRhLnRlc3QuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJ0ZXN0LnNxcy1sYW1iZGEudGVzdC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7Ozs7Ozs7Ozs7O0dBV0c7O0FBRUgsVUFBVTtBQUNWLHdDQUFzQztBQUN0QyxnQ0FBdUQ7QUFDdkQsOENBQThDO0FBQzlDLHdDQUF3QztBQUN4Qyw0Q0FBNkM7QUFDN0MsZ0NBQThCO0FBRTlCLGlFQUFpRTtBQUNqRSxnREFBZ0Q7QUFDaEQscUJBQXFCO0FBQ3JCLGlFQUFpRTtBQUNqRSxJQUFJLENBQUMsNkRBQTZELEVBQUUsR0FBRyxFQUFFO0lBQ3JFLGdCQUFnQjtJQUNoQixNQUFNLEtBQUssR0FBRyxJQUFJLFlBQUssRUFBRSxDQUFDO0lBQzFCLE1BQU0sS0FBSyxHQUFxQjtRQUM1QixtQkFBbUIsRUFBRTtZQUNqQixPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxXQUFXO1lBQ25DLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxHQUFHLFNBQVMsU0FBUyxDQUFDO1NBQ2pEO0tBQ0osQ0FBQztJQUNGLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDakQsY0FBYztJQUNkLE1BQU0sQ0FBQyxtQkFBVSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUMsZUFBZSxFQUFFLENBQUM7QUFDakUsQ0FBQyxDQUFDLENBQUM7QUFFSCxpRUFBaUU7QUFDakUsZ0RBQWdEO0FBQ2hELHdCQUF3QjtBQUN4QixpRUFBaUU7QUFDakUsSUFBSSxDQUFDLGdFQUFnRSxFQUFFLEdBQUcsRUFBRTtJQUN4RSxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLEtBQUssR0FBcUI7UUFDNUIsbUJBQW1CLEVBQUU7WUFDakIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztZQUNuQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztZQUM5QyxXQUFXLEVBQUU7Z0JBQ1QsUUFBUSxFQUFFLE1BQU07YUFDbkI7U0FDSjtRQUNELFVBQVUsRUFBRTtZQUNSLElBQUksRUFBRSxJQUFJO1NBQ2I7UUFDRCxxQkFBcUIsRUFBRSxLQUFLO1FBQzVCLGVBQWUsRUFBRSxDQUFDO0tBQ3JCLENBQUM7SUFDRixNQUFNLEdBQUcsR0FBRyxJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQzdELGNBQWM7SUFDZCxNQUFNLENBQUMsR0FBRyxDQUFDLGNBQWMsQ0FBQyxDQUFDLGNBQWMsQ0FBQyxzQkFBc0IsRUFBRSxNQUFNLENBQUMsQ0FBQztJQUMxRSxjQUFjO0lBQ2QsTUFBTSxDQUFDLEdBQUcsQ0FBQyxRQUFRLENBQUMsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDO0FBQ3RELENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLGlEQUFpRDtBQUNqRCxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLGdEQUFnRCxFQUFFLEdBQUcsRUFBRTtJQUN4RCxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLEVBQUUsR0FBRyxJQUFJLE1BQU0sQ0FBQyxRQUFRLENBQUMsS0FBSyxFQUFFLHdCQUF3QixFQUFFO1FBQzVELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7UUFDbkMsT0FBTyxFQUFFLGVBQWU7UUFDeEIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7S0FDakQsQ0FBQyxDQUFDO0lBQ0gsTUFBTSxLQUFLLEdBQXFCO1FBQzVCLGlCQUFpQixFQUFFLEVBQUU7UUFDckIscUJBQXFCLEVBQUUsS0FBSztRQUM1QixlQUFlLEVBQUUsQ0FBQztRQUNsQixVQUFVLEVBQUUsRUFBRTtLQUNqQixDQUFDO0lBQ0YsSUFBSSxpQkFBVyxDQUFDLEtBQUssRUFBRSx3QkFBd0IsRUFBRSxLQUFLLENBQUMsQ0FBQztJQUN4RCxjQUFjO0lBQ2QsTUFBTSxDQUFDLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztBQUNqRSxDQUFDLENBQUMsQ0FBQztBQUVILGlFQUFpRTtBQUNqRSwwQkFBMEI7QUFDMUIsaUVBQWlFO0FBQ2pFLElBQUksQ0FBQyxxQkFBcUIsRUFBRSxHQUFHLEVBQUU7SUFDN0IsZ0JBQWdCO0lBQ2hCLE1BQU0sS0FBSyxHQUFHLElBQUksWUFBSyxFQUFFLENBQUM7SUFDMUIsTUFBTSxLQUFLLEdBQXFCO1FBQzVCLG1CQUFtQixFQUFFO1lBQ2pCLE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDbkMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEdBQUcsU0FBUyxTQUFTLENBQUM7U0FDakQ7UUFDRCxxQkFBcUIsRUFBRSxJQUFJO1FBQzNCLGVBQWUsRUFBRSxDQUFDO1FBQ2xCLFVBQVUsRUFBRSxFQUFFO0tBQ2pCLENBQUM7SUFDRixNQUFNLEdBQUcsR0FBRyxJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLHdCQUF3QixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3BFLGNBQWM7SUFDZCxNQUFNLENBQUMsR0FBRyxDQUFDLGNBQWMsS0FBSyxJQUFJLENBQUMsQ0FBQztJQUNwQyxjQUFjO0lBQ2QsTUFBTSxDQUFDLEdBQUcsQ0FBQyxRQUFRLEtBQUssSUFBSSxDQUFDLENBQUM7SUFDOUIsY0FBYztJQUNkLE1BQU0sQ0FBQyxHQUFHLENBQUMsZUFBZSxLQUFLLElBQUksQ0FBQyxDQUFDO0FBQ3pDLENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLG1EQUFtRDtBQUNuRCxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLGtEQUFrRCxFQUFFLEdBQUcsRUFBRTtJQUMxRCxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLEtBQUssR0FBcUI7UUFDNUIsaUJBQWlCLEVBQUUsU0FBUztRQUM1QixxQkFBcUIsRUFBRSxLQUFLO1FBQzVCLGVBQWUsRUFBRSxDQUFDO1FBQ2xCLFVBQVUsRUFBRSxFQUFFO0tBQ2pCLENBQUM7SUFDRixjQUFjO0lBQ2QsTUFBTSxDQUFDLEdBQUcsRUFBRTtRQUNSLElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUsaUJBQWlCLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDckQsQ0FBQyxDQUFDLENBQUMsWUFBWSxFQUFFLENBQUM7QUFDdEIsQ0FBQyxDQUFDLENBQUM7QUFFSCxpRUFBaUU7QUFDakUsOENBQThDO0FBQzlDLDBCQUEwQjtBQUMxQixpRUFBaUU7QUFDakUsSUFBSSxDQUFDLHFFQUFxRSxFQUFFLEdBQUcsRUFBRTtJQUM3RSxnQkFBZ0I7SUFDaEIsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQixNQUFNLEtBQUssR0FBcUI7UUFDNUIscUJBQXFCLEVBQUUsS0FBSztRQUM1QixlQUFlLEVBQUUsQ0FBQztRQUNsQixVQUFVLEVBQUUsRUFBRTtLQUNqQixDQUFDO0lBQ0YsY0FBYztJQUNkLE1BQU0sQ0FBQyxHQUFHLEVBQUU7UUFDUixJQUFJLGlCQUFXLENBQUMsS0FBSyxFQUFFLGlCQUFpQixFQUFFLEtBQUssQ0FBQyxDQUFDO0lBQ3JELENBQUMsQ0FBQyxDQUFDLFlBQVksRUFBRSxDQUFDO0FBQ3RCLENBQUMsQ0FBQyxDQUFDO0FBRUgsaUVBQWlFO0FBQ2pFLG9DQUFvQztBQUNwQyxpRUFBaUU7QUFDakUsSUFBSSxDQUFDLG1DQUFtQyxFQUFFLEdBQUcsRUFBRTtJQUM3QyxRQUFRO0lBQ1IsTUFBTSxLQUFLLEdBQUcsSUFBSSxZQUFLLEVBQUUsQ0FBQztJQUMxQixxQkFBcUI7SUFDckIsTUFBTSxLQUFLLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSxvQkFBb0IsRUFBRTtRQUN2RCxTQUFTLEVBQUUsb0JBQW9CO0tBQ2hDLENBQUMsQ0FBQztJQUNILElBQUksaUJBQVcsQ0FBQyxLQUFLLEVBQUUscUJBQXFCLEVBQUU7UUFDNUMsbUJBQW1CLEVBQUU7WUFDbkIsT0FBTyxFQUFFLE1BQU0sQ0FBQyxPQUFPLENBQUMsV0FBVztZQUNuQyxPQUFPLEVBQUUsZUFBZTtZQUN4QixJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsR0FBRyxTQUFTLFNBQVMsQ0FBQztTQUMvQztRQUNELHFCQUFxQixFQUFFLEtBQUs7UUFDNUIsZ0JBQWdCLEVBQUUsS0FBSztLQUN4QixDQUFDLENBQUM7SUFDSCxjQUFjO0lBQ2QsTUFBTSxDQUFDLG1CQUFVLENBQUMsZ0JBQWdCLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxlQUFlLEVBQUUsQ0FBQztBQUMvRCxDQUFDLENBQUMsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbIi8qKlxuICogIENvcHlyaWdodCAyMDE5IEFtYXpvbi5jb20sIEluYy4gb3IgaXRzIGFmZmlsaWF0ZXMuIEFsbCBSaWdodHMgUmVzZXJ2ZWQuXG4gKlxuICogIExpY2Vuc2VkIHVuZGVyIHRoZSBBcGFjaGUgTGljZW5zZSwgVmVyc2lvbiAyLjAgKHRoZSBcIkxpY2Vuc2VcIikuIFlvdSBtYXkgbm90IHVzZSB0aGlzIGZpbGUgZXhjZXB0IGluIGNvbXBsaWFuY2VcbiAqICB3aXRoIHRoZSBMaWNlbnNlLiBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgaHR0cDovL3d3dy5hcGFjaGUub3JnL2xpY2Vuc2VzL0xJQ0VOU0UtMi4wXG4gKlxuICogIG9yIGluIHRoZSAnbGljZW5zZScgZmlsZSBhY2NvbXBhbnlpbmcgdGhpcyBmaWxlLiBUaGlzIGZpbGUgaXMgZGlzdHJpYnV0ZWQgb24gYW4gJ0FTIElTJyBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTXG4gKiAgT1IgQ09ORElUSU9OUyBPRiBBTlkgS0lORCwgZXhwcmVzcyBvciBpbXBsaWVkLiBTZWUgdGhlIExpY2Vuc2UgZm9yIHRoZSBzcGVjaWZpYyBsYW5ndWFnZSBnb3Zlcm5pbmcgcGVybWlzc2lvbnNcbiAqICBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuLy8gSW1wb3J0c1xuaW1wb3J0IHsgU3RhY2sgfSBmcm9tIFwiQGF3cy1jZGsvY29yZVwiO1xuaW1wb3J0IHsgU3FzVG9MYW1iZGEsIFNxc1RvTGFtYmRhUHJvcHMgfSBmcm9tIFwiLi4vbGliXCI7XG5pbXBvcnQgKiBhcyBsYW1iZGEgZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSc7XG5pbXBvcnQgKiBhcyBzcXMgZnJvbSAnQGF3cy1jZGsvYXdzLXNxcyc7XG5pbXBvcnQgeyBTeW50aFV0aWxzIH0gZnJvbSAnQGF3cy1jZGsvYXNzZXJ0JztcbmltcG9ydCAnQGF3cy1jZGsvYXNzZXJ0L2plc3QnO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gUGF0dGVybiBkZXBsb3ltZW50IHcvIG5ldyBMYW1iZGEgZnVuY3Rpb24gYW5kXG4vLyBkZWZhdWx0IHByb3BlcnRpZXNcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdQYXR0ZXJuIGRlcGxveW1lbnQgdy8gbmV3IExhbWJkYSBmdW5jdGlvbiBhbmQgZGVmYXVsdCBwcm9wcycsICgpID0+IHtcbiAgICAvLyBJbml0aWFsIFNldHVwXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBwcm9wczogU3FzVG9MYW1iZGFQcm9wcyA9IHtcbiAgICAgICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1gsXG4gICAgICAgICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgICAgICBjb2RlOiBsYW1iZGEuQ29kZS5hc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApXG4gICAgICAgIH1cbiAgICB9O1xuICAgIG5ldyBTcXNUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtc3FzLWxhbWJkYScsIHByb3BzKTtcbiAgICAvLyBBc3NlcnRpb24gMVxuICAgIGV4cGVjdChTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24oc3RhY2spKS50b01hdGNoU25hcHNob3QoKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gUGF0dGVybiBkZXBsb3ltZW50IHcvIG5ldyBMYW1iZGEgZnVuY3Rpb24gYW5kXG4vLyBvdmVycmlkZGVuIHByb3BlcnRpZXNcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdQYXR0ZXJuIGRlcGxveW1lbnQgdy8gbmV3IExhbWJkYSBmdW5jdGlvbiBhbmQgb3ZlcnJpZGRlbiBwcm9wcycsICgpID0+IHtcbiAgICAvLyBJbml0aWFsIFNldHVwXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBwcm9wczogU3FzVG9MYW1iZGFQcm9wcyA9IHtcbiAgICAgICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1gsXG4gICAgICAgICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgICAgICBjb2RlOiBsYW1iZGEuQ29kZS5hc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApLFxuICAgICAgICAgICAgZW52aXJvbm1lbnQ6IHtcbiAgICAgICAgICAgICAgICBPVkVSUklERTogXCJUUlVFXCJcbiAgICAgICAgICAgIH1cbiAgICAgICAgfSxcbiAgICAgICAgcXVldWVQcm9wczoge1xuICAgICAgICAgICAgZmlmbzogdHJ1ZVxuICAgICAgICB9LFxuICAgICAgICBkZXBsb3lEZWFkTGV0dGVyUXVldWU6IGZhbHNlLFxuICAgICAgICBtYXhSZWNlaXZlQ291bnQ6IDBcbiAgICB9O1xuICAgIGNvbnN0IGFwcCA9IG5ldyBTcXNUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtc3FzLWxhbWJkYScsIHByb3BzKTtcbiAgICAvLyBBc3NlcnRpb24gMVxuICAgIGV4cGVjdChhcHAubGFtYmRhRnVuY3Rpb24pLnRvSGF2ZVByb3BlcnR5KCdlbnZpcm9ubWVudC5PVkVSUklERScsICdUUlVFJyk7XG4gICAgLy8gQXNzZXJ0aW9uIDJcbiAgICBleHBlY3QoYXBwLnNxc1F1ZXVlKS50b0hhdmVQcm9wZXJ0eSgnZmlmbycsIHRydWUpO1xufSk7XG5cbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG4vLyBQYXR0ZXJuIERlcGxveW1lbnQgdy8gRXhpc3RpbmcgTGFtYmRhIGZ1bmN0aW9uXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnUGF0dGVybiBkZXBsb3ltZW50IHcvIEV4aXN0aW5nIExhbWJkYSBGdW5jdGlvbicsICgpID0+IHtcbiAgICAvLyBJbml0aWFsIFNldHVwXG4gICAgY29uc3Qgc3RhY2sgPSBuZXcgU3RhY2soKTtcbiAgICBjb25zdCBmbiA9IG5ldyBsYW1iZGEuRnVuY3Rpb24oc3RhY2ssICdFeGlzdGluZ0xhbWJkYUZ1bmN0aW9uJywge1xuICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgICBjb2RlOiBsYW1iZGEuQ29kZS5hc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApXG4gICAgfSk7XG4gICAgY29uc3QgcHJvcHM6IFNxc1RvTGFtYmRhUHJvcHMgPSB7XG4gICAgICAgIGV4aXN0aW5nTGFtYmRhT2JqOiBmbixcbiAgICAgICAgZGVwbG95RGVhZExldHRlclF1ZXVlOiBmYWxzZSxcbiAgICAgICAgbWF4UmVjZWl2ZUNvdW50OiAwLFxuICAgICAgICBxdWV1ZVByb3BzOiB7fVxuICAgIH07XG4gICAgbmV3IFNxc1RvTGFtYmRhKHN0YWNrLCAndGVzdC1hcGlnYXRld2F5LWxhbWJkYScsIHByb3BzKTtcbiAgICAvLyBBc3NlcnRpb24gMVxuICAgIGV4cGVjdChTeW50aFV0aWxzLnRvQ2xvdWRGb3JtYXRpb24oc3RhY2spKS50b01hdGNoU25hcHNob3QoKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCB0aGUgZ2V0dGVyIG1ldGhvZHNcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGdldHRlciBtZXRob2RzJywgKCkgPT4ge1xuICAgIC8vIEluaXRpYWwgU2V0dXBcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHByb3BzOiBTcXNUb0xhbWJkYVByb3BzID0ge1xuICAgICAgICBsYW1iZGFGdW5jdGlvblByb3BzOiB7XG4gICAgICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5OT0RFSlNfMTBfWCxcbiAgICAgICAgICAgIGhhbmRsZXI6ICdpbmRleC5oYW5kbGVyJyxcbiAgICAgICAgICAgIGNvZGU6IGxhbWJkYS5Db2RlLmFzc2V0KGAke19fZGlybmFtZX0vbGFtYmRhYClcbiAgICAgICAgfSxcbiAgICAgICAgZGVwbG95RGVhZExldHRlclF1ZXVlOiB0cnVlLFxuICAgICAgICBtYXhSZWNlaXZlQ291bnQ6IDAsXG4gICAgICAgIHF1ZXVlUHJvcHM6IHt9XG4gICAgfTtcbiAgICBjb25zdCBhcHAgPSBuZXcgU3FzVG9MYW1iZGEoc3RhY2ssICd0ZXN0LWFwaWdhdGV3YXktbGFtYmRhJywgcHJvcHMpO1xuICAgIC8vIEFzc2VydGlvbiAxXG4gICAgZXhwZWN0KGFwcC5sYW1iZGFGdW5jdGlvbiAhPT0gbnVsbCk7XG4gICAgLy8gQXNzZXJ0aW9uIDJcbiAgICBleHBlY3QoYXBwLnNxc1F1ZXVlICE9PSBudWxsKTtcbiAgICAvLyBBc3NlcnRpb24gM1xuICAgIGV4cGVjdChhcHAuZGVhZExldHRlclF1ZXVlICE9PSBudWxsKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBlcnJvciBoYW5kbGluZyBmb3IgZXhpc3RpbmcgTGFtYmRhIGZ1bmN0aW9uXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnVGVzdCBlcnJvciBoYW5kbGluZyBmb3IgZXhpc3RpbmcgTGFtYmRhIGZ1bmN0aW9uJywgKCkgPT4ge1xuICAgIC8vIEluaXRpYWwgU2V0dXBcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHByb3BzOiBTcXNUb0xhbWJkYVByb3BzID0ge1xuICAgICAgICBleGlzdGluZ0xhbWJkYU9iajogdW5kZWZpbmVkLFxuICAgICAgICBkZXBsb3lEZWFkTGV0dGVyUXVldWU6IGZhbHNlLFxuICAgICAgICBtYXhSZWNlaXZlQ291bnQ6IDAsXG4gICAgICAgIHF1ZXVlUHJvcHM6IHt9XG4gICAgfTtcbiAgICAvLyBBc3NlcnRpb24gMVxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICAgIG5ldyBTcXNUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtc3FzLWxhbWJkYScsIHByb3BzKTtcbiAgICB9KS50b1Rocm93RXJyb3IoKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBlcnJvciBoYW5kbGluZyBmb3IgbmV3IExhbWJkYSBmdW5jdGlvblxuLy8gdy9vIHJlcXVpcmVkIHByb3BlcnRpZXNcbi8vIC0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tXG50ZXN0KCdUZXN0IGVycm9yIGhhbmRsaW5nIGZvciBuZXcgTGFtYmRhIGZ1bmN0aW9uIHcvbyByZXF1aXJlZCBwcm9wZXJ0aWVzJywgKCkgPT4ge1xuICAgIC8vIEluaXRpYWwgU2V0dXBcbiAgICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAgIGNvbnN0IHByb3BzOiBTcXNUb0xhbWJkYVByb3BzID0ge1xuICAgICAgICBkZXBsb3lEZWFkTGV0dGVyUXVldWU6IGZhbHNlLFxuICAgICAgICBtYXhSZWNlaXZlQ291bnQ6IDAsXG4gICAgICAgIHF1ZXVlUHJvcHM6IHt9XG4gICAgfTtcbiAgICAvLyBBc3NlcnRpb24gMVxuICAgIGV4cGVjdCgoKSA9PiB7XG4gICAgICAgIG5ldyBTcXNUb0xhbWJkYShzdGFjaywgJ3Rlc3Qtc3FzLWxhbWJkYScsIHByb3BzKTtcbiAgICB9KS50b1Rocm93RXJyb3IoKTtcbn0pO1xuXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxuLy8gVGVzdCBkZXBsb3ltZW50IHcvIGV4aXN0aW5nIHF1ZXVlXG4vLyAtLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS0tLVxudGVzdCgnVGVzdCBkZXBsb3ltZW50IHcvIGV4aXN0aW5nIHF1ZXVlJywgKCkgPT4ge1xuICAvLyBTdGFja1xuICBjb25zdCBzdGFjayA9IG5ldyBTdGFjaygpO1xuICAvLyBIZWxwZXIgZGVjbGFyYXRpb25cbiAgY29uc3QgcXVldWUgPSBuZXcgc3FzLlF1ZXVlKHN0YWNrLCAnZXhpc3RpbmctcXVldWUtb2JqJywge1xuICAgIHF1ZXVlTmFtZTogJ2V4aXN0aW5nLXF1ZXVlLW9iaidcbiAgfSk7XG4gIG5ldyBTcXNUb0xhbWJkYShzdGFjaywgJ2xhbWJkYS10by1zcXMtc3RhY2snLCB7XG4gICAgbGFtYmRhRnVuY3Rpb25Qcm9wczoge1xuICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEwX1gsXG4gICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICBjb2RlOiBsYW1iZGEuQ29kZS5hc3NldChgJHtfX2Rpcm5hbWV9L2xhbWJkYWApXG4gICAgfSxcbiAgICBkZXBsb3lEZWFkTGV0dGVyUXVldWU6IGZhbHNlLFxuICAgIGV4aXN0aW5nUXVldWVPYmo6IHF1ZXVlXG4gIH0pO1xuICAvLyBBc3NlcnRpb24gMVxuICBleHBlY3QoU3ludGhVdGlscy50b0Nsb3VkRm9ybWF0aW9uKHN0YWNrKSkudG9NYXRjaFNuYXBzaG90KCk7XG59KTsiXX0=