import io
import shutil
from typing import List

import requests
import typer

try:
    from typer import Typer, Argument, Option
    from rich.console import Console
    from rich.status import Status
    from rich.progress import track, Progress
except ImportError:
    print("Please install malwarebazaar[cli] if you want to use the cli functionality.")
    raise

from enum import Enum
from json import dumps

from malwarebazaar.api import Bazaar
from malwarebazaar.config import Config
from malwarebazaar.models import Sample
from malwarebazaar.output import print_sample_table, print_vendor_info_table, sample_csv_output, simple_sample_output
from malwarebazaar.util import check_version


class QueryTypes(Enum):
    clamav = "clamav"
    filetype = "filetype"
    gimphash = "gimphash"
    hash = "hash"
    icon = "icon"
    imphash = "imphash"
    issuer = "issuer"
    signature = "signature"
    subject = "subject"
    tag = "tag"
    telfhash = "telfhash"
    tlsh = "tlsh"
    yara = "yara"


def complete_query_type(incomplete: str):
    for qtype in QueryTypes.__dict__["_member_names_"]:
        if qtype.startswith(incomplete):
            yield qtype


bazaar_app = Typer(name="MalwareBazaar cli", help="Query MalwareBazaar from the command line!")


@bazaar_app.command(name="init", help="Initialize bazaar config file.")
def init(
        yaraify_key: str = Option(None, "-y", "--yaraify", help="Optional API key from YARAify."),
        malpedia_key: str = Option(None, "-m", "--malpedia", help="Optional API key from Malpedia."),
        api_key: str = Argument(..., help="API key from your MalwareBazaar account.")
):
    c = Console()
    success = Config.init_config(
        api_key,
        yaraify_key,
        malpedia_key
    )
    if success:
        c.print("Successfully set API-Key!")


@bazaar_app.command(name="query", help="Query the MalwareBazaar API.")
def query(
        limit: int = Option(25, "-l", "--limit", help="Limit the amount of objects returned."),
        json: bool = Option(False, "-j", "--json", help="Output raw JSON response."),
        csv: bool = Option(False, "-c", "--csv", help="Output csv."),
        download: bool = Option(False, "-d", "--download", help="Download samples retrieved by query."),
        simple: bool = Option(False, "-s", "--simple", help="Just output SHA256 hashes."),
        query_type: QueryTypes = Argument(..., show_choices=True, help="The type of query to send to the API.",
                                          autocompletion=complete_query_type),
        query: List[str] = Argument(..., help="The search term to use for the query")
):
    c, ec = Console(), Console(stderr=True, style="bold red")
    bazaar = Bazaar(Config.get_instance().api_key)

    try:
        with Status("Querying MalwareBazaar..."):
            results = []
            for q in query:
                if query_type == QueryTypes.hash:
                    results.append(bazaar.query_hash(q))
                elif query_type == QueryTypes.imphash:
                    results.append(bazaar.query_imphash(q, limit=limit))
                elif query_type == QueryTypes.signature:
                    results.append(bazaar.query_signature(q, limit=limit))
                elif query_type == QueryTypes.yara:
                    results.append(bazaar.query_yara(q, limit=limit))
                elif query_type == QueryTypes.filetype:
                    results.append(bazaar.query_filetype(q, limit=limit))
                elif query_type == QueryTypes.clamav:
                    results.append(bazaar.query_clamav_signature(q, limit=limit))
                elif query_type == QueryTypes.tag:
                    results.append(bazaar.query_tag(q, limit=limit))
                elif query_type == QueryTypes.issuer:
                    results.append(bazaar.query_signing_issuer(q))
                elif query_type == QueryTypes.subject:
                    results.append(bazaar.query_signing_subject(q))
                elif query_type == QueryTypes.tlsh:
                    results.append(bazaar.query_tlsh(q, limit=limit))
                elif query_type == QueryTypes.telfhash:
                    results.append(bazaar.query_telfhash(q, limit=limit))
                elif query_type == QueryTypes.gimphash:
                    results.append(bazaar.query_gimphash(q, limit=limit))
                elif query_type == QueryTypes.icon:
                    results.append(bazaar.query_icon_dhash(q, limit=limit))
    except (requests.ConnectionError, requests.ConnectTimeout):
        ec.print("[bold red]Could not connect to MalwareBazaar API.[/bold red]")
        raise typer.Exit(-1)

    if json or csv:
        data = []
        for result in results:
            r = result.get("data", None)
            if r:
                data.append(r)
        if json:
            c.print(dumps(data))
        else:
            samples = []
            for d in data:
                if isinstance(d, List):
                    samples.extend(d)
                else:
                    samples.append(d)
            sample_csv_output([Sample(**s) for s in samples])
        raise typer.Exit(0)

    tags = {}
    for idx, res in enumerate(results):
        if res["query_status"] != "ok":
            ec.print(f"Bazaar API returned an error for {query[idx]}: {res['query_status']}")
            continue

        samples = [Sample(**sample) for sample in res["data"]]
        for sample in samples:
            if simple:
                simple_sample_output(sample, tags, c)
            else:
                print_sample_table(sample, c)
                if sample.vendor_intel:
                    print_vendor_info_table(sample, c)

        if download:
            for sample in track(samples, description="Downloading samples..."):
                content = bazaar.download_file(sample.sha256_hash)
                with io.open(sample.sha256_hash, "wb") as fh:
                    fh.write(content)


@bazaar_app.command(
    name="recent",
    help="Get information about recently submitted samples. The API allows either the last 100 samples or "
         "samples uploaded in the last 60 minutes. As the amount is quite big, the default output type is "
         "csv."
)
def recent(
        number: bool = Option(False, "-n", "--number", help="Query last 100 samples instead of samples submitted in "
                                                            "the last hour."),
        csv: bool = Option(False, "-c", "--csv", help="CSV output."),
        simple: bool = Option(False, "-s", "--simple", help="Simple output."),
        limit: int = Option(None, "-l", "--limit", help="Limit the number of samples printed. This has no effect on the"
                                                        " MalwareBazaar API call itself.")
):
    c, ec = Console(), Console(stderr=True, style="bold red")
    bazaar = Bazaar(Config.get_instance().api_key)
    try:
        with Status(f"Querying MalwareBazaar..."):
            if number:
                res = bazaar.query_recent(last_100_samples=True)
            else:
                res = bazaar.query_recent()
    except (requests.ConnectionError, requests.ConnectTimeout):
        ec.print("[bold red]Could not connect to MalwareBazaar API.[/bold red]")
        raise typer.Exit(-1)

    if res["query_status"] != "ok":
        ec.print(f"Invalid Bazaar response: {res['query_status']}")
        exit(-1)

    samples = [Sample(**sample) for sample in res["data"]]
    if limit:
        samples = samples[:limit]
    if csv:
        sample_csv_output(samples)
    elif simple:
        tags = {}
        for sample in samples:
            simple_sample_output(sample, tags, c)
    else:
        for idx, sample in enumerate(samples):
            c.print(f"Sample {idx + 1}/{len(samples)}")
            print_sample_table(sample, c)


@bazaar_app.command(
    name="batch",
    help="Download daily malware batches. The DATE_STR argument needs to be in the format of YYYY-mm-dd."
)
def batch(
        output: str = Option(None, "-o", "--output", help="Output file name."),
        quiet: bool = Option(False, "-q", "--quiet",
                             help="Do not display any output. This also uses shutil to write the "
                                  "downloaded chunks direct to the file which *might* speed up your "
                                  "download."),
        hourly: bool = Option(False, "-H", "--hourly",
                              help="Download hourly batches. For hourly batches, the [DATE_STR] needs "
                                   "to be YYYY-mm-dd-HH"),
        chunk_size: int = Option(1024 ** 2, "-c", "--chunk-size", help="Sets chunk size of downloaded chunks "
                                                                       f"(default is {1024 ** 2}). "
                                                                       f"This can help speed-up your downloads."),
        date_str: str = Argument(..., help="Date string used to download the specific malware batch file.")
):
    ec = Console(stderr=True, style="bold red")
    if not hourly:
        url = f"https://datalake.abuse.ch/malware-bazaar/daily/{date_str}.zip"
    else:
        url = f"https://datalake.abuse.ch/malware-bazaar/hourly/{date_str}.zip"
    head = requests.head(url)
    if head.status_code != 200:
        ec.print(f"No batch file for given date found. "
                 f"Maybe you used the wrong date format? (Tried {head.request.url})")
        exit(-1)

    total = int(head.headers.get("Content-Length", 0))
    filename = output or f"{date_str}.zip"
    if quiet:
        with requests.get(url, stream=True) as r:
            with io.open(filename, "wb") as file:
                shutil.copyfileobj(r.raw, file, length=chunk_size)
    else:
        with Progress() as progress:
            progress.print(f"Downloading {date_str}.zip with a size of {total / 1024 ** 2:.2f}MB.")
            task = progress.add_task(f"Downloading {filename}...", total=total)
            with requests.get(url, stream=True) as r:
                with io.open(filename, "wb") as file:
                    for data_chunk in r.iter_content(chunk_size=chunk_size):
                        num_bytes = file.write(data_chunk)
                        progress.update(task, advance=num_bytes)


@bazaar_app.command(name="version", help="Print and check bazaar version.")
def version(
        check: bool = Option(False, "-c", "--check", help="Check if you're using the latest version via Github API.")
):
    check_version(check)


if __name__ == "__main__":
    bazaar_app()
