"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const console = require("console");
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const aws_xray_sdk_core_1 = require("aws-xray-sdk-core");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const couch_changes_lambda_shared_1 = require("./couch-changes.lambda-shared");
// eslint-disable-next-line @typescript-eslint/no-require-imports
const normalizeNPMMetadata = require('normalize-registry-metadata');
const CONSTRUCT_KEYWORDS = new Set(['cdk', 'aws-cdk', 'awscdk', 'cdk8s', 'cdktf']);
const NPM_REPLICA_REGISTRY_URL = 'https://replicate.npmjs.com/';
/**
 * The release date of `aws-cdk@0.8.0`. Anything earlier than this basically is
 * not a relevant package, as it cannot possibly be a constructs-based package.
 * This is used to fast-forward over boring stuff when the sequence number is
 * reset.
 */
const DAWN_OF_CONSTRUCTS = new Date('2018-07-31T13:43:04.615Z');
// Configure embedded metrics format
aws_embedded_metrics_1.Configuration.namespace = constants_lambda_shared_1.METRICS_NAMESPACE;
// Make sure X-Ray traces will include HTTP(s) calls.
// eslint-disable-next-line @typescript-eslint/no-require-imports
aws_xray_sdk_core_1.captureHTTPsGlobal(require('https'));
// eslint-disable-next-line @typescript-eslint/no-require-imports
aws_xray_sdk_core_1.captureHTTPsGlobal(require('http'));
/**
 * This function triggers on a fixed schedule and reads a stream of changes from npmjs couchdb _changes endpoint.
 * Upon invocation the function starts reading from a sequence stored in an s3 object - the `marker`.
 * If the marker fails to load (or do not exist), the stream will start from `now` - the latest change.
 * For each change:
 *  - the package version tarball will be copied from the npm registry to a stating bucket.
 *  - a message will be sent to an sqs queue
 * npm registry API docs: https://github.com/npm/registry/blob/master/docs/REGISTRY-API.md
 * @param context a Lambda execution context
 */
async function handler(event, context) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const stagingFunction = env_lambda_shared_1.requireEnv('FUNCTION_NAME');
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const npm = new couch_changes_lambda_shared_1.CouchChanges(NPM_REPLICA_REGISTRY_URL, 'registry');
    const { marker: initialMarker, knownVersions } = await loadLastTransactionMarker(stagingBucket, npm);
    // The last written marker seq id.
    let updatedMarker = initialMarker;
    // The slowest batch processing time so far (starts at 30 seconds). This is how much time should
    // be left before timeout if a new batch is to be fetched.
    let maxBatchProcessingTime = 30000;
    // Whether we should continue reading more items or not... This is set to false when the current
    // latest change is reached (i.e: next page of changes is empty).
    let shouldContinue = true;
    do {
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            var _a;
            const changes = await npm.changes(updatedMarker);
            // Clear automatically set dimensions - we don't need them (see https://github.com/awslabs/aws-embedded-metrics-node/issues/73)
            metrics.setDimensions();
            // Recording current seq range and updating the `updatedMarker`.
            metrics.setProperty('StartSeq', updatedMarker);
            updatedMarker = changes.last_seq;
            metrics.setProperty('EndSeq', updatedMarker);
            const startTime = Date.now();
            try {
                const batch = changes.results;
                // The most recent "modified" timestamp observed in the batch.
                let lastModified;
                // Emit npm.js replication lag
                for (const { doc } of batch) {
                    if ((_a = doc === null || doc === void 0 ? void 0 : doc.time) === null || _a === void 0 ? void 0 : _a.modified) {
                        const modified = new Date(doc.time.modified);
                        metrics.putMetric("NpmJsChangeAge" /* NPMJS_CHANGE_AGE */, startTime - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                        if (lastModified == null || lastModified < modified) {
                            lastModified = modified;
                        }
                    }
                }
                console.log(`Received a batch of ${batch.length} element(s)`);
                metrics.putMetric("ChangeCount" /* CHANGE_COUNT */, batch.length, aws_embedded_metrics_1.Unit.Count);
                if (lastModified && lastModified < DAWN_OF_CONSTRUCTS) {
                    console.log(`Skipping batch as the latest modification is ${lastModified}, which is pre-Constructs`);
                }
                else if (batch.length === 0) {
                    console.log('Received 0 changes, caught up to "now", exiting...');
                    shouldContinue = false;
                }
                else {
                    // Obtain the modified package version from the update event, and filter
                    // out packages that are not of interest to us (not construct libraries).
                    const versionInfos = getRelevantVersionInfos(batch, metrics, denyList, licenseList, knownVersions);
                    console.log(`Identified ${versionInfos.length} relevant package version update(s)`);
                    metrics.putMetric("RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */, versionInfos.length, aws_embedded_metrics_1.Unit.Count);
                    // Process all remaining updates
                    await Promise.all(versionInfos.map(async ({ infos, modified, seq }) => {
                        const invokeArgs = {
                            integrity: infos.dist.shasum,
                            modified: modified.toISOString(),
                            name: infos.name,
                            seq: seq === null || seq === void 0 ? void 0 : seq.toString(),
                            tarballUrl: infos.dist.tarball,
                            version: infos.version,
                        };
                        // "Fire-and-forget" invocation here.
                        await aws.lambda().invokeAsync({
                            FunctionName: stagingFunction,
                            InvokeArgs: JSON.stringify(invokeArgs, null, 2),
                        }).promise();
                        // Record that this is now a "known" version (no need to re-discover)
                        knownVersions.set(`${infos.name}@${infos.version}`, modified);
                    }));
                }
                // Updating the S3 stored marker with the new seq id as communicated by nano.
                await saveLastTransactionMarker(context, stagingBucket, updatedMarker, knownVersions);
            }
            finally {
                // Markers may not always be numeric (but in practice they are now), so we protect against that...
                if (typeof updatedMarker === 'number' || /^\d+$/.test(updatedMarker)) {
                    metrics.putMetric("LastSeq" /* LAST_SEQ */, typeof updatedMarker === 'number' ? updatedMarker : parseInt(updatedMarker), aws_embedded_metrics_1.Unit.None);
                }
                metrics.putMetric("BatchProcessingTime" /* BATCH_PROCESSING_TIME */, Date.now() - startTime, aws_embedded_metrics_1.Unit.Milliseconds);
                metrics.putMetric("RemainingTime" /* REMAINING_TIME */, context.getRemainingTimeInMillis(), aws_embedded_metrics_1.Unit.Milliseconds);
            }
        })();
    } while (shouldContinue && context.getRemainingTimeInMillis() >= maxBatchProcessingTime);
    console.log('All done here, we have success!');
    return { initialMarker, updatedMarker };
}
exports.handler = handler;
//#region Last transaction marker
/**
 * Loads the last transaction marker from S3.
 *
 * @param registry a Nano database corresponding to the Npmjs.com CouchDB instance.
 *
 * @returns the value of the last transaction marker and the map of package names + versions to the last modification
 *          of that package version that was processed.
 */
async function loadLastTransactionMarker(stagingBucket, registry) {
    try {
        const response = await aws.s3().getObject({
            Bucket: stagingBucket,
            Key: constants_lambda_shared_1.MARKER_FILE_NAME,
        }).promise();
        if (response.ContentEncoding === 'gzip') {
            response.Body = zlib_1.gunzipSync(Buffer.from(response.Body));
        }
        let data = JSON.parse(response.Body.toString('utf-8'), (key, value) => {
            if (key !== 'knownVersions') {
                return value;
            }
            const map = new Map();
            for (const [pkgVersion, iso] of Object.entries(value)) {
                if (typeof iso === 'string' || typeof iso === 'number') {
                    map.set(pkgVersion, new Date(iso));
                }
                else {
                    console.error(`Ignoring invalid entry: ${pkgVersion} => ${iso}`);
                }
            }
            return map;
        });
        if (typeof data === 'number') {
            data = { marker: data.toFixed(), knownVersions: new Map() };
        }
        console.log(`Read last transaction marker: ${data.marker}`);
        const dbUpdateSeq = (await registry.info()).update_seq;
        if (dbUpdateSeq < data.marker) {
            console.warn(`Current DB update_seq (${dbUpdateSeq}) is lower than marker (CouchDB instance was likely replaced), resetting to 0!`);
            return { marker: '0', knownVersions: data.knownVersion };
        }
        return data;
    }
    catch (error) {
        if (error.code !== 'NoSuchKey') {
            throw error;
        }
        console.warn(`Marker object (s3://${stagingBucket}/${constants_lambda_shared_1.MARKER_FILE_NAME}) does not exist, starting from scratch`);
        return { marker: '0', knownVersions: new Map() };
    }
}
/**
 * Updates the last transaction marker in S3.
 *
 * @param marker the last transaction marker value
 * @param knownVersions the map of package name + version to last modified timestamp of packages that have been processed.
 */
async function saveLastTransactionMarker(context, stagingBucket, marker, knownVersions) {
    console.log(`Updating last transaction marker to ${marker}`);
    return putObject(context, stagingBucket, constants_lambda_shared_1.MARKER_FILE_NAME, zlib_1.gzipSync(JSON.stringify({ marker, knownVersions }, (_, value) => {
        if (value instanceof Date) {
            return value.toISOString();
        }
        else if (value instanceof Map) {
            return Object.fromEntries(value);
        }
        else {
            return value;
        }
    }, 2), { level: 9 }), {
        ContentType: 'application/json',
        ContentEncoding: 'gzip',
    });
}
//#endregion
//#region Asynchronous Primitives
/**
 * Puts an object in the staging bucket, with standardized object metadata.
 *
 * @param key  the key for the object to be put.
 * @param body the body of the object to be put.
 * @param opts any other options to use when sending the S3 request.
 *
 * @returns the result of the S3 request.
 */
function putObject(context, bucket, key, body, opts = {}) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            ...opts.Metadata,
        },
        ...opts,
    }).promise();
}
//#endregion
/**
 * Obtains the `VersionInfo` corresponding to the modified version(s) in the
 * provided `Change` objects, ensures they are relevant (construct libraries),
 * and returns those only.
 *
 * @param changes the changes to be processed.
 * @param metrics the metrics logger to use.
 * @param denyList deny list client
 *
 * @returns a list of `VersionInfo` objects
 */
function getRelevantVersionInfos(changes, metrics, denyList, licenseList, knownVersions) {
    var _a, _b;
    const result = new Array();
    for (const change of changes) {
        // Filter out all elements that don't have a "name" in the document, as
        // these are schemas, which are not relevant to our business here.
        if (change.doc.name === undefined) {
            console.error(`[${change.seq}] Changed document contains no 'name': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // The normalize function change the object in place, if the doc object is invalid it will return undefined
        if (normalizeNPMMetadata(change.doc) === undefined) {
            console.error(`[${change.seq}] Changed document invalid, npm normalize returned undefined: ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there are no versions in the document. We skip those.
        if (change.doc.versions == null) {
            console.error(`[${change.seq}] Changed document contains no 'versions': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Sometimes, there is no 'time' entry in the document. We skip those.
        if (change.doc.time == null) {
            console.error(`[${change.seq}] Changed document contains no 'time': ${change.id}`);
            metrics.putMetric("UnprocessableEntity" /* UNPROCESSABLE_ENTITY */, 1, aws_embedded_metrics_1.Unit.Count);
            continue;
        }
        // Get the last modification date from the change
        const packageVersionUpdates = Object.entries(change.doc.time)
            // Ignore the "created" and "modified" keys here
            .filter(([key]) => key !== 'created' && key !== 'modified')
            // Parse all the dates to ensure they are comparable
            .map(([version, isoDate]) => [version, new Date(isoDate)]);
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, packageVersionUpdates.length, aws_embedded_metrics_1.Unit.Count);
        for (const [version, modified] of packageVersionUpdates) {
            const knownKey = `${change.doc.name}@${version}`;
            const known = knownVersions.get(knownKey);
            if (known == null || known < modified) {
                const infos = change.doc.versions[version];
                if (infos == null) {
                    // Could be the version in question was un-published.
                    console.log(`[${change.seq}] Could not find info for "${change.doc.name}@${version}". Was it un-published?`);
                }
                else if (isConstructLibrary(infos)) {
                    // skip if this package is denied
                    const denied = denyList.lookup(infos.name, infos.version);
                    if (denied) {
                        console.log(`[${change.seq}] Package denied: ${JSON.stringify(denied)}`);
                        knownVersions.set(knownKey, modified);
                        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
                        continue;
                    }
                    metrics.putMetric("PackageVersionAge" /* PACKAGE_VERSION_AGE */, Date.now() - modified.getTime(), aws_embedded_metrics_1.Unit.Milliseconds);
                    const isEligible = licenseList.lookup((_a = infos.license) !== null && _a !== void 0 ? _a : 'UNLICENSED') != null;
                    metrics.putMetric("IneligibleLicense" /* INELIGIBLE_LICENSE */, isEligible ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
                    if (isEligible) {
                        result.push({ infos, modified, seq: change.seq });
                    }
                    else {
                        console.log(`[${change.seq}] Package "${change.doc.name}@${version}" does not use allow-listed license: ${(_b = infos.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
                        knownVersions.set(knownKey, modified);
                    }
                }
                // Else this is not a construct library, so we'll just ignore it...
            }
        }
    }
    return result;
    /**
     * This determines whether a package is "interesting" to ConstructHub or not. This is related but
     * not necessarily identical to the logic in the ingestion process that annotates package metadata
     * with a construct framework name + version (those could ultimately be re-factored to share more
     * of the logic/heuristics, though).
     *
     * Concretely, it checks for a list of known "official" packages for various construct frameworks,
     * and packages that have a dependency on such a package. It also has a keywords allow-list as a
     * fall-back (the current dependency-based logic does not consider transitive dependencies and
     * might hence miss certain rare use-cases, which keywords would rescue).
     */
    function isConstructLibrary(infos) {
        var _a, _b, _c, _d;
        if (infos.jsii == null) {
            return false;
        }
        // The "constructs" package is a sign of a constructs library
        return isConstructFrameworkPackage(infos.name)
            // Recursively apply on dependencies
            || Object.keys((_a = infos.dependencies) !== null && _a !== void 0 ? _a : {}).some(isConstructFrameworkPackage)
            || Object.keys((_b = infos.devDependencies) !== null && _b !== void 0 ? _b : {}).some(isConstructFrameworkPackage)
            || Object.keys((_c = infos.peerDependencies) !== null && _c !== void 0 ? _c : {}).some(isConstructFrameworkPackage)
            // Keyword-based fallback
            || ((_d = infos.keywords) === null || _d === void 0 ? void 0 : _d.some((kw) => CONSTRUCT_KEYWORDS.has(kw)));
    }
    /**
     * Package is one of the known construct framework's first party packages:
     * - @aws-cdk/*
     * - @cdktf/*
     * - cdk8s or cdk8s-plus
     */
    function isConstructFrameworkPackage(name) {
        // IMPORTANT NOTE: Prefix matching should only be used for @scope/ names.
        // The low-level constructs package
        return name === 'constructs'
            // AWS CDK Packages
            || name === 'aws-cdk-lib'
            || name === 'monocdk'
            || name.startsWith('@aws-cdk/')
            // CDK8s packages
            || name === 'cdk8s'
            || /^cdk8s-plus(?:-(?:17|20|21|22))?$/.test(name)
            // CDKTf packages
            || name === 'cdktf'
            || name.startsWith('@cdktf/');
    }
}
//# sourceMappingURL=data:application/json;base64,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