"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const client_lambda_shared_1 = require("../../backend/deny-list/client.lambda-shared");
const client_lambda_shared_2 = require("../../backend/license-list/client.lambda-shared");
const aws = require("../../backend/shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../../backend/shared/env.lambda-shared");
const integrity_lambda_shared_1 = require("../../backend/shared/integrity.lambda-shared");
const tarball_lambda_shared_1 = require("../../backend/shared/tarball.lambda-shared");
const constants_lambda_shared_1 = require("./constants.lambda-shared");
const DETAIL_TYPE = 'CodeArtifact Package Version State Change';
exports.handler = aws_embedded_metrics_1.metricScope((metrics) => async (event, context) => {
    var _a, _b;
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    const stagingBucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const queueUrl = env_lambda_shared_1.requireEnv('QUEUE_URL');
    metrics.setNamespace(constants_lambda_shared_1.METRICS_NAMESPACE);
    metrics.setDimensions({
        [constants_lambda_shared_1.DOMAIN_OWNER_DIMENSION]: event.detail.domainOwner,
        [constants_lambda_shared_1.DOMAIN_NAME_DIMENSION]: event.detail.domainName,
        [constants_lambda_shared_1.REPOSITORY_NAME_DIMENSION]: event.detail.repositoryName,
    });
    const packageName = event.detail.packageNamespace
        ? `@${event.detail.packageNamespace}/${event.detail.packageName}`
        : event.detail.packageName;
    if (event.detail.operationType === 'Deleted') {
        console.log(`Operation type is "Deleted" for ${packageName}@${event.detail.packageVersion}`);
        metrics.putMetric("DeletedCount" /* DELETED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
        return;
    }
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    const denied = denyList.lookup(packageName, event.detail.packageVersion);
    if (denied) {
        console.log(`Package ${packageName}@${event.detail.packageVersion} denied: ${JSON.stringify(denied, null, 2)}`);
        metrics.putMetric("DenyListedCount" /* DENY_LISTED_COUNT */, 1, aws_embedded_metrics_1.Unit.Count);
        return;
    }
    const { asset, packageVersionRevision } = await aws.codeArtifact().getPackageVersionAsset({
        asset: 'package.tgz',
        domainOwner: event.detail.domainOwner,
        domain: event.detail.domainName,
        repository: event.detail.repositoryName,
        format: event.detail.packageFormat,
        namespace: event.detail.packageNamespace,
        package: event.detail.packageName,
        packageVersion: event.detail.packageVersion,
    }).promise();
    const tarball = Buffer.from(asset);
    const { assemblyJson, packageJson } = await tarball_lambda_shared_1.extractObjects(tarball, {
        assemblyJson: { path: 'package/.jsii' },
        packageJson: { path: 'package/package.json', required: true },
    });
    metrics.putMetric("NotJsiiEnabledCount" /* NOT_JSII_ENABLED_COUNT */, assemblyJson ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
    if (assemblyJson == null) {
        console.log(`Package "${packageName}@${event.detail.packageVersion}" does not contain a .jsii assembly`);
        return;
    }
    const metadata = JSON.parse(packageJson.toString('utf-8'));
    const licenseList = await client_lambda_shared_2.LicenseListClient.newClient();
    const eligibleLicense = licenseList.lookup((_a = metadata.license) !== null && _a !== void 0 ? _a : 'UNLICENSED');
    metrics.putMetric("IneligibleLicenseCount" /* INELIGIBLE_LICENSE */, eligibleLicense ? 0 : 1, aws_embedded_metrics_1.Unit.Count);
    if (!eligibleLicense) {
        console.log(`Package "${packageName}@${event.detail.packageVersion}" does not use allow-listed license: ${(_b = metadata.license) !== null && _b !== void 0 ? _b : 'UNLICENSED'}`);
        return;
    }
    const stagingKey = `${packageName}/${event.detail.packageVersion}/${packageVersionRevision}/package.tgz`;
    await aws.s3().putObject({
        Bucket: stagingBucket,
        Key: stagingKey,
        Body: asset,
        ContentType: 'application/octet-stream',
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
        },
    }).promise();
    const messageBase = {
        tarballUri: `s3://${stagingBucket}/${stagingKey}`,
        metadata: { resources: event.resources.join(', ') },
        time: event.time,
    };
    const message = {
        ...messageBase,
        integrity: integrity_lambda_shared_1.integrity(messageBase, tarball),
    };
    return aws.sqs().sendMessage({
        MessageAttributes: {
            AWS_REQUEST_ID: { DataType: 'String', StringValue: context.awsRequestId },
            LOG_GROUP_NAME: { DataType: 'String', StringValue: context.logGroupName },
            LOG_STREAM_NAME: { DataType: 'String', StringValue: context.logStreamName },
        },
        MessageBody: JSON.stringify(message),
        MessageDeduplicationId: event.detail.eventDeduplicationId,
        QueueUrl: queueUrl,
    }).promise();
});
//# sourceMappingURL=data:application/json;base64,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