"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.OpenApiGatewayPythonProject = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
/*********************************************************************************************************************
 Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.

 Licensed under the Apache License, Version 2.0 (the "License").
 You may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 ******************************************************************************************************************** */
const path = require("path");
const projen_1 = require("projen");
const python_1 = require("projen/lib/python");
const client_settings_1 = require("./codegen/components/client-settings");
const docs_project_1 = require("./codegen/docs-project");
const generate_1 = require("./codegen/generate");
const languages_1 = require("./languages");
const python_2 = require("./samples/python");
const open_api_spec_project_1 = require("./spec/open-api-spec-project");
const OPENAPI_GATEWAY_PDK_PACKAGE_NAME = "aws_prototyping_sdk.open_api_gateway";
/**
 * Synthesizes a Python Project with an OpenAPI spec, generated clients, a CDK construct for deploying the API
 * with API Gateway, and generated lambda handler wrappers for type-safe handling of requests.
 *
 * @pjid open-api-gateway-py
 */
class OpenApiGatewayPythonProject extends python_1.PythonProject {
    constructor(projectOptions) {
        super({
            ...projectOptions,
            sample: false,
            venv: true,
            venvOptions: {
                envdir: ".env",
                ...projectOptions?.venvOptions,
            },
            pip: true,
            poetry: false,
            // No tests by default, but allow user to override
            pytest: projectOptions.pytest ?? false,
            setuptools: true,
        });
        const options = this.preConstruct(projectOptions);
        if (options.specFile && !path.isAbsolute(options.specFile)) {
            this.specDir = path.dirname(options.specFile);
            this.specFileName = path.basename(options.specFile);
        }
        else {
            this.specDir = "spec";
            this.specFileName = "spec.yaml";
        }
        this.generatedCodeDir = options.generatedCodeDir ?? "generated";
        this.forceGenerateCodeAndDocs = options.forceGenerateCodeAndDocs ?? false;
        this.apiSrcDir = options.apiSrcDir ?? "api";
        // Generated project should have a dependency on this project, in order to run the generation scripts
        [OPENAPI_GATEWAY_PDK_PACKAGE_NAME, "constructs", "aws-cdk-lib", "cdk-nag"]
            .filter((dep) => !this.deps.tryGetDependency(dep, projen_1.DependencyType.RUNTIME))
            .forEach((dep) => this.addDependency(dep));
        // Synthesize the openapi spec early since it's used by the generated python client, which is also synth'd early
        const spec = new open_api_spec_project_1.OpenApiSpecProject({
            name: `${this.name}-spec`,
            parent: this,
            outdir: path.join(this.moduleName, this.specDir),
            specFileName: this.specFileName,
            parsedSpecFileName: options.parsedSpecFileName,
            ...(options.specFile && path.isAbsolute(options.specFile)
                ? {
                    overrideSpecPath: options.specFile,
                }
                : {}),
        });
        spec.synth();
        // Parent the generated code with this project's parent for better integration with monorepos
        this.hasParent = !!options.parent;
        const generatedCodeDirRelativeToParent = this.hasParent
            ? path.join(options.outdir, this.generatedCodeDir)
            : this.generatedCodeDir;
        // Always generate the python client since this project will take a dependency on it in order to produce the
        // type-safe cdk construct wrapper.
        const clientLanguages = new Set(options.clientLanguages);
        clientLanguages.add(languages_1.ClientLanguage.PYTHON);
        const clientSettings = new client_settings_1.ClientSettings(this, {
            clientLanguages: [...clientLanguages],
            defaultClientLanguage: languages_1.ClientLanguage.PYTHON,
            documentationFormats: options.documentationFormats ?? [],
            forceGenerateCodeAndDocs: this.forceGenerateCodeAndDocs,
            generatedCodeDir: this.generatedCodeDir,
            specChanged: spec.specChanged,
        });
        // Share the same env between this project and the generated client. Accept a custom venv if part of a monorepo
        const envDir = options.venvOptions?.envdir || ".env";
        // env directory relative to the generated python client
        const clientEnvDir = path.join("..", ...this.generatedCodeDir.split("/").map(() => ".."), envDir);
        this.generatedClients = generate_1.generateClientProjects(clientSettings.clientLanguageConfigs, {
            parent: this.hasParent ? options.parent : this,
            parentPackageName: this.name,
            generatedCodeDir: generatedCodeDirRelativeToParent,
            parsedSpecPath: spec.parsedSpecPath,
            typescriptOptions: {
                defaultReleaseBranch: "main",
                ...options.typescriptClientOptions,
            },
            pythonOptions: {
                authorName: options.authorName ?? "APJ Cope",
                authorEmail: options.authorEmail ?? "apj-cope@amazon.com",
                version: "0.0.0",
                ...options.pythonClientOptions,
                // We are more prescriptive about the generated client since we must set up a dependency in the shared env
                pip: true,
                poetry: false,
                venv: true,
                venvOptions: {
                    envdir: clientEnvDir,
                },
                generateLayer: true,
            },
            javaOptions: {
                version: "0.0.0",
                ...options.javaClientOptions,
            },
        });
        this.generatedPythonClient = this.generatedClients[languages_1.ClientLanguage.PYTHON];
        // Synth early so that the generated code is available prior to this project's install phase
        this.generatedPythonClient.synth();
        // Add a dependency on the generated python client, which should be available since we share the virtual env
        this.addDependency(this.generatedPythonClient.moduleName);
        if (this.hasParent) {
            // Since the generated python client project is parented by this project's parent rather than this project,
            // projen will clean up the generated client when synthesizing this project unless we add an explicit exclude.
            this.addExcludeFromCleanup(`${this.generatedCodeDir}/**/*`);
            if ("addImplicitDependency" in this.parent) {
                // If we're within a monorepo, add an implicit dependency to ensure the generated python client is built first
                this.parent.addImplicitDependency(this, this.generatedPythonClient);
            }
        }
        // Get the lambda layer dir relative to the root of this project
        const pythonLayerDistDir = path.join(this.generatedCodeDir, languages_1.ClientLanguage.PYTHON, this.generatedPythonClient.layerDistDir);
        // Ensure it's included in the package
        new projen_1.TextFile(this, "MANIFEST.in", {
            lines: [`recursive-include ${pythonLayerDistDir} *`],
        });
        // Generate the sample source and test code
        const sampleOptions = {
            openApiGatewayPackageName: OPENAPI_GATEWAY_PDK_PACKAGE_NAME,
            pythonClientPackageName: this.generatedPythonClient.moduleName,
            sampleCode: options.sample,
            specDir: this.specDir,
            parsedSpecFileName: spec.parsedSpecFileName,
            moduleName: this.moduleName,
        };
        // Define some helpers for resolving resource paths in spec_utils.py
        new projen_1.SampleFile(this, path.join(this.moduleName, "spec_utils.py"), {
            contents: `import pkgutil, json
from os import path
from pathlib import Path

SPEC_PATH = path.join(str(Path(__file__).absolute().parent), "${this.specDir}/${spec.parsedSpecFileName}")
SPEC = json.loads(pkgutil.get_data(__name__, "${this.specDir}/${spec.parsedSpecFileName}"))

def get_project_root():
    return Path(__file__).absolute().parent.parent

def get_generated_client_layer_directory():
    return path.join(str(get_project_root()), "${pythonLayerDistDir}")
`,
        });
        new projen_1.SampleFile(this, path.join(this.moduleName, "__init__.py"), {
            contents: "#",
        });
        new projen_1.SampleDir(this, path.join(this.moduleName, this.apiSrcDir), {
            files: python_2.getPythonSampleSource(sampleOptions),
        });
        // Generate documentation if needed
        new docs_project_1.DocsProject({
            parent: this,
            outdir: path.join(this.generatedCodeDir, "documentation"),
            name: "docs",
            formatConfigs: clientSettings.documentationFormatConfigs,
            specPath: spec.parsedSpecPath,
        });
    }
    /**
     * This method provides inheritors a chance to synthesize extra resources prior to those created by this project.
     * Return any options you wish to change, other than python project options.
     */
    preConstruct(options) {
        return options;
    }
}
exports.OpenApiGatewayPythonProject = OpenApiGatewayPythonProject;
_a = JSII_RTTI_SYMBOL_1;
OpenApiGatewayPythonProject[_a] = { fqn: "@aws-prototyping-sdk/open-api-gateway.OpenApiGatewayPythonProject", version: "0.11.1" };
//# sourceMappingURL=data:application/json;base64,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