"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const tar_stream_1 = require("tar-stream");
const caching_1 = require("../../caching");
const client_lambda_shared_1 = require("../deny-list/client.lambda-shared");
const aws = require("../shared/aws.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
/**
 * Regenerates the `catalog.json` object in the configured S3 bucket.
 *
 * @param event configuration for the rebuild job. In particular, the `rebuild`
 *              property can be set to `true` in order to trigger a full (i.e:
 *              non-incremental) rebuild of the object.
 * @param context the lambda context in which this execution runs.
 *
 * @returns the information about the updated S3 object.
 */
async function handler(event, context) {
    console.log(JSON.stringify(event, null, 2));
    const BUCKET_NAME = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    const packages = new Map();
    const denyList = await client_lambda_shared_1.DenyListClient.newClient();
    console.log('Loading existing catalog (if present)...');
    const data = await aws.s3().getObject({ Bucket: BUCKET_NAME, Key: constants.CATALOG_KEY }).promise()
        .catch((err) => err.code !== 'NoSuchKey'
        ? Promise.reject(err)
        : Promise.resolve({ /* no data */}));
    if (data.Body) {
        console.log('Catalog found. Loading...');
        const catalog = JSON.parse(data.Body.toString('utf-8'));
        for (const info of catalog.packages) {
            const denyRule = denyList.lookup(info.name, info.version);
            if (denyRule != null) {
                console.log(`Dropping ${info.name}@${info.version} from catalog: ${denyRule.reason}`);
                continue;
            }
            if (!packages.has(info.name)) {
                packages.set(info.name, new Map());
            }
            packages.get(info.name).set(info.major, info);
        }
    }
    // If defined, the function will invoke itself again to resume the work from that key (this
    // happens only in "from scratch" or "rebuild" cases).
    let nextStartAfter;
    if (event.package) {
        if (!event.package.key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            throw new Error(`The provided package key is invalid: ${event.package.key} does not end in ${constants.PACKAGE_KEY_SUFFIX}`);
        }
        console.log('Registering new packages...');
        // note that we intentionally don't catch errors here to let these
        // event go to the DLQ for manual inspection.
        await appendPackage(packages, event.package.key, BUCKET_NAME, denyList);
    }
    // If we don't have a package in event, then we're refreshing the catalog. This is also true if we
    // don't have a catalog body (from scratch) or if "startAfter" is set (continuation of from
    // scratch).
    if (!(event === null || event === void 0 ? void 0 : event.package) || !data.Body || event.startAfter) {
        console.log('Recreating or refreshing catalog...');
        const failures = {};
        for await (const { Key: pkgKey } of relevantObjects(BUCKET_NAME, event.startAfter)) {
            try {
                await appendPackage(packages, pkgKey, BUCKET_NAME, denyList);
            }
            catch (e) {
                failures[pkgKey] = e;
            }
            // If we're getting short on time (1 minute out of 15 left), we'll be continuing in a new
            // invocation after writing what we've done so far to S3...
            if (context.getRemainingTimeInMillis() <= 60000) {
                nextStartAfter = pkgKey;
                break;
            }
        }
        for (const [key, error] of Object.entries(failures)) {
            console.log(`Failed processing ${key}: ${error}`);
        }
        await aws_embedded_metrics_1.metricScope((metrics) => async () => {
            metrics.setDimensions();
            const failedCount = Object.keys(failures).length;
            console.log(`Marking ${failedCount} failed packages`);
            metrics.putMetric("FailedPackagesOnRecreation" /* FAILED_PACKAGES_ON_RECREATION */, failedCount, aws_embedded_metrics_1.Unit.Count);
        })();
    }
    // Build the final data package...
    console.log('Consolidating catalog...');
    const catalog = { packages: new Array(), updated: new Date().toISOString() };
    for (const majors of packages.values()) {
        for (const pkg of majors.values()) {
            catalog.packages.push(pkg);
        }
    }
    console.log(`There are now ${catalog.packages.length} registered package major versions`);
    await aws_embedded_metrics_1.metricScope((metrics) => async () => {
        metrics.setDimensions();
        metrics.putMetric("RegisteredPackagesMajorVersion" /* REGISTERED_PACKAGES_MAJOR_VERSION */, catalog.packages.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingConstructFrameworkCount" /* MISSING_CONSTRUCT_FRAMEWORK_COUNT */, catalog.packages.filter((pkg) => pkg.constructFramework == null).length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingConstructFrameworkVersionCount" /* MISSING_CONSTRUCT_FRAMEWORK_VERSION_COUNT */, catalog.packages.filter((pkg) => pkg.constructFramework && pkg.constructFramework.majorVersion == null).length, aws_embedded_metrics_1.Unit.Count);
    })();
    // Clean up existing entries if necessary. In particular, remove the license texts as they make
    // the catalog unnecessarily large, and may hinder some search queries' result quality.
    for (const entry of catalog.packages) {
        if (entry.metadata) {
            delete entry.metadata.licenseText;
        }
    }
    // Upload the result to S3 and exit.
    const result = await aws.s3().putObject({
        Bucket: BUCKET_NAME,
        Key: constants.CATALOG_KEY,
        Body: JSON.stringify(catalog, null, 2),
        ContentType: 'application/json',
        CacheControl: caching_1.CacheStrategy.default().toString(),
        Metadata: {
            'Lambda-Log-Group': context.logGroupName,
            'Lambda-Log-Stream': context.logStreamName,
            'Lambda-Run-Id': context.awsRequestId,
            'Package-Count': `${catalog.packages.length}`,
        },
    }).promise();
    if (nextStartAfter != null) {
        console.log(`Will continue from ${nextStartAfter} in new invocation...`);
        const nextEvent = { ...event, startAfter: nextStartAfter };
        // We start it asynchronously, as this function has a provisionned
        // concurrency of 1 (so a synchronous attempt would always be throttled).
        await aws.lambda().invokeAsync({
            FunctionName: context.functionName,
            InvokeArgs: JSON.stringify(nextEvent, null, 2),
        }).promise();
    }
    return result;
}
exports.handler = handler;
/**
 * A generator that asynchronously traverses the set of "interesting" objects
 * found by listing the configured S3 bucket. Those objects correspond to all
 * npm package tarballs present under the `packages/` prefix in the bucket.
 *
 * @param bucket the bucket in which to list objects
 * @param startAfter the key to start reading from, if provided.
 */
async function* relevantObjects(bucket, startAfter) {
    var _a, _b;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
        StartAfter: startAfter,
    };
    do {
        const result = await aws.s3().listObjectsV2(request).promise();
        for (const object of (_a = result.Contents) !== null && _a !== void 0 ? _a : []) {
            if (!((_b = object.Key) === null || _b === void 0 ? void 0 : _b.endsWith(constants.PACKAGE_KEY_SUFFIX))) {
                continue;
            }
            // We only register packages if they have AT LEAST docs in one language.
            const tsDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_TYPESCRIPT}`;
            const pyDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_PYTHON}`;
            const javaDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_JAVA}`;
            const csharpDocs = `${object.Key.substring(0, object.Key.length - constants.PACKAGE_KEY_SUFFIX.length)}${constants.DOCS_KEY_SUFFIX_CSHARP}`;
            if (!(await aws.s3ObjectExists(bucket, tsDocs)) &&
                !(await aws.s3ObjectExists(bucket, pyDocs)) &&
                !(await aws.s3ObjectExists(bucket, javaDocs)) &&
                !(await aws.s3ObjectExists(bucket, csharpDocs))) {
                continue;
            }
            yield object;
        }
        request.ContinuationToken = result.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
async function appendPackage(packages, pkgKey, bucketName, denyList) {
    var _a, _b, _c;
    console.log(`Processing key: ${pkgKey}`);
    const [, packageName, versionStr] = constants.STORAGE_KEY_FORMAT_REGEX.exec(pkgKey);
    const version = new semver_1.SemVer(versionStr);
    const found = (_a = packages.get(packageName)) === null || _a === void 0 ? void 0 : _a.get(version.major);
    // If the version is === to the current latest, we'll be replacing that (so re-generated metadata are taken into account)
    if (found != null && version.compare(found.version) < 0) {
        console.log(`Skipping ${packageName}@${version} because it is not newer than the existing ${found.version}`);
        return;
    }
    console.log(`Checking if ${packageName}@${version.version} matches a deny list rule`);
    const blocked = denyList.lookup(packageName, version.version);
    if (blocked) {
        console.log(`Skipping ${packageName}@${version.version} because it is blocked by the deny list rule: ${JSON.stringify(blocked)}`);
        return;
    }
    console.log(`Registering ${packageName}@${version}`);
    // Donwload the tarball to inspect the `package.json` data therein.
    const pkg = await aws.s3().getObject({ Bucket: bucketName, Key: pkgKey }).promise();
    const metadataKey = pkgKey.replace(constants.PACKAGE_KEY_SUFFIX, constants.METADATA_KEY_SUFFIX);
    const metadataResponse = await aws.s3().getObject({ Bucket: bucketName, Key: metadataKey }).promise();
    const manifest = await new Promise((ok, ko) => {
        zlib_1.gunzip(Buffer.from(pkg.Body), (err, tar) => {
            if (err) {
                return ko(err);
            }
            tar_stream_1.extract()
                .on('entry', (header, stream, next) => {
                if (header.name !== 'package/package.json') {
                    // Not the file we are looking for, skip ahead (next run-loop tick).
                    return setImmediate(next);
                }
                const chunks = new Array();
                return stream
                    .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                    .once('end', () => {
                    ok(Buffer.concat(chunks));
                    next();
                })
                    .resume();
            })
                .once('finish', () => {
                ko(new Error('Could not find package/package.json in tarball!'));
            })
                .write(tar, (writeErr) => {
                if (writeErr) {
                    ko(writeErr);
                }
            });
        });
    });
    // Add the PackageInfo into the working set
    const pkgMetadata = JSON.parse(manifest.toString('utf-8'));
    const npmMetadata = JSON.parse((_c = (_b = metadataResponse === null || metadataResponse === void 0 ? void 0 : metadataResponse.Body) === null || _b === void 0 ? void 0 : _b.toString('utf-8')) !== null && _c !== void 0 ? _c : '{}');
    const major = new semver_1.SemVer(pkgMetadata.version).major;
    if (!packages.has(pkgMetadata.name)) {
        packages.set(pkgMetadata.name, new Map());
    }
    packages.get(pkgMetadata.name).set(major, {
        author: pkgMetadata.author,
        description: pkgMetadata.description,
        keywords: pkgMetadata.keywords,
        languages: pkgMetadata.jsii.targets,
        license: pkgMetadata.license,
        major,
        metadata: npmMetadata,
        name: pkgMetadata.name,
        version: pkgMetadata.version,
    });
}
//# sourceMappingURL=data:application/json;base64,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