"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VersionTracker = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const events = require("@aws-cdk/aws-events");
const targets = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const constants_1 = require("../shared/constants");
const constants_2 = require("./constants");
const version_tracker_1 = require("./version-tracker");
/**
 * Periodically updates the `all-versions.json` object in the designated bucket
 * with the list of all ingested package versions.
 */
class VersionTracker extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        // the package data bucket is also used to store all-versions.json
        this.bucket = props.bucket;
        this.handler = new version_tracker_1.VersionTracker(this, 'Default', {
            description: `[${this.node.path}] Creates the all-versions.json in ${props.bucket.bucketName}`,
            environment: {
                [constants_2.ENV_PACKAGE_DATA_BUCKET_NAME]: this.bucket.bucketName,
                [constants_2.ENV_PACKAGE_DATA_KEY_PREFIX]: constants_1.STORAGE_KEY_PREFIX,
                [constants_2.ENV_VERSION_TRACKER_BUCKET_NAME]: this.bucket.bucketName,
                [constants_2.ENV_VERSION_TRACKER_OBJECT_KEY]: constants_1.VERSION_TRACKER_KEY,
            },
            logRetention: (_a = props.logRetention) !== null && _a !== void 0 ? _a : aws_logs_1.RetentionDays.TEN_YEARS,
            memorySize: 10240,
            reservedConcurrentExecutions: 1,
            timeout: core_1.Duration.minutes(1),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        const grant = this.bucket.grantReadWrite(this.handler);
        const updatePeriod = (_b = props.updatePeriod) !== null && _b !== void 0 ? _b : core_1.Duration.minutes(1);
        const event = new events.Rule(this, 'Rule', {
            schedule: events.Schedule.rate(updatePeriod),
            targets: [new targets.LambdaFunction(this.handler)],
        });
        // ensure event does not trigger until Lambda permissions are set up
        event.node.addDependency(grant);
        const failureAlarm = this.handler.metricErrors().createAlarm(scope, 'VersionTracker/Failures', {
            alarmName: `${scope.node.path}/VersionTracker/Failures`,
            alarmDescription: [
                'The version tracker function failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(this.handler)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 3,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        props.monitoring.addLowSeverityAlarm('VersionTracker Failures', failureAlarm);
        const notRunningAlarm = this.handler.metricInvocations({ period: updatePeriod })
            .createAlarm(scope, 'VersionTracker/NotRunning', {
            alarmName: `${scope.node.path}/VersionTracker/NotRunning`,
            alarmDescription: [
                'The version tracker function is not running!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(this.handler)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        props.monitoring.addLowSeverityAlarm('VersionTracker Not Running', notRunningAlarm);
    }
    metricTrackedPackagesCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "TrackedPackagesCount" /* TRACKED_PACKAGES_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricTrackedVersionsCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "TrackedVersionsCount" /* TRACKED_VERSIONS_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
}
exports.VersionTracker = VersionTracker;
//# sourceMappingURL=data:application/json;base64,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