# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['django_test_migrations',
 'django_test_migrations.contrib',
 'django_test_migrations.logic']

package_data = \
{'': ['*']}

install_requires = \
['typing_extensions>=3.7.4,<4.0.0']

entry_points = \
{'pytest11': ['django_test_migrations = '
              'django_test_migrations.contrib.pytest_plugin']}

setup_kwargs = {
    'name': 'django-test-migrations',
    'version': '1.0.0',
    'description': 'Test django schema and data migrations, including ordering',
    'long_description': '# django-test-migrations\n\n[![wemake.services](https://img.shields.io/badge/%20-wemake.services-green.svg?label=%20&logo=data%3Aimage%2Fpng%3Bbase64%2CiVBORw0KGgoAAAANSUhEUgAAABAAAAAQCAMAAAAoLQ9TAAAABGdBTUEAALGPC%2FxhBQAAAAFzUkdCAK7OHOkAAAAbUExURQAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAP%2F%2F%2F5TvxDIAAAAIdFJOUwAjRA8xXANAL%2Bv0SAAAADNJREFUGNNjYCAIOJjRBdBFWMkVQeGzcHAwksJnAPPZGOGAASzPzAEHEGVsLExQwE7YswCb7AFZSF3bbAAAAABJRU5ErkJggg%3D%3D)](https://wemake.services)\n[![Build status](https://github.com/wemake-services/django-test-migrations/workflows/test/badge.svg?branch=master&event=push)](https://github.com/wemake-services/django-test-migrations/actions?query=workflow%3Atest)\n[![codecov](https://codecov.io/gh/wemake-services/django-test-migrations/branch/master/graph/badge.svg)](https://codecov.io/gh/wemake-services/django-test-migrations)\n[![Python Version](https://img.shields.io/pypi/pyversions/django-test-migrations.svg)](https://pypi.org/project/django-test-migrations/)\n[![wemake-python-styleguide](https://img.shields.io/badge/style-wemake-000000.svg)](https://github.com/wemake-services/wemake-python-styleguide)\n\n\n## Features\n\n- Allows to test `django` schema and data migrations\n- Allows to test both forward and rollback migrations\n- Allows to test the migrations order\n- Allows to test migration names\n- Fully typed with annotations and checked with `mypy`, [PEP561 compatible](https://www.python.org/dev/peps/pep-0561/)\n- Easy to start: has lots of docs, tests, and tutorials\n\nRead the [announcing post](https://sobolevn.me/2019/10/testing-django-migrations).\nSee real-world [usage example](https://github.com/wemake-services/wemake-django-template).\n\n\n## Installation\n\n```bash\npip install django-test-migrations\n```\n\nWe support several `django` versions:\n\n- `1.11`\n- `2.1`\n- `2.2`\n- `3.0`\n\nOther versions might work too, but they are not officially supported.\n\n\n## Testing django migrations\n\nTesting migrations is not a frequent thing in `django` land.\nBut, sometimes it is totally required. When?\n\nWhen we do complex schema or data changes\nand what to be sure that existing data won\'t be corrupted.\nWe might also want to be sure that all migrations can be safely rolled back.\nAnd as a final touch we want to be sure that migrations\nare in the correct order and have correct dependencies.\n\n### Testing forward migrations\n\nTo test all migrations we have a [`Migrator`](https://github.com/wemake-services/django-test-migrations/blob/master/django_test_migrations/migrator.py) class.\n\nIt has three methods to work with:\n\n- `.apply_initial_migration()` which takes app and migration names to generate\n  a state before the actual migration happens. It creates the `before state`\n  by applying all migrations up to and including the ones passed as an argument.\n\n- `.apply_tested_migration()` which takes app and migration names to perform the\n  actual migration\n\n- `.reset()` to clean everything up after we are done with testing\n\nSo, here\'s an example:\n\n```python\nfrom django_test_migrations.migrator import Migrator\n\nmigrator = Migrator(database=\'default\')\n\n# Initial migration, currently our model has only a single string field:\n# Note:\n# We are testing migration `0002_someitem_is_clean`, so we are specifying\n# the name of the previous migration (`0001_initial`) in the\n# .apply_initial_migration() method in order to prepare a state of the database\n# before applying the migration we are going to test.\n#\nold_state = migrator.apply_initial_migration((\'main_app\', \'0001_initial\'))\nSomeItem = old_state.apps.get_model(\'main_app\', \'SomeItem\')\n\n# Let\'s create a model with just a single field specified:\nSomeItem.objects.create(string_field=\'a\')\nassert len(SomeItem._meta.get_fields()) == 2  # id + string_field\n\n# Now this migration will add `is_clean` field to the model:\nnew_state = migrator.apply_tested_migration(\n    (\'main_app\', \'0002_someitem_is_clean\'),\n)\nSomeItem = new_state.apps.get_model(\'main_app\', \'SomeItem\')\n\n# We can now test how our migration worked, new field is there:\nassert SomeItem.objects.filter(is_clean=True).count() == 0\nassert len(SomeItem._meta.get_fields()) == 3  # id + string_field + is_clean\n\n# Cleanup:\nmigrator.reset()\n```\n\nThat was an example of a forward migration.\n\n### Backward migration\n\nThe thing is that you can also test backward migrations.\nNothing really changes except migration names that you pass and your logic:\n\n```python\nmigrator = Migrator()\n\n# Currently our model has two field, but we need a rollback:\nold_state = migrator.apply_initial_migration(\n    (\'main_app\', \'0002_someitem_is_clean\'),\n)\nSomeItem = old_state.apps.get_model(\'main_app\', \'SomeItem\')\n\n# Create some data to illustrate your cases:\n# ...\n\n# Now this migration will drop `is_clean` field:\nnew_state = migrator.apply_tested_migration((\'main_app\', \'0001_initial\'))\n\n# Assert the results:\n# ...\n\n# Cleanup:\nmigrator.reset()\n```\n\n### Testing migrations ordering\n\nSometimes we also want to be sure that our migrations are in the correct order.\nAnd all our `dependecies = [...]` are correct.\n\nTo achieve that we have [`plan.py`](https://github.com/wemake-services/django-test-migrations/blob/master/django_test_migrations/plan.py) module.\n\nThat\'s how it can be used:\n\n```python\nfrom django_test_migrations.plan import all_migrations, nodes_to_tuples\n\nmain_migrations = all_migrations(\'default\', [\'main_app\', \'other_app\'])\nassert nodes_to_tuples(main_migrations) == [\n    (\'main_app\', \'0001_initial\'),\n    (\'main_app\', \'0002_someitem_is_clean\'),\n    (\'other_app\', \'0001_initial\'),\n    (\'main_app\', \'0003_update_is_clean\'),\n    (\'main_app\', \'0004_auto_20191119_2125\'),\n    (\'other_app\', \'0002_auto_20191120_2230\'),\n]\n```\n\nThis way you can be sure that migrations\nand apps that depend on each other will be executed in the correct order.\n\n\n## Test framework integrations 🐍\n\nWe support several test frameworks as first-class citizens.\nThat\'s a testing tool after all!\n\nNote that the Django `post_migrate` signal\'s receiver list is cleared at\nthe start of tests and restored afterwards. If you need to test your\nown `post_migrate` signals then attach/remove them during a test.\n\n### pytest\n\nWe ship `django-test-migrations` with a `pytest` plugin\nthat provides two convinient fixtures:\n\n- `migrator_factory` that gives you an opportunity\n  to create `Migrator` classes for any database\n- `migrator` instance for the `\'default\'` database\n\nThat\'s how it can be used:\n\n```python\nimport pytest\n\n@pytest.mark.django_db\ndef test_pytest_plugin_initial(migrator):\n    """Ensures that the initial migration works."""\n    old_state = migrator.apply_initial_migration((\'main_app\', None))\n\n    with pytest.raises(LookupError):\n        # Models does not yet exist:\n        old_state.apps.get_model(\'main_app\', \'SomeItem\')\n\n    new_state = migrator.apply_tested_migration((\'main_app\', \'0001_initial\'))\n    # After the initial migration is done, we can use the model state:\n    SomeItem = new_state.apps.get_model(\'main_app\', \'SomeItem\')\n    assert SomeItem.objects.filter(string_field=\'\').count() == 0\n```\n\n### unittest\n\nWe also ship an integration with the built-in `unittest` framework.\n\nHere\'s how it can be used:\n\n```python\nfrom django_test_migrations.contrib.unittest_case import MigratorTestCase\n\nclass TestDirectMigration(MigratorTestCase):\n    """This class is used to test direct migrations."""\n\n    migrate_from = (\'main_app\', \'0002_someitem_is_clean\')\n    migrate_to = (\'main_app\', \'0003_update_is_clean\')\n\n    def prepare(self):\n        """Prepare some data before the migration."""\n        SomeItem = self.old_state.apps.get_model(\'main_app\', \'SomeItem\')\n        SomeItem.objects.create(string_field=\'a\')\n        SomeItem.objects.create(string_field=\'a b\')\n\n    def test_migration_main0003(self):\n        """Run the test itself."""\n        SomeItem = self.new_state.apps.get_model(\'main_app\', \'SomeItem\')\n\n        assert SomeItem.objects.count() == 2\n        assert SomeItem.objects.filter(is_clean=True).count() == 1\n```\n\n\n## Testing migration names\n\n`django` generates migration names for you when you run `makemigrations`.\nAnd these names are bad ([read more](https://adamj.eu/tech/2020/02/24/how-to-disallow-auto-named-django-migrations/) about why it is bad)!\nJust look at this: `0004_auto_20191119_2125.py`\n\nWhat does this migration do? What changes does it have?\n\nOne can also pass `--name` attribute when creating migrations, but it is easy to forget.\n\nWe offer an automated solution: `django` check\nthat produces a warning for each badly named migration.\n\nAdd our check into your `INSTALLED_APPS`:\n\n```python\nINSTALLED_APPS = [\n    # ...\n\n    # Our custom check:\n    \'django_test_migrations.contrib.django_checks.AutoNames\',\n]\n```\n\nAnd then in your CI run:\n\n```bash\npython manage.py check --deploy --fail-level WARNING\n```\n\nThis way you will be safe from wrong names in your migrations.\n\nDo you have a migrations that cannot be renamed? Add them to the ignore list:\n\n```python\n# settings.py\n\nDTM_IGNORED_MIGRATIONS = {\n    (\'main_app\', \'0004_auto_20191119_2125\'),\n    (\'dependency_app\', \'0001_auto_20201110_2100\'),\n}\n```\n\nAnd we won\'t complain about them.\n\nOr you can completely ignore entire app:\n\n```python\n# settings.py\n\nDTM_IGNORED_MIGRATIONS = {\n    (\'dependency_app\', \'*\'),\n    (\'another_dependency_app\', \'*\'),\n}\n```\n\n\n## Credits\n\nThis project is based on work of other awesome people:\n\n- [@asfaltboy](https://gist.github.com/asfaltboy/b3e6f9b5d95af8ba2cc46f2ba6eae5e2)\n- [@blueyed](https://gist.github.com/blueyed/4fb0a807104551f103e6)\n- [@fernandogrd](https://gist.github.com/blueyed/4fb0a807104551f103e6#gistcomment-1546191)\n- [@adamchainz](https://adamj.eu/tech/2020/02/24/how-to-disallow-auto-named-django-migrations/)\n\n\n## License\n\nMIT.\n',
    'author': 'sobolevn',
    'author_email': 'mail@sobolevn.me',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://github.com/wemake-services/django-test-migrations',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.6,<4.0',
}


setup(**setup_kwargs)
