from django import forms, template
from django.template.loader import render_to_string

from ..fieldsets import TapeformFieldset


register = template.Library()


@register.simple_tag(takes_context=True)
def form(context, form, **kwargs):
    """
    The `form` template tag will render a tape-form enabled form using the template
    provided by `get_layout_template` method of the form using the context generated
    by `get_layout_context` method of the form.

    Usage::

        {% load tapeforms %}
        {% form my_form %}

    You can override the used layout template using the keyword argument `using`::

        {% load tapeforms %}
        {% form my_form using='other_form_layout_template.html' %}

    :param form: The Django form to render.
    :return: Rendered form (errors + hidden fields + fields) as HTML.
    """

    if not isinstance(form, (forms.BaseForm, TapeformFieldset)):
        raise template.TemplateSyntaxError(
            'Provided form should be a `Form` instance, actual type: {0}'.format(
                form.__class__.__name__))

    return render_to_string(
        form.get_layout_template(kwargs.get('using', None)),
        form.get_layout_context(),
    )


@register.simple_tag(takes_context=True)
def formfield(context, bound_field, **kwargs):
    """
    The `formfield` template tag will render a form field of a tape-form enabled form
    using the template provided by `get_field_template` method of the form together with
    the context generated by `get_field_context` method of the form.

    Usage::

        {% load tapeforms %}
        {% formfield my_form.my_field %}

    You can override the used field template using the keyword argument `using`::

        {% load tapeforms %}
        {% formfield my_form.my_field using='other_field_template.html' %}

    :param bound_field: The `BoundField` from a Django form to render.
    :return: Rendered field (label + widget + other stuff) as HTML.
    """

    if not isinstance(bound_field, forms.BoundField):
        raise template.TemplateSyntaxError(
            'Provided field should be a `BoundField` instance, actual type: {0}'.format(
                bound_field.__class__.__name__))

    return render_to_string(
        bound_field.form.get_field_template(bound_field, kwargs.get('using', None)),
        bound_field.form.get_field_context(bound_field),
    )
