"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.TextractGenericSyncSfnTask = exports.integrationResourceArn = exports.validatePatternSupported = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const path = require("path");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const aws_lambda_event_sources_1 = require("aws-cdk-lib/aws-lambda-event-sources");
const aws_logs_1 = require("aws-cdk-lib/aws-logs");
const sqs = require("aws-cdk-lib/aws-sqs");
const sfn = require("aws-cdk-lib/aws-stepfunctions");
const tasks = require("aws-cdk-lib/aws-stepfunctions-tasks");
function validatePatternSupported(integrationPattern, supportedPatterns) {
    if (!supportedPatterns.includes(integrationPattern)) {
        throw new Error(`Unsupported service integration pattern. Supported Patterns: ${supportedPatterns}. Received: ${integrationPattern}`);
    }
}
exports.validatePatternSupported = validatePatternSupported;
/**
 * Suffixes corresponding to different service integration patterns
 *
 * Key is the service integration pattern, value is the resource ARN suffix.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-to-resource.html
 */
const resourceArnSuffix = {
    [sfn.IntegrationPattern.REQUEST_RESPONSE]: '',
    [sfn.IntegrationPattern.RUN_JOB]: '.sync',
    [sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN]: '.waitForTaskToken',
};
function integrationResourceArn(service, api, integrationPattern) {
    if (!service || !api) {
        throw new Error("Both 'service' and 'api' must be provided to build the resource ARN.");
    }
    return `arn:${aws_cdk_lib_1.Aws.PARTITION}:states:::${service}:${api}` + (integrationPattern ? resourceArnSuffix[integrationPattern] : '');
}
exports.integrationResourceArn = integrationResourceArn;
class TextractGenericSyncSfnTask extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.version = '0.0.1';
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        validatePatternSupported(this.integrationPattern, TextractGenericSyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN && !sfn.FieldUtils.containsTaskToken(props.input)) {
            throw new Error('Task Token is required in `input` for callback. Use JsonPath.taskToken to set the token.');
        }
        if (this.props.associateWithParent && props.input && props.input.type !== sfn.InputType.OBJECT) {
            throw new Error('Could not enable `associateWithParent` because `input` is taken directly from a JSON path. Use `sfn.TaskInput.fromObject` instead.');
        }
        if (!props.s3OutputBucket || !props.s3OutputPrefix) {
            throw new Error('no s3OutputBucket and s3OutputPrefix passed in');
        }
        if (!props.s3OutputBucket || !props.s3OutputPrefix) {
            throw new Error('no s3OutputBucket and s3OutputPrefix passed in');
        }
        var textractAPI = props.textractAPI === undefined ? 'GENERIC' : props.textractAPI;
        var textractStateMachineTimeoutMinutes = props.textractStateMachineTimeoutMinutes === undefined ? 2880 : props.textractStateMachineTimeoutMinutes;
        var lambdaLogLevel = props.lambdaLogLevel === undefined ? 'DEBUG' : props.lambdaLogLevel;
        const duration = 100;
        this.textractSyncSQS = new sqs.Queue(this, 'SyncRequests', {
            visibilityTimeout: aws_cdk_lib_1.Duration.seconds(duration),
        });
        this.textractPutOnSQSFunction = new lambda.DockerImageFunction(this, 'PutOnSQS', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/put_on_sqs/')),
            memorySize: 128,
            timeout: aws_cdk_lib_1.Duration.seconds(30),
            environment: {
                LOG_LEVEL: lambdaLogLevel,
                SQS_QUEUE_URL: this.textractSyncSQS.queueUrl,
            },
        });
        this.textractPutOnSQSFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sqs:SendMessage'], resources: [this.textractSyncSQS.queueArn] }));
        this.putOnSQSLambdaLogGroup = this.textractPutOnSQSFunction.logGroup;
        const putOnSQSFunctionInvoke = new tasks.LambdaInvoke(this, 'PutOnSQSFunctionInvoke', {
            lambdaFunction: this.textractPutOnSQSFunction,
        });
        this.textractSyncCallFunction = new lambda.DockerImageFunction(this, 'TextractSyncCall', {
            code: lambda.DockerImageCode.fromImageAsset(path.join(__dirname, '../lambda/textract_sync/')),
            memorySize: 512,
            timeout: aws_cdk_lib_1.Duration.seconds(duration),
            environment: {
                S3_OUTPUT_BUCKET: props.s3OutputBucket,
                S3_OUTPUT_PREFIX: props.s3OutputPrefix,
                LOG_LEVEL: lambdaLogLevel,
                SQS_QUEUE_URL: this.textractSyncSQS.queueName,
                TEXTRACT_API: textractAPI,
            },
        });
        this.textractSyncCallFunction.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(this.textractSyncSQS, { batchSize: 1 }));
        this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['textract:*'], resources: ['*'] }));
        this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                's3:GetObject', 's3:ListBucket', 's3:PutObject',
            ],
            resources: [`arn:aws:s3:::${props.s3OutputBucket}`, `arn:aws:s3:::${props.s3OutputBucket}/*`, '*'],
        }));
        // this.textractAsyncCallFunction.addToRolePolicy(new iam.PolicyStatement({ actions: ['sns:*'], resources: [this.textractAsyncSNS.topicArn] }));
        this.textractSyncLambdaLogGroup = this.textractSyncCallFunction.logGroup;
        const workflow_chain = sfn.Chain.start(putOnSQSFunctionInvoke);
        this.stateMachine = new sfn.StateMachine(this, 'StateMachine', {
            definition: workflow_chain,
            timeout: aws_cdk_lib_1.Duration.hours(textractStateMachineTimeoutMinutes),
            tracingEnabled: true,
        });
        this.textractSyncCallFunction.addToRolePolicy(new iam.PolicyStatement({
            actions: [
                'states:SendTaskFailure', 'states:SendTaskSuccess',
            ],
            resources: ['*'],
        }));
        // =========
        // DASHBOARD
        // =========
        const appName = this.node.tryGetContext('appName');
        const customMetricNamespace = 'TextractConstructGenericSync';
        // OPERATIONAL
        const asyncDurationMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-DurationFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'Duration',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'durationMs')
                .whereString('message', '=', 'textract_sync_generic_call_duration_in_ms:'),
            metricValue: '$durationMs',
        });
        const asyncDurationMetric = asyncDurationMetricFilter.metric({ statistic: 'avg' });
        const syncNumberPagesMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberPages',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', 'textract_sync_generic_number_of_pages_processed:'),
            metricValue: '$pages',
        });
        const syncNumberPagesMetric = syncNumberPagesMetricFilter.metric({ statistic: 'sum' });
        const syncNumberPagesSendMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-NumberPagesSendFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'NumberPages',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', 'textract_sync_generic_number_of_pages_send_to_process:'),
            metricValue: '$pages',
        });
        const syncNumberPagesSendMetric = syncNumberPagesSendMetricFilter.metric({ statistic: 'sum' });
        // TODO: expose all filters as properties of Construct, so users can build their own metrics
        const asyncJobFinshedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsFinishedFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'JobsFinished',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message', 'pages')
                .whereString('message', '=', 'textract_sync_generic_number_of_pages_processed:'),
            metricValue: '1',
        });
        const asyncJobFinshedMetric = asyncJobFinshedMetricFilter.metric({ statistic: 'sum' });
        const asyncJobStartedMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-JobsStartedFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'JobsStarted',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message')
                .whereString('message', '=', 'textract_async_generic_job_started'),
            metricValue: '1',
        });
        const asyncJobStartedMetric = asyncJobStartedMetricFilter.metric({ statistic: 'sum' });
        const syncTimedOutMetricFilter = new aws_logs_1.MetricFilter(this, `${appName}-TimedOutFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'DurationSeconds',
            filterPattern: aws_logs_1.FilterPattern.spaceDelimited('INFO', 'timestamp', 'id', 'message1', 'durationS', 'message2')
                .whereString('message1', '=', 'Task timed out after').whereString('message2', '=', 'seconds'),
            metricValue: '$durationS',
        });
        const syncTimedOutMetric = syncTimedOutMetricFilter.metric({ statistic: 'sum' });
        // OPERATIONAL STOP
        // CALCUATED OPERATIONAL METRICS
        const pagesPerSecond = new cloudwatch.MathExpression({
            expression: 'pages / (duration / 1000)',
            usingMetrics: {
                pages: syncNumberPagesMetric,
                duration: asyncDurationMetric,
            },
        });
        const openJobs = new cloudwatch.MathExpression({
            expression: 'startedJobs - finishedJobs',
            usingMetrics: {
                startedJobs: asyncJobStartedMetric,
                finishedJobs: asyncJobFinshedMetric,
            },
        });
        // CALCUATED OPERATIONAL METRICS STOP
        const errorFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-ErrorFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'Errors',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('ERROR', 'Error', 'error'),
            metricValue: '1',
        });
        const limitExceededExceptionFilterMetric = new aws_logs_1.MetricFilter(this, `${appName}-limitExceededExceptionFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'LimitExceededException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.LimitExceededException'),
            metricValue: '1',
        });
        const throttlingException = new aws_logs_1.MetricFilter(this, `${appName}-throttlingExceptionFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'ThrottlingException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ThrottlingException'),
            metricValue: '1',
        });
        const provisionedThroughputExceededException = new aws_logs_1.MetricFilter(this, `${appName}-provisionedThroughputFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'ProvisionedThroughputExceededException',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.ProvisionedThroughputExceededException'),
            metricValue: '1',
        });
        const internalServerError = new aws_logs_1.MetricFilter(this, `${appName}-internalServerErrorFilter`, {
            logGroup: this.textractSyncCallFunction.logGroup,
            metricNamespace: customMetricNamespace,
            metricName: 'InternalServerError',
            filterPattern: aws_logs_1.FilterPattern.anyTerm('textract.exceptions.InternalServerError'),
            metricValue: '1',
        });
        const errorMetric = errorFilterMetric.metric({ statistic: 'sum' });
        const limitExceededMetric = limitExceededExceptionFilterMetric.metric({ statistic: 'sum' });
        const throttlingMetric = throttlingException.metric({ statistic: 'sum' });
        const provisionedThroughputMetric = provisionedThroughputExceededException.metric({ statistic: 'sum' });
        const internalServerErrorMetric = internalServerError.metric({ statistic: 'sum' });
        const textractStartDocumentTextThrottle = new cloudwatch.Metric({
            namespace: 'AWS/Textract',
            metricName: 'ThrottledCount',
            dimensionsMap: {
                Operation: 'StartDocumentTextDetection',
            },
            statistic: 'sum',
        });
        // DASHBOARD LAMBDA
        const dashboardWidth = 24;
        // const widgetStandardHeight=9;
        // DASHBOARD
        const dashboard = new cloudwatch.Dashboard(this, '${appName}-TestDashboard', {
            end: 'end',
            periodOverride: cloudwatch.PeriodOverride.AUTO,
            start: 'start',
            widgets: [
                [
                    new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Operational Data Row widgets', width: dashboardWidth })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'PagesPerSecond', left: [pagesPerSecond], width: Math.floor(dashboardWidth / 2) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'OpenJobs', left: [openJobs], width: Math.floor(dashboardWidth / 2) })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'Duration', left: [asyncDurationMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPagesProcessed', left: [syncNumberPagesMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'NumberPagesSendToProcess', left: [syncNumberPagesSendMetric], width: Math.floor(dashboardWidth / 4) })),
                    new cloudwatch.Column(new cloudwatch.GraphWidget({ title: 'JobsStartedAndFinished', left: [asyncJobFinshedMetric], right: [asyncJobStartedMetric], width: Math.floor(dashboardWidth / 4) })),
                ],
                [
                    new cloudwatch.Column(new cloudwatch.TextWidget({ markdown: '# Async Textract Exceptions Row', width: dashboardWidth })),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Errors', left: [errorMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'LimitExceeded', left: [limitExceededMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'Throttling', left: [throttlingMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'ProvisionedThrougput', left: [provisionedThroughputMetric], width: Math.floor(dashboardWidth / 5) }),
                    new cloudwatch.GraphWidget({ title: 'InternalServerError', left: [internalServerErrorMetric], width: Math.floor(dashboardWidth / 5) }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# Textract', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Textract-StartDetectText-ThrottledCount', left: [textractStartDocumentTextThrottle] }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# textractSyncCallFunction', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'Sync-Function-Errors', left: [this.textractSyncCallFunction.metricErrors()], width: Math.floor(dashboardWidth / 4) }),
                    new cloudwatch.GraphWidget({ title: 'Sync-Function-Invocations', left: [this.textractSyncCallFunction.metricInvocations()], width: Math.floor(dashboardWidth / 4) }),
                    new cloudwatch.GraphWidget({ title: 'Sync-Function-Throttles', left: [this.textractSyncCallFunction.metricThrottles()], width: Math.floor(dashboardWidth / 4) }),
                    new cloudwatch.GraphWidget({ title: 'Sync-Function-TimedOut', left: [syncTimedOutMetric], width: Math.floor(dashboardWidth / 4) }),
                ],
                [
                    new cloudwatch.TextWidget({ markdown: '# SQS ', width: dashboardWidth }),
                ],
                [
                    new cloudwatch.GraphWidget({ title: 'sqsNumberMessagesSent', left: [this.textractSyncSQS.metricNumberOfMessagesSent()] }),
                    new cloudwatch.GraphWidget({ title: 'sqsNumberMessagesDeleted', left: [this.textractSyncSQS.metricNumberOfMessagesDeleted()] }),
                    new cloudwatch.GraphWidget({ title: 'sqsNumberMessagesVisible', left: [this.textractSyncSQS.metricApproximateNumberOfMessagesVisible()] }),
                    new cloudwatch.GraphWidget({ title: 'sqsNumberMessagesNotVisible', left: [this.textractSyncSQS.metricApproximateNumberOfMessagesNotVisible()] }),
                    new cloudwatch.GraphWidget({ title: 'sqsAgeOldestMessage', left: [this.textractSyncSQS.metricApproximateAgeOfOldestMessage()] }),
                ],
            ],
        });
        this.dashboardName = dashboard.dashboardName;
        // END DASHBOARD
        this.taskPolicies = this.createScopedAccessPolicy();
    }
    /**
         * @internal
         */
    _renderTask() {
        // suffix of ':2' indicates that the output of the nested state machine should be JSON
        // suffix is only applicable when waiting for a nested state machine to complete (RUN_JOB)
        // https://docs.aws.amazon.com/step-functions/latest/dg/connect-stepfunctions.html
        const suffix = this.integrationPattern === sfn.IntegrationPattern.RUN_JOB ? ':2' : '';
        let input;
        if (this.props.associateWithParent) {
            const associateWithParentEntry = {
                AWS_STEP_FUNCTIONS_STARTED_BY_EXECUTION_ID: sfn.JsonPath.stringAt('$$.Execution.Id'),
            };
            input = this.props.input ? {
                ...this.props.input.value,
                ...associateWithParentEntry,
            } : associateWithParentEntry;
        }
        else {
            input = this.props.input ? this.props.input.value : sfn.TaskInput.fromJsonPathAt('$').value;
        }
        return {
            Resource: `${integrationResourceArn('states', 'startExecution', this.integrationPattern)}${suffix}`,
            Parameters: sfn.FieldUtils.renderObject({
                Input: input,
                StateMachineArn: this.stateMachine.stateMachineArn,
                Name: this.props.name,
            }),
        };
    }
    /**
         * As StateMachineArn is extracted automatically from the state machine object included in the constructor,
         *
         * the scoped access policy should be generated accordingly.
         *
         * This means the action of StartExecution should be restricted on the given state machine, instead of being granted to all the resources (*).
         */
    createScopedAccessPolicy() {
        const stack = aws_cdk_lib_1.Stack.of(this);
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['states:StartExecution'],
                resources: [this.stateMachine.stateMachineArn],
            }),
        ];
        // Step Functions use Cloud Watch managed rules to deal with synchronous tasks.
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'states:DescribeExecution', 'states:StopExecution',
                ],
                // https://docs.aws.amazon.com/step-functions/latest/dg/concept-create-iam-advanced.html#concept-create-iam-advanced-execution
                resources: [
                    stack.formatArn({
                        service: 'states',
                        resource: 'execution',
                        arnFormat: aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME,
                        resourceName: `${stack.splitArn(this.stateMachine.stateMachineArn, aws_cdk_lib_1.ArnFormat.COLON_RESOURCE_NAME).resourceName}*`,
                    }),
                ],
            }));
            policyStatements.push(new iam.PolicyStatement({
                actions: [
                    'events:PutTargets', 'events:PutRule', 'events:DescribeRule',
                ],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForStepFunctionsExecutionRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.TextractGenericSyncSfnTask = TextractGenericSyncSfnTask;
_a = JSII_RTTI_SYMBOL_1;
TextractGenericSyncSfnTask[_a] = { fqn: "schadem-cdk-construct-textract-generic-async.TextractGenericSyncSfnTask", version: "0.0.4" };
TextractGenericSyncSfnTask.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGV4dHJhY3RTeW5jLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vc3JjL3RleHRyYWN0U3luYy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7OztBQUFBLDZCQUE2QjtBQUM3Qiw2Q0FBOEQ7QUFDOUQseURBQXlEO0FBQ3pELDJDQUEyQztBQUMzQyxpREFBaUQ7QUFDakQsbUZBQXNFO0FBQ3RFLG1EQUE4RTtBQUM5RSwyQ0FBMkM7QUFDM0MscURBQXFEO0FBQ3JELDZEQUE2RDtBQUk3RCxTQUFnQix3QkFBd0IsQ0FBQyxrQkFBMEMsRUFBRSxpQkFBMkM7SUFDOUgsSUFBSSxDQUFFLGlCQUFpQixDQUFDLFFBQVEsQ0FBQyxrQkFBa0IsQ0FBQyxFQUFFO1FBQ3BELE1BQU0sSUFBSSxLQUFLLENBQUMsZ0VBQWdFLGlCQUFpQixlQUFlLGtCQUFrQixFQUFFLENBQUMsQ0FBQztLQUN2STtBQUNILENBQUM7QUFKRCw0REFJQztBQUNEOzs7Ozs7R0FNRztBQUNILE1BQU0saUJBQWlCLEdBQTJDO0lBQ2hFLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQixDQUFDLEVBQUUsRUFBRTtJQUM3QyxDQUFDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxPQUFPLENBQUMsRUFBRSxPQUFPO0lBQ3pDLENBQUMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLG1CQUFtQixDQUFDLEVBQUUsbUJBQW1CO0NBQ2xFLENBQUM7QUFDRixTQUFnQixzQkFBc0IsQ0FBQyxPQUFlLEVBQUUsR0FBVyxFQUFFLGtCQUEyQztJQUM5RyxJQUFJLENBQUUsT0FBTyxJQUFJLENBQUUsR0FBRyxFQUFFO1FBQ3RCLE1BQU0sSUFBSSxLQUFLLENBQUMsc0VBQXNFLENBQUMsQ0FBQztLQUN6RjtJQUNELE9BQU8sT0FDTCxpQkFBRyxDQUFDLFNBQ04sYUFBYSxPQUFPLElBQUksR0FBRyxFQUFFLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsaUJBQWlCLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7QUFDcEcsQ0FBQztBQVBELHdEQU9DO0FBcURELE1BQWEsMEJBQTJCLFNBQVEsR0FBRyxDQUFDLGFBQWE7SUFvQi9ELFlBQVksS0FBaUIsRUFBRSxFQUFXLEVBQW1CLEtBQXVDO1FBQ2xHLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBRG1DLFVBQUssR0FBTCxLQUFLLENBQWtDO1FBR2xHLElBQUksQ0FBQyxPQUFPLEdBQUcsT0FBTyxDQUFDO1FBQ3ZCLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLElBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQixDQUFDO1FBQzlGLHdCQUF3QixDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSwwQkFBMEIsQ0FBQyw4QkFBOEIsQ0FBQyxDQUFDO1FBRTdHLElBQUksSUFBSSxDQUFDLGtCQUFrQixLQUFLLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUIsSUFBSSxDQUFDLEdBQUcsQ0FBQyxVQUFVLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxFQUFFO1lBQzVILE1BQU0sSUFBSSxLQUFLLENBQUMsMEZBQTBGLENBQUMsQ0FBQztTQUM3RztRQUVELElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsSUFBSSxLQUFLLENBQUMsS0FBSyxJQUFJLEtBQUssQ0FBQyxLQUFLLENBQUMsSUFBSSxLQUFLLEdBQUcsQ0FBQyxTQUFTLENBQUMsTUFBTSxFQUFFO1lBQzlGLE1BQU0sSUFBSSxLQUFLLENBQUMsb0lBQW9JLENBQUMsQ0FBQztTQUN2SjtRQUVELElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsRUFBRTtZQUNsRCxNQUFNLElBQUksS0FBSyxDQUFDLGdEQUFnRCxDQUFDLENBQUM7U0FDbkU7UUFFRCxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEVBQUU7WUFDbEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxnREFBZ0QsQ0FBQyxDQUFDO1NBQ25FO1FBRUQsSUFBSSxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNsRixJQUFJLGtDQUFrQyxHQUFHLEtBQUssQ0FBQyxrQ0FBa0MsS0FBSyxTQUFTLENBQUMsQ0FBQyxDQUFDLElBQUksQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLGtDQUFrQyxDQUFDO1FBQ2xKLElBQUksY0FBYyxHQUFHLEtBQUssQ0FBQyxjQUFjLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7UUFFekYsTUFBTSxRQUFRLEdBQVcsR0FBRyxDQUFDO1FBQzdCLElBQUksQ0FBQyxlQUFlLEdBQUcsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLElBQUksRUFBRSxjQUFjLEVBQUU7WUFDekQsaUJBQWlCLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDO1NBQzlDLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyx3QkFBd0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQy9FLElBQUksRUFBRSxNQUFNLENBQUMsZUFBZSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSx1QkFBdUIsQ0FBQyxDQUFDO1lBQzFGLFVBQVUsRUFBRSxHQUFHO1lBQ2YsT0FBTyxFQUFFLHNCQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUM3QixXQUFXLEVBQUU7Z0JBQ1gsU0FBUyxFQUFFLGNBQWM7Z0JBQ3pCLGFBQWEsRUFBRSxJQUFJLENBQUMsZUFBZSxDQUFDLFFBQVE7YUFDN0M7U0FDRixDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsd0JBQXdCLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLGlCQUFpQixDQUFDLEVBQUUsU0FBUyxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxRQUFRLENBQUMsRUFBRSxDQUFDLENBQUMsQ0FBQztRQUVySixJQUFJLENBQUMsc0JBQXNCLEdBQXNCLElBQUksQ0FBQyx3QkFBeUIsQ0FBQyxRQUFRLENBQUM7UUFFekYsTUFBTSxzQkFBc0IsR0FBRyxJQUFJLEtBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLHdCQUF3QixFQUFFO1lBQ3BGLGNBQWMsRUFBRSxJQUFJLENBQUMsd0JBQXdCO1NBQzlDLENBQUMsQ0FBQztRQUdILElBQUksQ0FBQyx3QkFBd0IsR0FBRyxJQUFJLE1BQU0sQ0FBQyxtQkFBbUIsQ0FBQyxJQUFJLEVBQUUsa0JBQWtCLEVBQUU7WUFDdkYsSUFBSSxFQUFFLE1BQU0sQ0FBQyxlQUFlLENBQUMsY0FBYyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLDBCQUEwQixDQUFDLENBQUM7WUFDN0YsVUFBVSxFQUFFLEdBQUc7WUFDZixPQUFPLEVBQUUsc0JBQVEsQ0FBQyxPQUFPLENBQUMsUUFBUSxDQUFDO1lBQ25DLFdBQVcsRUFBRTtnQkFDWCxnQkFBZ0IsRUFBRSxLQUFLLENBQUMsY0FBYztnQkFDdEMsZ0JBQWdCLEVBQUUsS0FBSyxDQUFDLGNBQWM7Z0JBQ3RDLFNBQVMsRUFBRSxjQUFjO2dCQUN6QixhQUFhLEVBQUUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxTQUFTO2dCQUM3QyxZQUFZLEVBQUUsV0FBVzthQUMxQjtTQUNGLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxjQUFjLENBQUMsSUFBSSx5Q0FBYyxDQUFDLElBQUksQ0FBQyxlQUFlLEVBQUUsRUFBRSxTQUFTLEVBQUUsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO1FBQ3pHLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDLEVBQUUsT0FBTyxFQUFFLENBQUMsWUFBWSxDQUFDLEVBQUUsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUM7UUFDdEgsSUFBSSxDQUFDLHdCQUF3QixDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDcEUsT0FBTyxFQUFFO2dCQUNQLGNBQWMsRUFBRSxlQUFlLEVBQUUsY0FBYzthQUNoRDtZQUNELFNBQVMsRUFBRSxDQUFDLGdCQUNWLEtBQUssQ0FBQyxjQUNSLEVBQUUsRUFBRSxnQkFDRixLQUFLLENBQUMsY0FDUixJQUFJLEVBQUUsR0FBRyxDQUFDO1NBQ1gsQ0FBQyxDQUFDLENBQUM7UUFDSixnSkFBZ0o7UUFDaEosSUFBSSxDQUFDLDBCQUEwQixHQUFvQixJQUFJLENBQUMsd0JBQXlCLENBQUMsUUFBUSxDQUFDO1FBRTNGLE1BQU0sY0FBYyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFFL0QsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLEdBQUcsQ0FBQyxZQUFZLENBQUMsSUFBSSxFQUFFLGNBQWMsRUFBRTtZQUM3RCxVQUFVLEVBQUUsY0FBYztZQUMxQixPQUFPLEVBQUUsc0JBQVEsQ0FBQyxLQUFLLENBQUMsa0NBQWtDLENBQUM7WUFDM0QsY0FBYyxFQUFFLElBQUk7U0FDckIsQ0FBQyxDQUFDO1FBRUgsSUFBSSxDQUFDLHdCQUF3QixDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7WUFDcEUsT0FBTyxFQUFFO2dCQUNQLHdCQUF3QixFQUFFLHdCQUF3QjthQUNuRDtZQUNELFNBQVMsRUFBRSxDQUFDLEdBQUcsQ0FBQztTQUNqQixDQUFDLENBQUMsQ0FBQztRQUNKLFlBQVk7UUFDWixZQUFZO1FBQ1osWUFBWTtRQUNaLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsYUFBYSxDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRW5ELE1BQU0scUJBQXFCLEdBQUcsOEJBQThCLENBQUM7UUFFN0QsY0FBYztRQUNkLE1BQU0seUJBQXlCLEdBQUcsSUFBSSx1QkFBWSxDQUFDLElBQUksRUFBRSxHQUFHLE9BQU8saUJBQWlCLEVBQUU7WUFDcEYsUUFBUSxFQUFxQixJQUFJLENBQUMsd0JBQXlCLENBQUMsUUFBUTtZQUNwRSxlQUFlLEVBQUUscUJBQXFCO1lBQ3RDLFVBQVUsRUFBRSxVQUFVO1lBQ3RCLGFBQWEsRUFBRSx3QkFBYSxDQUFDLGNBQWMsQ0FBQyxNQUFNLEVBQUUsV0FBVyxFQUFFLElBQUksRUFBRSxTQUFTLEVBQUUsWUFBWSxDQUFDO2lCQUM1RixXQUFXLENBQUMsU0FBUyxFQUFFLEdBQUcsRUFBRSw0Q0FBNEMsQ0FBQztZQUM1RSxXQUFXLEVBQUUsYUFBYTtTQUMzQixDQUFDLENBQUM7UUFDSCxNQUFNLG1CQUFtQixHQUFzQix5QkFBeUIsQ0FBQyxNQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUV0RyxNQUFNLDJCQUEyQixHQUFHLElBQUksdUJBQVksQ0FBQyxJQUFJLEVBQUUsR0FBRyxPQUFPLG9CQUFvQixFQUFFO1lBQ3pGLFFBQVEsRUFBcUIsSUFBSSxDQUFDLHdCQUF5QixDQUFDLFFBQVE7WUFDcEUsZUFBZSxFQUFFLHFCQUFxQjtZQUN0QyxVQUFVLEVBQUUsYUFBYTtZQUN6QixhQUFhLEVBQUUsd0JBQWEsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLE9BQU8sQ0FBQztpQkFDdkYsV0FBVyxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUUsa0RBQWtELENBQUM7WUFDbEYsV0FBVyxFQUFFLFFBQVE7U0FDdEIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxxQkFBcUIsR0FBc0IsMkJBQTJCLENBQUMsTUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFFMUcsTUFBTSwrQkFBK0IsR0FBRyxJQUFJLHVCQUFZLENBQUMsSUFBSSxFQUFFLEdBQUcsT0FBTyx3QkFBd0IsRUFBRTtZQUNqRyxRQUFRLEVBQXFCLElBQUksQ0FBQyx3QkFBeUIsQ0FBQyxRQUFRO1lBQ3BFLGVBQWUsRUFBRSxxQkFBcUI7WUFDdEMsVUFBVSxFQUFFLGFBQWE7WUFDekIsYUFBYSxFQUFFLHdCQUFhLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxXQUFXLEVBQUUsSUFBSSxFQUFFLFNBQVMsRUFBRSxPQUFPLENBQUM7aUJBQ3ZGLFdBQVcsQ0FBQyxTQUFTLEVBQUUsR0FBRyxFQUFFLHdEQUF3RCxDQUFDO1lBQ3hGLFdBQVcsRUFBRSxRQUFRO1NBQ3RCLENBQUMsQ0FBQztRQUNILE1BQU0seUJBQXlCLEdBQXNCLCtCQUErQixDQUFDLE1BQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBRWxILDRGQUE0RjtRQUM1RixNQUFNLDJCQUEyQixHQUFHLElBQUksdUJBQVksQ0FBQyxJQUFJLEVBQUUsR0FBRyxPQUFPLHFCQUFxQixFQUFFO1lBQzFGLFFBQVEsRUFBcUIsSUFBSSxDQUFDLHdCQUF5QixDQUFDLFFBQVE7WUFDcEUsZUFBZSxFQUFFLHFCQUFxQjtZQUN0QyxVQUFVLEVBQUUsY0FBYztZQUMxQixhQUFhLEVBQUUsd0JBQWEsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsU0FBUyxFQUFFLE9BQU8sQ0FBQztpQkFDdkYsV0FBVyxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUUsa0RBQWtELENBQUM7WUFDbEYsV0FBVyxFQUFFLEdBQUc7U0FDakIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxxQkFBcUIsR0FBc0IsMkJBQTJCLENBQUMsTUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFFMUcsTUFBTSwyQkFBMkIsR0FBRyxJQUFJLHVCQUFZLENBQUMsSUFBSSxFQUFFLEdBQUcsT0FBTyxvQkFBb0IsRUFBRTtZQUN6RixRQUFRLEVBQXFCLElBQUksQ0FBQyx3QkFBeUIsQ0FBQyxRQUFRO1lBQ3BFLGVBQWUsRUFBRSxxQkFBcUI7WUFDdEMsVUFBVSxFQUFFLGFBQWE7WUFDekIsYUFBYSxFQUFFLHdCQUFhLENBQUMsY0FBYyxDQUFDLE1BQU0sRUFBRSxXQUFXLEVBQUUsSUFBSSxFQUFFLFNBQVMsQ0FBQztpQkFDOUUsV0FBVyxDQUFDLFNBQVMsRUFBRSxHQUFHLEVBQUUsb0NBQW9DLENBQUM7WUFDcEUsV0FBVyxFQUFFLEdBQUc7U0FDakIsQ0FBQyxDQUFDO1FBQ0gsTUFBTSxxQkFBcUIsR0FBc0IsMkJBQTJCLENBQUMsTUFBTSxDQUFDLEVBQUUsU0FBUyxFQUFFLEtBQUssRUFBRSxDQUFDLENBQUM7UUFFMUcsTUFBTSx3QkFBd0IsR0FBRyxJQUFJLHVCQUFZLENBQUMsSUFBSSxFQUFFLEdBQUcsT0FBTyxpQkFBaUIsRUFBRTtZQUNuRixRQUFRLEVBQXFCLElBQUksQ0FBQyx3QkFBeUIsQ0FBQyxRQUFRO1lBQ3BFLGVBQWUsRUFBRSxxQkFBcUI7WUFDdEMsVUFBVSxFQUFFLGlCQUFpQjtZQUM3QixhQUFhLEVBQUUsd0JBQWEsQ0FBQyxjQUFjLENBQUMsTUFBTSxFQUFFLFdBQVcsRUFBRSxJQUFJLEVBQUUsVUFBVSxFQUFFLFdBQVcsRUFBRSxVQUFVLENBQUM7aUJBQ3hHLFdBQVcsQ0FBQyxVQUFVLEVBQUUsR0FBRyxFQUFFLHNCQUFzQixDQUFDLENBQUMsV0FBVyxDQUFDLFVBQVUsRUFBRSxHQUFHLEVBQUUsU0FBUyxDQUFDO1lBQy9GLFdBQVcsRUFBRSxZQUFZO1NBQzFCLENBQUMsQ0FBQztRQUNILE1BQU0sa0JBQWtCLEdBQXNCLHdCQUF3QixDQUFDLE1BQU0sQ0FBQyxFQUFFLFNBQVMsRUFBRSxLQUFLLEVBQUUsQ0FBQyxDQUFDO1FBQ3BHLG1CQUFtQjtRQUVuQixnQ0FBZ0M7UUFFaEMsTUFBTSxjQUFjLEdBQUcsSUFBSSxVQUFVLENBQUMsY0FBYyxDQUFDO1lBQ25ELFVBQVUsRUFBRSwyQkFBMkI7WUFDdkMsWUFBWSxFQUFFO2dCQUNaLEtBQUssRUFBRSxxQkFBcUI7Z0JBQzVCLFFBQVEsRUFBRSxtQkFBbUI7YUFDOUI7U0FDRixDQUFDLENBQUM7UUFDSCxNQUFNLFFBQVEsR0FBRyxJQUFJLFVBQVUsQ0FBQyxjQUFjLENBQUM7WUFDN0MsVUFBVSxFQUFFLDRCQUE0QjtZQUN4QyxZQUFZLEVBQUU7Z0JBQ1osV0FBVyxFQUFFLHFCQUFxQjtnQkFDbEMsWUFBWSxFQUFFLHFCQUFxQjthQUNwQztTQUNGLENBQUMsQ0FBQztRQUNILHFDQUFxQztRQUVyQyxNQUFNLGlCQUFpQixHQUFHLElBQUksdUJBQVksQ0FBQyxJQUFJLEVBQUUsR0FBRyxPQUFPLGNBQWMsRUFBRTtZQUN6RSxRQUFRLEVBQXFCLElBQUksQ0FBQyx3QkFBeUIsQ0FBQyxRQUFRO1lBQ3BFLGVBQWUsRUFBRSxxQkFBcUI7WUFDdEMsVUFBVSxFQUFFLFFBQVE7WUFDcEIsYUFBYSxFQUFFLHdCQUFhLENBQUMsT0FBTyxDQUFDLE9BQU8sRUFBRSxPQUFPLEVBQUUsT0FBTyxDQUFDO1lBQy9ELFdBQVcsRUFBRSxHQUFHO1NBQ2pCLENBQUMsQ0FBQztRQUVILE1BQU0sa0NBQWtDLEdBQUcsSUFBSSx1QkFBWSxDQUFDLElBQUksRUFBRSxHQUFHLE9BQU8sK0JBQStCLEVBQUU7WUFDM0csUUFBUSxFQUFxQixJQUFJLENBQUMsd0JBQXlCLENBQUMsUUFBUTtZQUNwRSxlQUFlLEVBQUUscUJBQXFCO1lBQ3RDLFVBQVUsRUFBRSx3QkFBd0I7WUFDcEMsYUFBYSxFQUFFLHdCQUFhLENBQUMsT0FBTyxDQUFDLDRDQUE0QyxDQUFDO1lBQ2xGLFdBQVcsRUFBRSxHQUFHO1NBQ2pCLENBQUMsQ0FBQztRQUNILE1BQU0sbUJBQW1CLEdBQUcsSUFBSSx1QkFBWSxDQUFDLElBQUksRUFBRSxHQUFHLE9BQU8sNEJBQTRCLEVBQUU7WUFDekYsUUFBUSxFQUFxQixJQUFJLENBQUMsd0JBQXlCLENBQUMsUUFBUTtZQUNwRSxlQUFlLEVBQUUscUJBQXFCO1lBQ3RDLFVBQVUsRUFBRSxxQkFBcUI7WUFDakMsYUFBYSxFQUFFLHdCQUFhLENBQUMsT0FBTyxDQUFDLHlDQUF5QyxDQUFDO1lBQy9FLFdBQVcsRUFBRSxHQUFHO1NBQ2pCLENBQUMsQ0FBQztRQUVILE1BQU0sc0NBQXNDLEdBQUcsSUFBSSx1QkFBWSxDQUFDLElBQUksRUFBRSxHQUFHLE9BQU8sOEJBQThCLEVBQUU7WUFDOUcsUUFBUSxFQUFxQixJQUFJLENBQUMsd0JBQXlCLENBQUMsUUFBUTtZQUNwRSxlQUFlLEVBQUUscUJBQXFCO1lBQ3RDLFVBQVUsRUFBRSx3Q0FBd0M7WUFDcEQsYUFBYSxFQUFFLHdCQUFhLENBQUMsT0FBTyxDQUFDLDREQUE0RCxDQUFDO1lBQ2xHLFdBQVcsRUFBRSxHQUFHO1NBQ2pCLENBQUMsQ0FBQztRQUNILE1BQU0sbUJBQW1CLEdBQUcsSUFBSSx1QkFBWSxDQUFDLElBQUksRUFBRSxHQUFHLE9BQU8sNEJBQTRCLEVBQUU7WUFDekYsUUFBUSxFQUFxQixJQUFJLENBQUMsd0JBQXlCLENBQUMsUUFBUTtZQUNwRSxlQUFlLEVBQUUscUJBQXFCO1lBQ3RDLFVBQVUsRUFBRSxxQkFBcUI7WUFDakMsYUFBYSxFQUFFLHdCQUFhLENBQUMsT0FBTyxDQUFDLHlDQUF5QyxDQUFDO1lBQy9FLFdBQVcsRUFBRSxHQUFHO1NBQ2pCLENBQUMsQ0FBQztRQUVILE1BQU0sV0FBVyxHQUFzQixpQkFBaUIsQ0FBQyxNQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUN0RixNQUFNLG1CQUFtQixHQUFzQixrQ0FBa0MsQ0FBQyxNQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUMvRyxNQUFNLGdCQUFnQixHQUFzQixtQkFBbUIsQ0FBQyxNQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUM3RixNQUFNLDJCQUEyQixHQUFzQixzQ0FBc0MsQ0FBQyxNQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUMzSCxNQUFNLHlCQUF5QixHQUFzQixtQkFBbUIsQ0FBQyxNQUFNLENBQUMsRUFBRSxTQUFTLEVBQUUsS0FBSyxFQUFFLENBQUMsQ0FBQztRQUV0RyxNQUFNLGlDQUFpQyxHQUFxQixJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUM7WUFDaEYsU0FBUyxFQUFFLGNBQWM7WUFDekIsVUFBVSxFQUFFLGdCQUFnQjtZQUM1QixhQUFhLEVBQUU7Z0JBQ2IsU0FBUyxFQUFFLDRCQUE0QjthQUN4QztZQUNELFNBQVMsRUFBRSxLQUFLO1NBQ2pCLENBQUMsQ0FBQztRQUNILG1CQUFtQjtRQUVuQixNQUFNLGNBQWMsR0FBQyxFQUFFLENBQUM7UUFDeEIsZ0NBQWdDO1FBRWhDLFlBQVk7UUFDWixNQUFNLFNBQVMsR0FBRyxJQUFJLFVBQVUsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFLDBCQUEwQixFQUFFO1lBQzNFLEdBQUcsRUFBRSxLQUFLO1lBQ1YsY0FBYyxFQUFFLFVBQVUsQ0FBQyxjQUFjLENBQUMsSUFBSTtZQUM5QyxLQUFLLEVBQUUsT0FBTztZQUNkLE9BQU8sRUFBRTtnQkFDUDtvQkFDRSxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUUsUUFBUSxFQUFFLGdDQUFnQyxFQUFFLEtBQUssRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDO2lCQUN4SDtnQkFDRDtvQkFDRSxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLGdCQUFnQixFQUFFLElBQUksRUFBRSxDQUFDLGNBQWMsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsR0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7b0JBQzNJLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQyxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxDQUFDLFFBQVEsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsR0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7aUJBQ2hJO2dCQUNEO29CQUNFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQyxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLElBQUksRUFBRSxDQUFDLG1CQUFtQixDQUFDLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxHQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQztvQkFDMUksSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSxzQkFBc0IsRUFBRSxJQUFJLEVBQUUsQ0FBQyxxQkFBcUIsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsR0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLENBQUM7b0JBQ3hKLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQyxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsMEJBQTBCLEVBQUUsSUFBSSxFQUFFLENBQUMseUJBQXlCLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO29CQUNoSyxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLHdCQUF3QixFQUFFLElBQUksRUFBRSxDQUFDLHFCQUFxQixDQUFDLEVBQUUsS0FBSyxFQUFFLENBQUMscUJBQXFCLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDO2lCQUMzTDtnQkFDRDtvQkFDRSxJQUFJLFVBQVUsQ0FBQyxNQUFNLENBQUMsSUFBSSxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUUsUUFBUSxFQUFFLGlDQUFpQyxFQUFFLEtBQUssRUFBRSxjQUFjLEVBQUUsQ0FBQyxDQUFDO2lCQUN6SDtnQkFDRDtvQkFDRSxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsUUFBUSxFQUFFLElBQUksRUFBRSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsR0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO29CQUN6RyxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsZUFBZSxFQUFFLElBQUksRUFBRSxDQUFDLG1CQUFtQixDQUFDLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxHQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7b0JBQ3hILElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSxZQUFZLEVBQUUsSUFBSSxFQUFFLENBQUMsZ0JBQWdCLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztvQkFDbEgsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLHNCQUFzQixFQUFFLElBQUksRUFBRSxDQUFDLDJCQUEyQixDQUFDLEVBQUUsS0FBSyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxHQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7b0JBQ3ZJLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSxxQkFBcUIsRUFBRSxJQUFJLEVBQUUsQ0FBQyx5QkFBeUIsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsR0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO2lCQUNySTtnQkFDRDtvQkFDRSxJQUFJLFVBQVUsQ0FBQyxVQUFVLENBQUMsRUFBRSxRQUFRLEVBQUUsWUFBWSxFQUFFLEtBQUssRUFBRSxjQUFjLEVBQUUsQ0FBQztpQkFDN0U7Z0JBQ0Q7b0JBQ0UsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLHlDQUF5QyxFQUFFLElBQUksRUFBRSxDQUFDLGlDQUFpQyxDQUFDLEVBQUUsQ0FBQztpQkFDNUg7Z0JBQ0Q7b0JBQ0UsSUFBSSxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUUsUUFBUSxFQUFFLDRCQUE0QixFQUFFLEtBQUssRUFBRSxjQUFjLEVBQUUsQ0FBQztpQkFDN0Y7Z0JBQ0Q7b0JBQ0UsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLHNCQUFzQixFQUFFLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxZQUFZLEVBQUUsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsR0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO29CQUN4SixJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsMkJBQTJCLEVBQUUsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLHdCQUF3QixDQUFDLGlCQUFpQixFQUFFLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztvQkFDbEssSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLHlCQUF5QixFQUFFLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyx3QkFBd0IsQ0FBQyxlQUFlLEVBQUUsQ0FBQyxFQUFFLEtBQUssRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWMsR0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO29CQUM5SixJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsd0JBQXdCLEVBQUUsSUFBSSxFQUFFLENBQUMsa0JBQWtCLENBQUMsRUFBRSxLQUFLLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLEdBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztpQkFDakk7Z0JBQ0Q7b0JBQ0UsSUFBSSxVQUFVLENBQUMsVUFBVSxDQUFDLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRSxLQUFLLEVBQUUsY0FBYyxFQUFFLENBQUM7aUJBQ3pFO2dCQUNEO29CQUNFLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSx1QkFBdUIsRUFBRSxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLDBCQUEwQixFQUFFLENBQUMsRUFBRSxDQUFDO29CQUN6SCxJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUsMEJBQTBCLEVBQUUsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyw2QkFBNkIsRUFBRSxDQUFDLEVBQUUsQ0FBQztvQkFDL0gsSUFBSSxVQUFVLENBQUMsV0FBVyxDQUFDLEVBQUUsS0FBSyxFQUFFLDBCQUEwQixFQUFFLElBQUksRUFBRSxDQUFDLElBQUksQ0FBQyxlQUFlLENBQUMsd0NBQXdDLEVBQUUsQ0FBQyxFQUFFLENBQUM7b0JBQzFJLElBQUksVUFBVSxDQUFDLFdBQVcsQ0FBQyxFQUFFLEtBQUssRUFBRSw2QkFBNkIsRUFBRSxJQUFJLEVBQUUsQ0FBQyxJQUFJLENBQUMsZUFBZSxDQUFDLDJDQUEyQyxFQUFFLENBQUMsRUFBRSxDQUFDO29CQUNoSixJQUFJLFVBQVUsQ0FBQyxXQUFXLENBQUMsRUFBRSxLQUFLLEVBQUUscUJBQXFCLEVBQUUsSUFBSSxFQUFFLENBQUMsSUFBSSxDQUFDLGVBQWUsQ0FBQyxtQ0FBbUMsRUFBRSxDQUFDLEVBQUUsQ0FBQztpQkFDakk7YUFDRjtTQUNGLENBQUMsQ0FBQztRQUVILElBQUksQ0FBQyxhQUFhLEdBQUMsU0FBUyxDQUFDLGFBQWEsQ0FBQztRQUMzQyxnQkFBZ0I7UUFDaEIsSUFBSSxDQUFDLFlBQVksR0FBRyxJQUFJLENBQUMsd0JBQXdCLEVBQUUsQ0FBQztJQUN0RCxDQUFDO0lBQ0Q7O1dBRU87SUFDRyxXQUFXO1FBQ25CLHNGQUFzRjtRQUN0RiwwRkFBMEY7UUFDMUYsa0ZBQWtGO1FBQ2xGLE1BQU0sTUFBTSxHQUFHLElBQUksQ0FBQyxrQkFBa0IsS0FBSyxHQUFHLENBQUMsa0JBQWtCLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUN0RixJQUFJLEtBQVUsQ0FBQztRQUNmLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxtQkFBbUIsRUFBRTtZQUNsQyxNQUFNLHdCQUF3QixHQUFHO2dCQUMvQiwwQ0FBMEMsRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLFFBQVEsQ0FBQyxpQkFBaUIsQ0FBQzthQUNyRixDQUFDO1lBQ0YsS0FBSyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQztnQkFDekIsR0FBRyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLO2dCQUN6QixHQUFJLHdCQUF3QjthQUM3QixDQUFDLENBQUMsQ0FBQyx3QkFBd0IsQ0FBQztTQUM5QjthQUFNO1lBQ0wsS0FBSyxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsS0FBSyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLEtBQUssQ0FBQztTQUM3RjtRQUdELE9BQU87WUFDTCxRQUFRLEVBQUUsR0FDUixzQkFBc0IsQ0FBQyxRQUFRLEVBQUUsZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUM1RSxHQUFHLE1BQU0sRUFBRTtZQUNYLFVBQVUsRUFBRSxHQUFHLENBQUMsVUFBVSxDQUFDLFlBQVksQ0FDckM7Z0JBQ0UsS0FBSyxFQUFFLEtBQUs7Z0JBQ1osZUFBZSxFQUFFLElBQUksQ0FBQyxZQUFZLENBQUMsZUFBZTtnQkFDbEQsSUFBSSxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSTthQUN0QixDQUNGO1NBQ0YsQ0FBQztJQUNKLENBQUM7SUFDRDs7Ozs7O1dBTU87SUFDQyx3QkFBd0I7UUFDOUIsTUFBTSxLQUFLLEdBQUcsbUJBQUssQ0FBQyxFQUFFLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFN0IsTUFBTSxnQkFBZ0IsR0FBRztZQUN2QixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQ3JCO2dCQUNFLE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDO2dCQUNsQyxTQUFTLEVBQUUsQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLGVBQWUsQ0FBQzthQUMvQyxDQUNGO1NBQ0YsQ0FBQztRQUVGLCtFQUErRTtRQUMvRSxJQUFJLElBQUksQ0FBQyxrQkFBa0IsS0FBSyxHQUFHLENBQUMsa0JBQWtCLENBQUMsT0FBTyxFQUFFO1lBQzlELGdCQUFnQixDQUFDLElBQUksQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQzVDLE9BQU8sRUFBRTtvQkFDUCwwQkFBMEIsRUFBRSxzQkFBc0I7aUJBQ25EO2dCQUNELDhIQUE4SDtnQkFDOUgsU0FBUyxFQUFFO29CQUNULEtBQUssQ0FBQyxTQUFTLENBQ2I7d0JBQ0UsT0FBTyxFQUFFLFFBQVE7d0JBQ2pCLFFBQVEsRUFBRSxXQUFXO3dCQUNyQixTQUFTLEVBQUUsdUJBQVMsQ0FBQyxtQkFBbUI7d0JBQ3hDLFlBQVksRUFBRSxHQUNaLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxlQUFlLEVBQUUsdUJBQVMsQ0FBQyxtQkFBbUIsQ0FBQyxDQUFDLFlBQ25GLEdBQUc7cUJBQ0osQ0FDRjtpQkFDRjthQUNGLENBQUMsQ0FBQyxDQUFDO1lBRUosZ0JBQWdCLENBQUMsSUFBSSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztnQkFDNUMsT0FBTyxFQUFFO29CQUNQLG1CQUFtQixFQUFFLGdCQUFnQixFQUFFLHFCQUFxQjtpQkFDN0Q7Z0JBQ0QsU0FBUyxFQUFFO29CQUNULEtBQUssQ0FBQyxTQUFTLENBQ2I7d0JBQ0UsT0FBTyxFQUFFLFFBQVE7d0JBQ2pCLFFBQVEsRUFBRSxNQUFNO3dCQUNoQixZQUFZLEVBQUUscURBQXFEO3FCQUNwRSxDQUNGO2lCQUNGO2FBQ0YsQ0FBQyxDQUFDLENBQUM7U0FDTDtRQUVELE9BQU8sZ0JBQWdCLENBQUM7SUFDMUIsQ0FBQzs7QUF6WkgsZ0VBMFpDOzs7QUF6WnlCLHlEQUE4QixHQUFHO0lBQ3ZELEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxnQkFBZ0I7SUFDdkMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLE9BQU87SUFDOUIsR0FBRyxDQUFDLGtCQUFrQixDQUFDLG1CQUFtQjtDQUMzQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCB7IER1cmF0aW9uLCBBd3MsIEFybkZvcm1hdCwgU3RhY2sgfSBmcm9tICdhd3MtY2RrLWxpYic7XG5pbXBvcnQgKiBhcyBjbG91ZHdhdGNoIGZyb20gJ2F3cy1jZGstbGliL2F3cy1jbG91ZHdhdGNoJztcbmltcG9ydCAqIGFzIGlhbSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtaWFtJztcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tICdhd3MtY2RrLWxpYi9hd3MtbGFtYmRhJztcbmltcG9ydCB7IFNxc0V2ZW50U291cmNlIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWxhbWJkYS1ldmVudC1zb3VyY2VzJztcbmltcG9ydCB7IEZpbHRlclBhdHRlcm4sIElMb2dHcm91cCwgTWV0cmljRmlsdGVyIH0gZnJvbSAnYXdzLWNkay1saWIvYXdzLWxvZ3MnO1xuaW1wb3J0ICogYXMgc3FzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zcXMnO1xuaW1wb3J0ICogYXMgc2ZuIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zdGVwZnVuY3Rpb25zJztcbmltcG9ydCAqIGFzIHRhc2tzIGZyb20gJ2F3cy1jZGstbGliL2F3cy1zdGVwZnVuY3Rpb25zLXRhc2tzJztcbmltcG9ydCB7IExhbWJkYUludm9rZSB9IGZyb20gJ2F3cy1jZGstbGliL2F3cy1zdGVwZnVuY3Rpb25zLXRhc2tzJztcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuXG5leHBvcnQgZnVuY3Rpb24gdmFsaWRhdGVQYXR0ZXJuU3VwcG9ydGVkKGludGVncmF0aW9uUGF0dGVybjogc2ZuLkludGVncmF0aW9uUGF0dGVybiwgc3VwcG9ydGVkUGF0dGVybnM6IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm5bXSkge1xuICBpZiAoISBzdXBwb3J0ZWRQYXR0ZXJucy5pbmNsdWRlcyhpbnRlZ3JhdGlvblBhdHRlcm4pKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKGBVbnN1cHBvcnRlZCBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm4uIFN1cHBvcnRlZCBQYXR0ZXJuczogJHtzdXBwb3J0ZWRQYXR0ZXJuc30uIFJlY2VpdmVkOiAke2ludGVncmF0aW9uUGF0dGVybn1gKTtcbiAgfVxufVxuLyoqXG4gKiBTdWZmaXhlcyBjb3JyZXNwb25kaW5nIHRvIGRpZmZlcmVudCBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm5zXG4gKlxuICogS2V5IGlzIHRoZSBzZXJ2aWNlIGludGVncmF0aW9uIHBhdHRlcm4sIHZhbHVlIGlzIHRoZSByZXNvdXJjZSBBUk4gc3VmZml4LlxuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9kZy9jb25uZWN0LXRvLXJlc291cmNlLmh0bWxcbiAqL1xuY29uc3QgcmVzb3VyY2VBcm5TdWZmaXg6IFJlY29yZDxzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLCBzdHJpbmc+ID0ge1xuICBbc2ZuLkludGVncmF0aW9uUGF0dGVybi5SRVFVRVNUX1JFU1BPTlNFXTogJycsXG4gIFtzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJVTl9KT0JdOiAnLnN5bmMnLFxuICBbc2ZuLkludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOXTogJy53YWl0Rm9yVGFza1Rva2VuJyxcbn07XG5leHBvcnQgZnVuY3Rpb24gaW50ZWdyYXRpb25SZXNvdXJjZUFybihzZXJ2aWNlOiBzdHJpbmcsIGFwaTogc3RyaW5nLCBpbnRlZ3JhdGlvblBhdHRlcm4gPzpzZm4uSW50ZWdyYXRpb25QYXR0ZXJuKTogc3RyaW5nIHtcbiAgaWYgKCEgc2VydmljZSB8fCAhIGFwaSkge1xuICAgIHRocm93IG5ldyBFcnJvcihcIkJvdGggJ3NlcnZpY2UnIGFuZCAnYXBpJyBtdXN0IGJlIHByb3ZpZGVkIHRvIGJ1aWxkIHRoZSByZXNvdXJjZSBBUk4uXCIpO1xuICB9XG4gIHJldHVybiBgYXJuOiR7XG4gICAgQXdzLlBBUlRJVElPTlxuICB9OnN0YXRlczo6OiR7c2VydmljZX06JHthcGl9YCArIChpbnRlZ3JhdGlvblBhdHRlcm4gPyByZXNvdXJjZUFyblN1ZmZpeFtpbnRlZ3JhdGlvblBhdHRlcm5dIDogJycpO1xufVxuZXhwb3J0IGludGVyZmFjZSBUZXh0cmFjdEdlbmVyaWNTeW5jU2ZuVGFza1Byb3BzIGV4dGVuZHMgc2ZuLlRhc2tTdGF0ZUJhc2VQcm9wcyB7XG4gIC8vIFRPRE86IHBhc3MgaW4gbGFtYmRhIG1lbW9yeSBjb25maWdzIGFuZCB0aW1lb3V0cyBhcyBwYXJhbWV0ZXJzIGFzIHdlbGxcbiAgcmVhZG9ubHkgczNPdXRwdXRCdWNrZXQgOiBzdHJpbmc7XG4gIC8qKiBUaGUgcHJlZml4IHRvIHVzZSBmb3IgdGhlIG91dHB1dCBmaWxlcyAqL1xuICByZWFkb25seSBzM091dHB1dFByZWZpeCA6IHN0cmluZztcbiAgLyoqIGxvY2F0aW9uIG9mIGlucHV0IFMzIG9iamVjdHMgLSBpZiBsZWZ0IGVtcHR5IHdpbGwgZ2VuZXJhdGUgcnVsZSBmb3IgczMgYWNjZXNzIHRvIGFsbCBbKl0gKi9cbiAgcmVhZG9ubHkgczNJbnB1dEJ1Y2tldD86IHN0cmluZztcbiAgLyoqIHByZWZpeCBmb3IgaW5wdXQgUzMgb2JqZWN0cyAtIGlmIGxlZnQgZW1wdHkgd2lsbCBnZW5lcmF0ZSBydWxlIGZvciBzMyBhY2Nlc3MgdG8gYWxsIFsqXSAqL1xuICByZWFkb25seSBzM0lucHV0UHJlZml4Pzogc3RyaW5nO1xuICAvKiogVGhlIHByZWZpeCB0byB1c2UgZm9yIHRoZSBvdXRwdXQgZmlsZXMgKi9cbiAgcmVhZG9ubHkgbGFtYmRhTG9nTGV2ZWw/IDogc3RyaW5nO1xuICAvKipub3QgaW1wbGVtZW50ZWQgeWV0ICovXG4gIHJlYWRvbmx5IGVuYWJsZU1vbml0b3Jpbmc/IDogYm9vbGVhbjtcbiAgLyoqIG5vdCBpbXBsZW1lbnRlZCB5ZXQgKi9cbiAgcmVhZG9ubHkgZW5hYmxlRGFzaGJvYXJkPyA6IGJvb2xlYW47XG4gIC8qKiBub3QgaW1wbGVtZW50ZWQgeWV0ICovXG4gIHJlYWRvbmx5IGN1c3RvbUZ1bmN0aW9uPyA6IExhbWJkYUludm9rZTtcbiAgcmVhZG9ubHkgd29ya2Zsb3dUcmFjaW5nRW5hYmxlZD8gOiBib29sZWFuO1xuICAvKiogaG93IGxvbmcgY2FuIHdlIHdhaXQgZm9yIHRoZSBwcm9jZXNzIChkZWZhdWx0IGlzIDQ4IGhvdXJzICg2MCo0OD0yODgwKSkgKi9cbiAgcmVhZG9ubHkgdGV4dHJhY3RTdGF0ZU1hY2hpbmVUaW1lb3V0TWludXRlcz8gOiBudW1iZXI7XG4gIHJlYWRvbmx5IHRleHRyYWN0QVBJPzogJ0dFTkVSSUMnIHwgJ0FOQUxZWkVJRCcgfCAnRVhQRU5TRSc7XG4gIC8qKlxuICAgICAgICogVGhlIEpTT04gaW5wdXQgZm9yIHRoZSBleGVjdXRpb24sIHNhbWUgYXMgdGhhdCBvZiBTdGFydEV4ZWN1dGlvbi5cbiAgICAgICAqXG4gICAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9TdGFydEV4ZWN1dGlvbi5odG1sXG4gICAgICAgKlxuICAgICAgICogQGRlZmF1bHQgLSBUaGUgc3RhdGUgaW5wdXQgKEpTT04gcGF0aCAnJCcpXG4gICAgICAgKi9cbiAgcmVhZG9ubHkgaW5wdXQ/IDogc2ZuLlRhc2tJbnB1dDtcblxuICAvKipcbiAgICAgICAgICAqIFRoZSBuYW1lIG9mIHRoZSBleGVjdXRpb24sIHNhbWUgYXMgdGhhdCBvZiBTdGFydEV4ZWN1dGlvbi5cbiAgICAgICAgICAqXG4gICAgICAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvYXBpcmVmZXJlbmNlL0FQSV9TdGFydEV4ZWN1dGlvbi5odG1sXG4gICAgICAgICAgKlxuICAgICAgICAgICogQGRlZmF1bHQgLSBOb25lXG4gICAgICAgICAgKi9cbiAgcmVhZG9ubHkgbmFtZT8gOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAgICAgICAgKiBQYXNzIHRoZSBleGVjdXRpb24gSUQgZnJvbSB0aGUgY29udGV4dCBvYmplY3QgdG8gdGhlIGV4ZWN1dGlvbiBpbnB1dC5cbiAgICAgICAgICAqIFRoaXMgYWxsb3dzIHRoZSBTdGVwIEZ1bmN0aW9ucyBVSSB0byBsaW5rIGNoaWxkIGV4ZWN1dGlvbnMgZnJvbSBwYXJlbnQgZXhlY3V0aW9ucywgbWFraW5nIGl0IGVhc2llciB0byB0cmFjZSBleGVjdXRpb24gZmxvdyBhY3Jvc3Mgc3RhdGUgbWFjaGluZXMuXG4gICAgICAgICAgKlxuICAgICAgICAgICogSWYgeW91IHNldCB0aGlzIHByb3BlcnR5IHRvIGB0cnVlYCwgdGhlIGBpbnB1dGAgcHJvcGVydHkgbXVzdCBiZSBhbiBvYmplY3QgKHByb3ZpZGVkIGJ5IGBzZm4uVGFza0lucHV0LmZyb21PYmplY3RgKSBvciBvbWl0dGVkIGVudGlyZWx5LlxuICAgICAgICAgICpcbiAgICAgICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9kZy9jb25jZXB0cy1uZXN0ZWQtd29ya2Zsb3dzLmh0bWwjbmVzdGVkLWV4ZWN1dGlvbi1zdGFydGlkXG4gICAgICAgICAgKlxuICAgICAgICAgICogQGRlZmF1bHQgLSBmYWxzZVxuICAgICAgICAgICovXG4gIHJlYWRvbmx5IGFzc29jaWF0ZVdpdGhQYXJlbnQ/IDogYm9vbGVhbjtcbn1cblxuZXhwb3J0IGNsYXNzIFRleHRyYWN0R2VuZXJpY1N5bmNTZm5UYXNrIGV4dGVuZHMgc2ZuLlRhc2tTdGF0ZUJhc2Uge1xuICBwcml2YXRlIHN0YXRpYyByZWFkb25seSBTVVBQT1JURURfSU5URUdSQVRJT05fUEFUVEVSTlMgPSBbXG4gICAgc2ZuLkludGVncmF0aW9uUGF0dGVybi5SRVFVRVNUX1JFU1BPTlNFLFxuICAgIHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUlVOX0pPQixcbiAgICBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLldBSVRfRk9SX1RBU0tfVE9LRU4sXG4gIF07XG5cbiAgcHJvdGVjdGVkIHJlYWRvbmx5IHRhc2tNZXRyaWNzPzogc2ZuLlRhc2tNZXRyaWNzQ29uZmlnO1xuICBwcm90ZWN0ZWQgcmVhZG9ubHkgdGFza1BvbGljaWVzPzogaWFtLlBvbGljeVN0YXRlbWVudFtdO1xuXG4gIHByaXZhdGUgcmVhZG9ubHkgaW50ZWdyYXRpb25QYXR0ZXJuIDogc2ZuLkludGVncmF0aW9uUGF0dGVybjtcbiAgcHVibGljIHN0YXRlTWFjaGluZSA6IHNmbi5JU3RhdGVNYWNoaW5lO1xuICBwdWJsaWMgcHV0T25TUVNMYW1iZGFMb2dHcm91cDpJTG9nR3JvdXA7XG4gIHB1YmxpYyB0ZXh0cmFjdFN5bmNMYW1iZGFMb2dHcm91cDpJTG9nR3JvdXA7XG4gIHB1YmxpYyBkYXNoYm9hcmROYW1lOnN0cmluZztcbiAgcHVibGljIHRleHRyYWN0U3luY1NRUzpzcXMuSVF1ZXVlO1xuICBwdWJsaWMgdmVyc2lvbjpzdHJpbmc7XG4gIHB1YmxpYyB0ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb246bGFtYmRhLklGdW5jdGlvbjtcbiAgcHVibGljIHRleHRyYWN0UHV0T25TUVNGdW5jdGlvbjogbGFtYmRhLklGdW5jdGlvbjtcblxuICBjb25zdHJ1Y3RvcihzY29wZSA6IENvbnN0cnVjdCwgaWQgOiBzdHJpbmcsIHByaXZhdGUgcmVhZG9ubHkgcHJvcHMgOiBUZXh0cmFjdEdlbmVyaWNTeW5jU2ZuVGFza1Byb3BzKSB7XG4gICAgc3VwZXIoc2NvcGUsIGlkLCBwcm9wcyk7XG5cbiAgICB0aGlzLnZlcnNpb24gPSAnMC4wLjEnO1xuICAgIHRoaXMuaW50ZWdyYXRpb25QYXR0ZXJuID0gcHJvcHMuaW50ZWdyYXRpb25QYXR0ZXJuIHx8IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRTtcbiAgICB2YWxpZGF0ZVBhdHRlcm5TdXBwb3J0ZWQodGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4sIFRleHRyYWN0R2VuZXJpY1N5bmNTZm5UYXNrLlNVUFBPUlRFRF9JTlRFR1JBVElPTl9QQVRURVJOUyk7XG5cbiAgICBpZiAodGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4gPT09IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uV0FJVF9GT1JfVEFTS19UT0tFTiAmJiAhc2ZuLkZpZWxkVXRpbHMuY29udGFpbnNUYXNrVG9rZW4ocHJvcHMuaW5wdXQpKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1Rhc2sgVG9rZW4gaXMgcmVxdWlyZWQgaW4gYGlucHV0YCBmb3IgY2FsbGJhY2suIFVzZSBKc29uUGF0aC50YXNrVG9rZW4gdG8gc2V0IHRoZSB0b2tlbi4nKTtcbiAgICB9XG5cbiAgICBpZiAodGhpcy5wcm9wcy5hc3NvY2lhdGVXaXRoUGFyZW50ICYmIHByb3BzLmlucHV0ICYmIHByb3BzLmlucHV0LnR5cGUgIT09IHNmbi5JbnB1dFR5cGUuT0JKRUNUKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ0NvdWxkIG5vdCBlbmFibGUgYGFzc29jaWF0ZVdpdGhQYXJlbnRgIGJlY2F1c2UgYGlucHV0YCBpcyB0YWtlbiBkaXJlY3RseSBmcm9tIGEgSlNPTiBwYXRoLiBVc2UgYHNmbi5UYXNrSW5wdXQuZnJvbU9iamVjdGAgaW5zdGVhZC4nKTtcbiAgICB9XG5cbiAgICBpZiAoIXByb3BzLnMzT3V0cHV0QnVja2V0IHx8ICFwcm9wcy5zM091dHB1dFByZWZpeCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdubyBzM091dHB1dEJ1Y2tldCBhbmQgczNPdXRwdXRQcmVmaXggcGFzc2VkIGluJyk7XG4gICAgfVxuXG4gICAgaWYgKCFwcm9wcy5zM091dHB1dEJ1Y2tldCB8fCAhcHJvcHMuczNPdXRwdXRQcmVmaXgpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignbm8gczNPdXRwdXRCdWNrZXQgYW5kIHMzT3V0cHV0UHJlZml4IHBhc3NlZCBpbicpO1xuICAgIH1cblxuICAgIHZhciB0ZXh0cmFjdEFQSSA9IHByb3BzLnRleHRyYWN0QVBJID09PSB1bmRlZmluZWQgPyAnR0VORVJJQycgOiBwcm9wcy50ZXh0cmFjdEFQSTtcbiAgICB2YXIgdGV4dHJhY3RTdGF0ZU1hY2hpbmVUaW1lb3V0TWludXRlcyA9IHByb3BzLnRleHRyYWN0U3RhdGVNYWNoaW5lVGltZW91dE1pbnV0ZXMgPT09IHVuZGVmaW5lZCA/IDI4ODAgOiBwcm9wcy50ZXh0cmFjdFN0YXRlTWFjaGluZVRpbWVvdXRNaW51dGVzO1xuICAgIHZhciBsYW1iZGFMb2dMZXZlbCA9IHByb3BzLmxhbWJkYUxvZ0xldmVsID09PSB1bmRlZmluZWQgPyAnREVCVUcnIDogcHJvcHMubGFtYmRhTG9nTGV2ZWw7XG5cbiAgICBjb25zdCBkdXJhdGlvbjogbnVtYmVyID0gMTAwO1xuICAgIHRoaXMudGV4dHJhY3RTeW5jU1FTID0gbmV3IHNxcy5RdWV1ZSh0aGlzLCAnU3luY1JlcXVlc3RzJywge1xuICAgICAgdmlzaWJpbGl0eVRpbWVvdXQ6IER1cmF0aW9uLnNlY29uZHMoZHVyYXRpb24pLFxuICAgIH0pO1xuXG4gICAgdGhpcy50ZXh0cmFjdFB1dE9uU1FTRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkRvY2tlckltYWdlRnVuY3Rpb24odGhpcywgJ1B1dE9uU1FTJywge1xuICAgICAgY29kZTogbGFtYmRhLkRvY2tlckltYWdlQ29kZS5mcm9tSW1hZ2VBc3NldChwYXRoLmpvaW4oX19kaXJuYW1lLCAnLi4vbGFtYmRhL3B1dF9vbl9zcXMvJykpLFxuICAgICAgbWVtb3J5U2l6ZTogMTI4LFxuICAgICAgdGltZW91dDogRHVyYXRpb24uc2Vjb25kcygzMCksXG4gICAgICBlbnZpcm9ubWVudDoge1xuICAgICAgICBMT0dfTEVWRUw6IGxhbWJkYUxvZ0xldmVsLFxuICAgICAgICBTUVNfUVVFVUVfVVJMOiB0aGlzLnRleHRyYWN0U3luY1NRUy5xdWV1ZVVybCxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgdGhpcy50ZXh0cmFjdFB1dE9uU1FTRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHsgYWN0aW9uczogWydzcXM6U2VuZE1lc3NhZ2UnXSwgcmVzb3VyY2VzOiBbdGhpcy50ZXh0cmFjdFN5bmNTUVMucXVldWVBcm5dIH0pKTtcblxuICAgIHRoaXMucHV0T25TUVNMYW1iZGFMb2dHcm91cCA9ICg8bGFtYmRhLkZ1bmN0aW9uPiB0aGlzLnRleHRyYWN0UHV0T25TUVNGdW5jdGlvbikubG9nR3JvdXA7XG5cbiAgICBjb25zdCBwdXRPblNRU0Z1bmN0aW9uSW52b2tlID0gbmV3IHRhc2tzLkxhbWJkYUludm9rZSh0aGlzLCAnUHV0T25TUVNGdW5jdGlvbkludm9rZScsIHtcbiAgICAgIGxhbWJkYUZ1bmN0aW9uOiB0aGlzLnRleHRyYWN0UHV0T25TUVNGdW5jdGlvbixcbiAgICB9KTtcblxuXG4gICAgdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24gPSBuZXcgbGFtYmRhLkRvY2tlckltYWdlRnVuY3Rpb24odGhpcywgJ1RleHRyYWN0U3luY0NhbGwnLCB7XG4gICAgICBjb2RlOiBsYW1iZGEuRG9ja2VySW1hZ2VDb2RlLmZyb21JbWFnZUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICcuLi9sYW1iZGEvdGV4dHJhY3Rfc3luYy8nKSksXG4gICAgICBtZW1vcnlTaXplOiA1MTIsXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5zZWNvbmRzKGR1cmF0aW9uKSxcbiAgICAgIGVudmlyb25tZW50OiB7XG4gICAgICAgIFMzX09VVFBVVF9CVUNLRVQ6IHByb3BzLnMzT3V0cHV0QnVja2V0LFxuICAgICAgICBTM19PVVRQVVRfUFJFRklYOiBwcm9wcy5zM091dHB1dFByZWZpeCxcbiAgICAgICAgTE9HX0xFVkVMOiBsYW1iZGFMb2dMZXZlbCxcbiAgICAgICAgU1FTX1FVRVVFX1VSTDogdGhpcy50ZXh0cmFjdFN5bmNTUVMucXVldWVOYW1lLFxuICAgICAgICBURVhUUkFDVF9BUEk6IHRleHRyYWN0QVBJLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICB0aGlzLnRleHRyYWN0U3luY0NhbGxGdW5jdGlvbi5hZGRFdmVudFNvdXJjZShuZXcgU3FzRXZlbnRTb3VyY2UodGhpcy50ZXh0cmFjdFN5bmNTUVMsIHsgYmF0Y2hTaXplOiAxIH0pKTtcbiAgICB0aGlzLnRleHRyYWN0U3luY0NhbGxGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoeyBhY3Rpb25zOiBbJ3RleHRyYWN0OionXSwgcmVzb3VyY2VzOiBbJyonXSB9KSk7XG4gICAgdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ3MzOkdldE9iamVjdCcsICdzMzpMaXN0QnVja2V0JywgJ3MzOlB1dE9iamVjdCcsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbYGFybjphd3M6czM6Ojoke1xuICAgICAgICBwcm9wcy5zM091dHB1dEJ1Y2tldFxuICAgICAgfWAsIGBhcm46YXdzOnMzOjo6JHtcbiAgICAgICAgcHJvcHMuczNPdXRwdXRCdWNrZXRcbiAgICAgIH0vKmAsICcqJ10sXG4gICAgfSkpO1xuICAgIC8vIHRoaXMudGV4dHJhY3RBc3luY0NhbGxGdW5jdGlvbi5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoeyBhY3Rpb25zOiBbJ3NuczoqJ10sIHJlc291cmNlczogW3RoaXMudGV4dHJhY3RBc3luY1NOUy50b3BpY0Fybl0gfSkpO1xuICAgIHRoaXMudGV4dHJhY3RTeW5jTGFtYmRhTG9nR3JvdXA9KDxsYW1iZGEuRnVuY3Rpb24+IHRoaXMudGV4dHJhY3RTeW5jQ2FsbEZ1bmN0aW9uKS5sb2dHcm91cDtcblxuICAgIGNvbnN0IHdvcmtmbG93X2NoYWluID0gc2ZuLkNoYWluLnN0YXJ0KHB1dE9uU1FTRnVuY3Rpb25JbnZva2UpO1xuXG4gICAgdGhpcy5zdGF0ZU1hY2hpbmUgPSBuZXcgc2ZuLlN0YXRlTWFjaGluZSh0aGlzLCAnU3RhdGVNYWNoaW5lJywge1xuICAgICAgZGVmaW5pdGlvbjogd29ya2Zsb3dfY2hhaW4sXG4gICAgICB0aW1lb3V0OiBEdXJhdGlvbi5ob3Vycyh0ZXh0cmFjdFN0YXRlTWFjaGluZVRpbWVvdXRNaW51dGVzKSxcbiAgICAgIHRyYWNpbmdFbmFibGVkOiB0cnVlLFxuICAgIH0pO1xuXG4gICAgdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24uYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgJ3N0YXRlczpTZW5kVGFza0ZhaWx1cmUnLCAnc3RhdGVzOlNlbmRUYXNrU3VjY2VzcycsXG4gICAgICBdLFxuICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICB9KSk7XG4gICAgLy8gPT09PT09PT09XG4gICAgLy8gREFTSEJPQVJEXG4gICAgLy8gPT09PT09PT09XG4gICAgY29uc3QgYXBwTmFtZSA9IHRoaXMubm9kZS50cnlHZXRDb250ZXh0KCdhcHBOYW1lJyk7XG5cbiAgICBjb25zdCBjdXN0b21NZXRyaWNOYW1lc3BhY2UgPSAnVGV4dHJhY3RDb25zdHJ1Y3RHZW5lcmljU3luYyc7XG5cbiAgICAvLyBPUEVSQVRJT05BTFxuICAgIGNvbnN0IGFzeW5jRHVyYXRpb25NZXRyaWNGaWx0ZXIgPSBuZXcgTWV0cmljRmlsdGVyKHRoaXMsIGAke2FwcE5hbWV9LUR1cmF0aW9uRmlsdGVyYCwge1xuICAgICAgbG9nR3JvdXA6ICg8bGFtYmRhLkZ1bmN0aW9uPiB0aGlzLnRleHRyYWN0U3luY0NhbGxGdW5jdGlvbikubG9nR3JvdXAsXG4gICAgICBtZXRyaWNOYW1lc3BhY2U6IGN1c3RvbU1ldHJpY05hbWVzcGFjZSxcbiAgICAgIG1ldHJpY05hbWU6ICdEdXJhdGlvbicsXG4gICAgICBmaWx0ZXJQYXR0ZXJuOiBGaWx0ZXJQYXR0ZXJuLnNwYWNlRGVsaW1pdGVkKCdJTkZPJywgJ3RpbWVzdGFtcCcsICdpZCcsICdtZXNzYWdlJywgJ2R1cmF0aW9uTXMnKVxuICAgICAgICAud2hlcmVTdHJpbmcoJ21lc3NhZ2UnLCAnPScsICd0ZXh0cmFjdF9zeW5jX2dlbmVyaWNfY2FsbF9kdXJhdGlvbl9pbl9tczonKSxcbiAgICAgIG1ldHJpY1ZhbHVlOiAnJGR1cmF0aW9uTXMnLFxuICAgIH0pO1xuICAgIGNvbnN0IGFzeW5jRHVyYXRpb25NZXRyaWM6Y2xvdWR3YXRjaC5JTWV0cmljID0gYXN5bmNEdXJhdGlvbk1ldHJpY0ZpbHRlci5tZXRyaWMoeyBzdGF0aXN0aWM6ICdhdmcnIH0pO1xuXG4gICAgY29uc3Qgc3luY051bWJlclBhZ2VzTWV0cmljRmlsdGVyID0gbmV3IE1ldHJpY0ZpbHRlcih0aGlzLCBgJHthcHBOYW1lfS1OdW1iZXJQYWdlc0ZpbHRlcmAsIHtcbiAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24pLmxvZ0dyb3VwLFxuICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICBtZXRyaWNOYW1lOiAnTnVtYmVyUGFnZXMnLFxuICAgICAgZmlsdGVyUGF0dGVybjogRmlsdGVyUGF0dGVybi5zcGFjZURlbGltaXRlZCgnSU5GTycsICd0aW1lc3RhbXAnLCAnaWQnLCAnbWVzc2FnZScsICdwYWdlcycpXG4gICAgICAgIC53aGVyZVN0cmluZygnbWVzc2FnZScsICc9JywgJ3RleHRyYWN0X3N5bmNfZ2VuZXJpY19udW1iZXJfb2ZfcGFnZXNfcHJvY2Vzc2VkOicpLFxuICAgICAgbWV0cmljVmFsdWU6ICckcGFnZXMnLFxuICAgIH0pO1xuICAgIGNvbnN0IHN5bmNOdW1iZXJQYWdlc01ldHJpYzpjbG91ZHdhdGNoLklNZXRyaWMgPSBzeW5jTnVtYmVyUGFnZXNNZXRyaWNGaWx0ZXIubWV0cmljKHsgc3RhdGlzdGljOiAnc3VtJyB9KTtcblxuICAgIGNvbnN0IHN5bmNOdW1iZXJQYWdlc1NlbmRNZXRyaWNGaWx0ZXIgPSBuZXcgTWV0cmljRmlsdGVyKHRoaXMsIGAke2FwcE5hbWV9LU51bWJlclBhZ2VzU2VuZEZpbHRlcmAsIHtcbiAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24pLmxvZ0dyb3VwLFxuICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICBtZXRyaWNOYW1lOiAnTnVtYmVyUGFnZXMnLFxuICAgICAgZmlsdGVyUGF0dGVybjogRmlsdGVyUGF0dGVybi5zcGFjZURlbGltaXRlZCgnSU5GTycsICd0aW1lc3RhbXAnLCAnaWQnLCAnbWVzc2FnZScsICdwYWdlcycpXG4gICAgICAgIC53aGVyZVN0cmluZygnbWVzc2FnZScsICc9JywgJ3RleHRyYWN0X3N5bmNfZ2VuZXJpY19udW1iZXJfb2ZfcGFnZXNfc2VuZF90b19wcm9jZXNzOicpLFxuICAgICAgbWV0cmljVmFsdWU6ICckcGFnZXMnLFxuICAgIH0pO1xuICAgIGNvbnN0IHN5bmNOdW1iZXJQYWdlc1NlbmRNZXRyaWM6Y2xvdWR3YXRjaC5JTWV0cmljID0gc3luY051bWJlclBhZ2VzU2VuZE1ldHJpY0ZpbHRlci5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuXG4gICAgLy8gVE9ETzogZXhwb3NlIGFsbCBmaWx0ZXJzIGFzIHByb3BlcnRpZXMgb2YgQ29uc3RydWN0LCBzbyB1c2VycyBjYW4gYnVpbGQgdGhlaXIgb3duIG1ldHJpY3NcbiAgICBjb25zdCBhc3luY0pvYkZpbnNoZWRNZXRyaWNGaWx0ZXIgPSBuZXcgTWV0cmljRmlsdGVyKHRoaXMsIGAke2FwcE5hbWV9LUpvYnNGaW5pc2hlZEZpbHRlcmAsIHtcbiAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24pLmxvZ0dyb3VwLFxuICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICBtZXRyaWNOYW1lOiAnSm9ic0ZpbmlzaGVkJyxcbiAgICAgIGZpbHRlclBhdHRlcm46IEZpbHRlclBhdHRlcm4uc3BhY2VEZWxpbWl0ZWQoJ0lORk8nLCAndGltZXN0YW1wJywgJ2lkJywgJ21lc3NhZ2UnLCAncGFnZXMnKVxuICAgICAgICAud2hlcmVTdHJpbmcoJ21lc3NhZ2UnLCAnPScsICd0ZXh0cmFjdF9zeW5jX2dlbmVyaWNfbnVtYmVyX29mX3BhZ2VzX3Byb2Nlc3NlZDonKSxcbiAgICAgIG1ldHJpY1ZhbHVlOiAnMScsXG4gICAgfSk7XG4gICAgY29uc3QgYXN5bmNKb2JGaW5zaGVkTWV0cmljOmNsb3Vkd2F0Y2guSU1ldHJpYyA9IGFzeW5jSm9iRmluc2hlZE1ldHJpY0ZpbHRlci5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuXG4gICAgY29uc3QgYXN5bmNKb2JTdGFydGVkTWV0cmljRmlsdGVyID0gbmV3IE1ldHJpY0ZpbHRlcih0aGlzLCBgJHthcHBOYW1lfS1Kb2JzU3RhcnRlZEZpbHRlcmAsIHtcbiAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24pLmxvZ0dyb3VwLFxuICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICBtZXRyaWNOYW1lOiAnSm9ic1N0YXJ0ZWQnLFxuICAgICAgZmlsdGVyUGF0dGVybjogRmlsdGVyUGF0dGVybi5zcGFjZURlbGltaXRlZCgnSU5GTycsICd0aW1lc3RhbXAnLCAnaWQnLCAnbWVzc2FnZScpXG4gICAgICAgIC53aGVyZVN0cmluZygnbWVzc2FnZScsICc9JywgJ3RleHRyYWN0X2FzeW5jX2dlbmVyaWNfam9iX3N0YXJ0ZWQnKSxcbiAgICAgIG1ldHJpY1ZhbHVlOiAnMScsXG4gICAgfSk7XG4gICAgY29uc3QgYXN5bmNKb2JTdGFydGVkTWV0cmljOmNsb3Vkd2F0Y2guSU1ldHJpYyA9IGFzeW5jSm9iU3RhcnRlZE1ldHJpY0ZpbHRlci5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuXG4gICAgY29uc3Qgc3luY1RpbWVkT3V0TWV0cmljRmlsdGVyID0gbmV3IE1ldHJpY0ZpbHRlcih0aGlzLCBgJHthcHBOYW1lfS1UaW1lZE91dEZpbHRlcmAsIHtcbiAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24pLmxvZ0dyb3VwLFxuICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICBtZXRyaWNOYW1lOiAnRHVyYXRpb25TZWNvbmRzJyxcbiAgICAgIGZpbHRlclBhdHRlcm46IEZpbHRlclBhdHRlcm4uc3BhY2VEZWxpbWl0ZWQoJ0lORk8nLCAndGltZXN0YW1wJywgJ2lkJywgJ21lc3NhZ2UxJywgJ2R1cmF0aW9uUycsICdtZXNzYWdlMicpXG4gICAgICAgIC53aGVyZVN0cmluZygnbWVzc2FnZTEnLCAnPScsICdUYXNrIHRpbWVkIG91dCBhZnRlcicpLndoZXJlU3RyaW5nKCdtZXNzYWdlMicsICc9JywgJ3NlY29uZHMnKSxcbiAgICAgIG1ldHJpY1ZhbHVlOiAnJGR1cmF0aW9uUycsXG4gICAgfSk7XG4gICAgY29uc3Qgc3luY1RpbWVkT3V0TWV0cmljOmNsb3Vkd2F0Y2guSU1ldHJpYyA9IHN5bmNUaW1lZE91dE1ldHJpY0ZpbHRlci5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuICAgIC8vIE9QRVJBVElPTkFMIFNUT1BcblxuICAgIC8vIENBTENVQVRFRCBPUEVSQVRJT05BTCBNRVRSSUNTXG5cbiAgICBjb25zdCBwYWdlc1BlclNlY29uZCA9IG5ldyBjbG91ZHdhdGNoLk1hdGhFeHByZXNzaW9uKHtcbiAgICAgIGV4cHJlc3Npb246ICdwYWdlcyAvIChkdXJhdGlvbiAvIDEwMDApJyxcbiAgICAgIHVzaW5nTWV0cmljczoge1xuICAgICAgICBwYWdlczogc3luY051bWJlclBhZ2VzTWV0cmljLFxuICAgICAgICBkdXJhdGlvbjogYXN5bmNEdXJhdGlvbk1ldHJpYyxcbiAgICAgIH0sXG4gICAgfSk7XG4gICAgY29uc3Qgb3BlbkpvYnMgPSBuZXcgY2xvdWR3YXRjaC5NYXRoRXhwcmVzc2lvbih7XG4gICAgICBleHByZXNzaW9uOiAnc3RhcnRlZEpvYnMgLSBmaW5pc2hlZEpvYnMnLFxuICAgICAgdXNpbmdNZXRyaWNzOiB7XG4gICAgICAgIHN0YXJ0ZWRKb2JzOiBhc3luY0pvYlN0YXJ0ZWRNZXRyaWMsXG4gICAgICAgIGZpbmlzaGVkSm9iczogYXN5bmNKb2JGaW5zaGVkTWV0cmljLFxuICAgICAgfSxcbiAgICB9KTtcbiAgICAvLyBDQUxDVUFURUQgT1BFUkFUSU9OQUwgTUVUUklDUyBTVE9QXG5cbiAgICBjb25zdCBlcnJvckZpbHRlck1ldHJpYyA9IG5ldyBNZXRyaWNGaWx0ZXIodGhpcywgYCR7YXBwTmFtZX0tRXJyb3JGaWx0ZXJgLCB7XG4gICAgICBsb2dHcm91cDogKDxsYW1iZGEuRnVuY3Rpb24+IHRoaXMudGV4dHJhY3RTeW5jQ2FsbEZ1bmN0aW9uKS5sb2dHcm91cCxcbiAgICAgIG1ldHJpY05hbWVzcGFjZTogY3VzdG9tTWV0cmljTmFtZXNwYWNlLFxuICAgICAgbWV0cmljTmFtZTogJ0Vycm9ycycsXG4gICAgICBmaWx0ZXJQYXR0ZXJuOiBGaWx0ZXJQYXR0ZXJuLmFueVRlcm0oJ0VSUk9SJywgJ0Vycm9yJywgJ2Vycm9yJyksXG4gICAgICBtZXRyaWNWYWx1ZTogJzEnLFxuICAgIH0pO1xuXG4gICAgY29uc3QgbGltaXRFeGNlZWRlZEV4Y2VwdGlvbkZpbHRlck1ldHJpYyA9IG5ldyBNZXRyaWNGaWx0ZXIodGhpcywgYCR7YXBwTmFtZX0tbGltaXRFeGNlZWRlZEV4Y2VwdGlvbkZpbHRlcmAsIHtcbiAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24pLmxvZ0dyb3VwLFxuICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICBtZXRyaWNOYW1lOiAnTGltaXRFeGNlZWRlZEV4Y2VwdGlvbicsXG4gICAgICBmaWx0ZXJQYXR0ZXJuOiBGaWx0ZXJQYXR0ZXJuLmFueVRlcm0oJ3RleHRyYWN0LmV4Y2VwdGlvbnMuTGltaXRFeGNlZWRlZEV4Y2VwdGlvbicpLFxuICAgICAgbWV0cmljVmFsdWU6ICcxJyxcbiAgICB9KTtcbiAgICBjb25zdCB0aHJvdHRsaW5nRXhjZXB0aW9uID0gbmV3IE1ldHJpY0ZpbHRlcih0aGlzLCBgJHthcHBOYW1lfS10aHJvdHRsaW5nRXhjZXB0aW9uRmlsdGVyYCwge1xuICAgICAgbG9nR3JvdXA6ICg8bGFtYmRhLkZ1bmN0aW9uPiB0aGlzLnRleHRyYWN0U3luY0NhbGxGdW5jdGlvbikubG9nR3JvdXAsXG4gICAgICBtZXRyaWNOYW1lc3BhY2U6IGN1c3RvbU1ldHJpY05hbWVzcGFjZSxcbiAgICAgIG1ldHJpY05hbWU6ICdUaHJvdHRsaW5nRXhjZXB0aW9uJyxcbiAgICAgIGZpbHRlclBhdHRlcm46IEZpbHRlclBhdHRlcm4uYW55VGVybSgndGV4dHJhY3QuZXhjZXB0aW9ucy5UaHJvdHRsaW5nRXhjZXB0aW9uJyksXG4gICAgICBtZXRyaWNWYWx1ZTogJzEnLFxuICAgIH0pO1xuXG4gICAgY29uc3QgcHJvdmlzaW9uZWRUaHJvdWdocHV0RXhjZWVkZWRFeGNlcHRpb24gPSBuZXcgTWV0cmljRmlsdGVyKHRoaXMsIGAke2FwcE5hbWV9LXByb3Zpc2lvbmVkVGhyb3VnaHB1dEZpbHRlcmAsIHtcbiAgICAgIGxvZ0dyb3VwOiAoPGxhbWJkYS5GdW5jdGlvbj4gdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24pLmxvZ0dyb3VwLFxuICAgICAgbWV0cmljTmFtZXNwYWNlOiBjdXN0b21NZXRyaWNOYW1lc3BhY2UsXG4gICAgICBtZXRyaWNOYW1lOiAnUHJvdmlzaW9uZWRUaHJvdWdocHV0RXhjZWVkZWRFeGNlcHRpb24nLFxuICAgICAgZmlsdGVyUGF0dGVybjogRmlsdGVyUGF0dGVybi5hbnlUZXJtKCd0ZXh0cmFjdC5leGNlcHRpb25zLlByb3Zpc2lvbmVkVGhyb3VnaHB1dEV4Y2VlZGVkRXhjZXB0aW9uJyksXG4gICAgICBtZXRyaWNWYWx1ZTogJzEnLFxuICAgIH0pO1xuICAgIGNvbnN0IGludGVybmFsU2VydmVyRXJyb3IgPSBuZXcgTWV0cmljRmlsdGVyKHRoaXMsIGAke2FwcE5hbWV9LWludGVybmFsU2VydmVyRXJyb3JGaWx0ZXJgLCB7XG4gICAgICBsb2dHcm91cDogKDxsYW1iZGEuRnVuY3Rpb24+IHRoaXMudGV4dHJhY3RTeW5jQ2FsbEZ1bmN0aW9uKS5sb2dHcm91cCxcbiAgICAgIG1ldHJpY05hbWVzcGFjZTogY3VzdG9tTWV0cmljTmFtZXNwYWNlLFxuICAgICAgbWV0cmljTmFtZTogJ0ludGVybmFsU2VydmVyRXJyb3InLFxuICAgICAgZmlsdGVyUGF0dGVybjogRmlsdGVyUGF0dGVybi5hbnlUZXJtKCd0ZXh0cmFjdC5leGNlcHRpb25zLkludGVybmFsU2VydmVyRXJyb3InKSxcbiAgICAgIG1ldHJpY1ZhbHVlOiAnMScsXG4gICAgfSk7XG5cbiAgICBjb25zdCBlcnJvck1ldHJpYzpjbG91ZHdhdGNoLklNZXRyaWMgPSBlcnJvckZpbHRlck1ldHJpYy5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuICAgIGNvbnN0IGxpbWl0RXhjZWVkZWRNZXRyaWM6Y2xvdWR3YXRjaC5JTWV0cmljID0gbGltaXRFeGNlZWRlZEV4Y2VwdGlvbkZpbHRlck1ldHJpYy5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuICAgIGNvbnN0IHRocm90dGxpbmdNZXRyaWM6Y2xvdWR3YXRjaC5JTWV0cmljID0gdGhyb3R0bGluZ0V4Y2VwdGlvbi5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuICAgIGNvbnN0IHByb3Zpc2lvbmVkVGhyb3VnaHB1dE1ldHJpYzpjbG91ZHdhdGNoLklNZXRyaWMgPSBwcm92aXNpb25lZFRocm91Z2hwdXRFeGNlZWRlZEV4Y2VwdGlvbi5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuICAgIGNvbnN0IGludGVybmFsU2VydmVyRXJyb3JNZXRyaWM6Y2xvdWR3YXRjaC5JTWV0cmljID0gaW50ZXJuYWxTZXJ2ZXJFcnJvci5tZXRyaWMoeyBzdGF0aXN0aWM6ICdzdW0nIH0pO1xuXG4gICAgY29uc3QgdGV4dHJhY3RTdGFydERvY3VtZW50VGV4dFRocm90dGxlOmNsb3Vkd2F0Y2guSU1ldHJpYz0gbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9UZXh0cmFjdCcsXG4gICAgICBtZXRyaWNOYW1lOiAnVGhyb3R0bGVkQ291bnQnLFxuICAgICAgZGltZW5zaW9uc01hcDoge1xuICAgICAgICBPcGVyYXRpb246ICdTdGFydERvY3VtZW50VGV4dERldGVjdGlvbicsXG4gICAgICB9LFxuICAgICAgc3RhdGlzdGljOiAnc3VtJyxcbiAgICB9KTtcbiAgICAvLyBEQVNIQk9BUkQgTEFNQkRBXG5cbiAgICBjb25zdCBkYXNoYm9hcmRXaWR0aD0yNDtcbiAgICAvLyBjb25zdCB3aWRnZXRTdGFuZGFyZEhlaWdodD05O1xuXG4gICAgLy8gREFTSEJPQVJEXG4gICAgY29uc3QgZGFzaGJvYXJkID0gbmV3IGNsb3Vkd2F0Y2guRGFzaGJvYXJkKHRoaXMsICcke2FwcE5hbWV9LVRlc3REYXNoYm9hcmQnLCB7XG4gICAgICBlbmQ6ICdlbmQnLFxuICAgICAgcGVyaW9kT3ZlcnJpZGU6IGNsb3Vkd2F0Y2guUGVyaW9kT3ZlcnJpZGUuQVVUTyxcbiAgICAgIHN0YXJ0OiAnc3RhcnQnLFxuICAgICAgd2lkZ2V0czogW1xuICAgICAgICBbXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guQ29sdW1uKG5ldyBjbG91ZHdhdGNoLlRleHRXaWRnZXQoeyBtYXJrZG93bjogJyMgT3BlcmF0aW9uYWwgRGF0YSBSb3cgd2lkZ2V0cycsIHdpZHRoOiBkYXNoYm9hcmRXaWR0aCB9KSksXG4gICAgICAgIF0sXG4gICAgICAgIFtcbiAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5Db2x1bW4obmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ1BhZ2VzUGVyU2Vjb25kJywgbGVmdDogW3BhZ2VzUGVyU2Vjb25kXSwgd2lkdGg6IE1hdGguZmxvb3IoZGFzaGJvYXJkV2lkdGgvMikgfSkpLFxuICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkNvbHVtbihuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnT3BlbkpvYnMnLCBsZWZ0OiBbb3BlbkpvYnNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC8yKSB9KSksXG4gICAgICAgIF0sXG4gICAgICAgIFtcbiAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5Db2x1bW4obmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ0R1cmF0aW9uJywgbGVmdDogW2FzeW5jRHVyYXRpb25NZXRyaWNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC80KSB9KSksXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guQ29sdW1uKG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdOdW1iZXJQYWdlc1Byb2Nlc3NlZCcsIGxlZnQ6IFtzeW5jTnVtYmVyUGFnZXNNZXRyaWNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC80KSB9KSksXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guQ29sdW1uKG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdOdW1iZXJQYWdlc1NlbmRUb1Byb2Nlc3MnLCBsZWZ0OiBbc3luY051bWJlclBhZ2VzU2VuZE1ldHJpY10sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzQpIH0pKSxcbiAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5Db2x1bW4obmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ0pvYnNTdGFydGVkQW5kRmluaXNoZWQnLCBsZWZ0OiBbYXN5bmNKb2JGaW5zaGVkTWV0cmljXSwgcmlnaHQ6IFthc3luY0pvYlN0YXJ0ZWRNZXRyaWNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC80KSB9KSksXG4gICAgICAgIF0sXG4gICAgICAgIFtcbiAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5Db2x1bW4obmV3IGNsb3Vkd2F0Y2guVGV4dFdpZGdldCh7IG1hcmtkb3duOiAnIyBBc3luYyBUZXh0cmFjdCBFeGNlcHRpb25zIFJvdycsIHdpZHRoOiBkYXNoYm9hcmRXaWR0aCB9KSksXG4gICAgICAgIF0sXG4gICAgICAgIFtcbiAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnRXJyb3JzJywgbGVmdDogW2Vycm9yTWV0cmljXSwgd2lkdGg6IE1hdGguZmxvb3IoZGFzaGJvYXJkV2lkdGgvNSkgfSksXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ0xpbWl0RXhjZWVkZWQnLCBsZWZ0OiBbbGltaXRFeGNlZWRlZE1ldHJpY10sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzUpIH0pLFxuICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdUaHJvdHRsaW5nJywgbGVmdDogW3Rocm90dGxpbmdNZXRyaWNdLCB3aWR0aDogTWF0aC5mbG9vcihkYXNoYm9hcmRXaWR0aC81KSB9KSxcbiAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnUHJvdmlzaW9uZWRUaHJvdWdwdXQnLCBsZWZ0OiBbcHJvdmlzaW9uZWRUaHJvdWdocHV0TWV0cmljXSwgd2lkdGg6IE1hdGguZmxvb3IoZGFzaGJvYXJkV2lkdGgvNSkgfSksXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ0ludGVybmFsU2VydmVyRXJyb3InLCBsZWZ0OiBbaW50ZXJuYWxTZXJ2ZXJFcnJvck1ldHJpY10sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzUpIH0pLFxuICAgICAgICBdLFxuICAgICAgICBbXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guVGV4dFdpZGdldCh7IG1hcmtkb3duOiAnIyBUZXh0cmFjdCcsIHdpZHRoOiBkYXNoYm9hcmRXaWR0aCB9KSxcbiAgICAgICAgXSxcbiAgICAgICAgW1xuICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdUZXh0cmFjdC1TdGFydERldGVjdFRleHQtVGhyb3R0bGVkQ291bnQnLCBsZWZ0OiBbdGV4dHJhY3RTdGFydERvY3VtZW50VGV4dFRocm90dGxlXSB9KSxcbiAgICAgICAgXSxcbiAgICAgICAgW1xuICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLlRleHRXaWRnZXQoeyBtYXJrZG93bjogJyMgdGV4dHJhY3RTeW5jQ2FsbEZ1bmN0aW9uJywgd2lkdGg6IGRhc2hib2FyZFdpZHRoIH0pLFxuICAgICAgICBdLFxuICAgICAgICBbXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ1N5bmMtRnVuY3Rpb24tRXJyb3JzJywgbGVmdDogW3RoaXMudGV4dHJhY3RTeW5jQ2FsbEZ1bmN0aW9uLm1ldHJpY0Vycm9ycygpXSwgd2lkdGg6IE1hdGguZmxvb3IoZGFzaGJvYXJkV2lkdGgvNCkgfSksXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ1N5bmMtRnVuY3Rpb24tSW52b2NhdGlvbnMnLCBsZWZ0OiBbdGhpcy50ZXh0cmFjdFN5bmNDYWxsRnVuY3Rpb24ubWV0cmljSW52b2NhdGlvbnMoKV0sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzQpIH0pLFxuICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdTeW5jLUZ1bmN0aW9uLVRocm90dGxlcycsIGxlZnQ6IFt0aGlzLnRleHRyYWN0U3luY0NhbGxGdW5jdGlvbi5tZXRyaWNUaHJvdHRsZXMoKV0sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzQpIH0pLFxuICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdTeW5jLUZ1bmN0aW9uLVRpbWVkT3V0JywgbGVmdDogW3N5bmNUaW1lZE91dE1ldHJpY10sIHdpZHRoOiBNYXRoLmZsb29yKGRhc2hib2FyZFdpZHRoLzQpIH0pLFxuICAgICAgICBdLFxuICAgICAgICBbXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guVGV4dFdpZGdldCh7IG1hcmtkb3duOiAnIyBTUVMgJywgd2lkdGg6IGRhc2hib2FyZFdpZHRoIH0pLFxuICAgICAgICBdLFxuICAgICAgICBbXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ3Nxc051bWJlck1lc3NhZ2VzU2VudCcsIGxlZnQ6IFt0aGlzLnRleHRyYWN0U3luY1NRUy5tZXRyaWNOdW1iZXJPZk1lc3NhZ2VzU2VudCgpXSB9KSxcbiAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnc3FzTnVtYmVyTWVzc2FnZXNEZWxldGVkJywgbGVmdDogW3RoaXMudGV4dHJhY3RTeW5jU1FTLm1ldHJpY051bWJlck9mTWVzc2FnZXNEZWxldGVkKCldIH0pLFxuICAgICAgICAgIG5ldyBjbG91ZHdhdGNoLkdyYXBoV2lkZ2V0KHsgdGl0bGU6ICdzcXNOdW1iZXJNZXNzYWdlc1Zpc2libGUnLCBsZWZ0OiBbdGhpcy50ZXh0cmFjdFN5bmNTUVMubWV0cmljQXBwcm94aW1hdGVOdW1iZXJPZk1lc3NhZ2VzVmlzaWJsZSgpXSB9KSxcbiAgICAgICAgICBuZXcgY2xvdWR3YXRjaC5HcmFwaFdpZGdldCh7IHRpdGxlOiAnc3FzTnVtYmVyTWVzc2FnZXNOb3RWaXNpYmxlJywgbGVmdDogW3RoaXMudGV4dHJhY3RTeW5jU1FTLm1ldHJpY0FwcHJveGltYXRlTnVtYmVyT2ZNZXNzYWdlc05vdFZpc2libGUoKV0gfSksXG4gICAgICAgICAgbmV3IGNsb3Vkd2F0Y2guR3JhcGhXaWRnZXQoeyB0aXRsZTogJ3Nxc0FnZU9sZGVzdE1lc3NhZ2UnLCBsZWZ0OiBbdGhpcy50ZXh0cmFjdFN5bmNTUVMubWV0cmljQXBwcm94aW1hdGVBZ2VPZk9sZGVzdE1lc3NhZ2UoKV0gfSksXG4gICAgICAgIF0sXG4gICAgICBdLFxuICAgIH0pO1xuXG4gICAgdGhpcy5kYXNoYm9hcmROYW1lPWRhc2hib2FyZC5kYXNoYm9hcmROYW1lO1xuICAgIC8vIEVORCBEQVNIQk9BUkRcbiAgICB0aGlzLnRhc2tQb2xpY2llcyA9IHRoaXMuY3JlYXRlU2NvcGVkQWNjZXNzUG9saWN5KCk7XG4gIH1cbiAgLyoqXG4gICAgICAgKiBAaW50ZXJuYWxcbiAgICAgICAqL1xuICBwcm90ZWN0ZWQgX3JlbmRlclRhc2soKTogYW55IHtcbiAgICAvLyBzdWZmaXggb2YgJzoyJyBpbmRpY2F0ZXMgdGhhdCB0aGUgb3V0cHV0IG9mIHRoZSBuZXN0ZWQgc3RhdGUgbWFjaGluZSBzaG91bGQgYmUgSlNPTlxuICAgIC8vIHN1ZmZpeCBpcyBvbmx5IGFwcGxpY2FibGUgd2hlbiB3YWl0aW5nIGZvciBhIG5lc3RlZCBzdGF0ZSBtYWNoaW5lIHRvIGNvbXBsZXRlIChSVU5fSk9CKVxuICAgIC8vIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9zdGVwLWZ1bmN0aW9ucy9sYXRlc3QvZGcvY29ubmVjdC1zdGVwZnVuY3Rpb25zLmh0bWxcbiAgICBjb25zdCBzdWZmaXggPSB0aGlzLmludGVncmF0aW9uUGF0dGVybiA9PT0gc2ZuLkludGVncmF0aW9uUGF0dGVybi5SVU5fSk9CID8gJzoyJyA6ICcnO1xuICAgIGxldCBpbnB1dDogYW55O1xuICAgIGlmICh0aGlzLnByb3BzLmFzc29jaWF0ZVdpdGhQYXJlbnQpIHtcbiAgICAgIGNvbnN0IGFzc29jaWF0ZVdpdGhQYXJlbnRFbnRyeSA9IHtcbiAgICAgICAgQVdTX1NURVBfRlVOQ1RJT05TX1NUQVJURURfQllfRVhFQ1VUSU9OX0lEOiBzZm4uSnNvblBhdGguc3RyaW5nQXQoJyQkLkV4ZWN1dGlvbi5JZCcpLFxuICAgICAgfTtcbiAgICAgIGlucHV0ID0gdGhpcy5wcm9wcy5pbnB1dCA/IHtcbiAgICAgICAgLi4udGhpcy5wcm9wcy5pbnB1dC52YWx1ZSxcbiAgICAgICAgLi4uIGFzc29jaWF0ZVdpdGhQYXJlbnRFbnRyeSxcbiAgICAgIH0gOiBhc3NvY2lhdGVXaXRoUGFyZW50RW50cnk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGlucHV0ID0gdGhpcy5wcm9wcy5pbnB1dCA/IHRoaXMucHJvcHMuaW5wdXQudmFsdWUgOiBzZm4uVGFza0lucHV0LmZyb21Kc29uUGF0aEF0KCckJykudmFsdWU7XG4gICAgfVxuXG5cbiAgICByZXR1cm4ge1xuICAgICAgUmVzb3VyY2U6IGAke1xuICAgICAgICBpbnRlZ3JhdGlvblJlc291cmNlQXJuKCdzdGF0ZXMnLCAnc3RhcnRFeGVjdXRpb24nLCB0aGlzLmludGVncmF0aW9uUGF0dGVybilcbiAgICAgIH0ke3N1ZmZpeH1gLFxuICAgICAgUGFyYW1ldGVyczogc2ZuLkZpZWxkVXRpbHMucmVuZGVyT2JqZWN0KFxuICAgICAgICB7XG4gICAgICAgICAgSW5wdXQ6IGlucHV0LFxuICAgICAgICAgIFN0YXRlTWFjaGluZUFybjogdGhpcy5zdGF0ZU1hY2hpbmUuc3RhdGVNYWNoaW5lQXJuLFxuICAgICAgICAgIE5hbWU6IHRoaXMucHJvcHMubmFtZSxcbiAgICAgICAgfSxcbiAgICAgICksXG4gICAgfTtcbiAgfVxuICAvKipcbiAgICAgICAqIEFzIFN0YXRlTWFjaGluZUFybiBpcyBleHRyYWN0ZWQgYXV0b21hdGljYWxseSBmcm9tIHRoZSBzdGF0ZSBtYWNoaW5lIG9iamVjdCBpbmNsdWRlZCBpbiB0aGUgY29uc3RydWN0b3IsXG4gICAgICAgKlxuICAgICAgICogdGhlIHNjb3BlZCBhY2Nlc3MgcG9saWN5IHNob3VsZCBiZSBnZW5lcmF0ZWQgYWNjb3JkaW5nbHkuXG4gICAgICAgKlxuICAgICAgICogVGhpcyBtZWFucyB0aGUgYWN0aW9uIG9mIFN0YXJ0RXhlY3V0aW9uIHNob3VsZCBiZSByZXN0cmljdGVkIG9uIHRoZSBnaXZlbiBzdGF0ZSBtYWNoaW5lLCBpbnN0ZWFkIG9mIGJlaW5nIGdyYW50ZWQgdG8gYWxsIHRoZSByZXNvdXJjZXMgKCopLlxuICAgICAgICovXG4gIHByaXZhdGUgY3JlYXRlU2NvcGVkQWNjZXNzUG9saWN5KCk6IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXSB7XG4gICAgY29uc3Qgc3RhY2sgPSBTdGFjay5vZih0aGlzKTtcblxuICAgIGNvbnN0IHBvbGljeVN0YXRlbWVudHMgPSBbXG4gICAgICBuZXcgaWFtLlBvbGljeVN0YXRlbWVudChcbiAgICAgICAge1xuICAgICAgICAgIGFjdGlvbnM6IFsnc3RhdGVzOlN0YXJ0RXhlY3V0aW9uJ10sXG4gICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5zdGF0ZU1hY2hpbmUuc3RhdGVNYWNoaW5lQXJuXSxcbiAgICAgICAgfSxcbiAgICAgICksXG4gICAgXTtcblxuICAgIC8vIFN0ZXAgRnVuY3Rpb25zIHVzZSBDbG91ZCBXYXRjaCBtYW5hZ2VkIHJ1bGVzIHRvIGRlYWwgd2l0aCBzeW5jaHJvbm91cyB0YXNrcy5cbiAgICBpZiAodGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4gPT09IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUlVOX0pPQikge1xuICAgICAgcG9saWN5U3RhdGVtZW50cy5wdXNoKG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgYWN0aW9uczogW1xuICAgICAgICAgICdzdGF0ZXM6RGVzY3JpYmVFeGVjdXRpb24nLCAnc3RhdGVzOlN0b3BFeGVjdXRpb24nLFxuICAgICAgICBdLFxuICAgICAgICAvLyBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vc3RlcC1mdW5jdGlvbnMvbGF0ZXN0L2RnL2NvbmNlcHQtY3JlYXRlLWlhbS1hZHZhbmNlZC5odG1sI2NvbmNlcHQtY3JlYXRlLWlhbS1hZHZhbmNlZC1leGVjdXRpb25cbiAgICAgICAgcmVzb3VyY2VzOiBbXG4gICAgICAgICAgc3RhY2suZm9ybWF0QXJuKFxuICAgICAgICAgICAge1xuICAgICAgICAgICAgICBzZXJ2aWNlOiAnc3RhdGVzJyxcbiAgICAgICAgICAgICAgcmVzb3VyY2U6ICdleGVjdXRpb24nLFxuICAgICAgICAgICAgICBhcm5Gb3JtYXQ6IEFybkZvcm1hdC5DT0xPTl9SRVNPVVJDRV9OQU1FLFxuICAgICAgICAgICAgICByZXNvdXJjZU5hbWU6IGAke1xuICAgICAgICAgICAgICAgIHN0YWNrLnNwbGl0QXJuKHRoaXMuc3RhdGVNYWNoaW5lLnN0YXRlTWFjaGluZUFybiwgQXJuRm9ybWF0LkNPTE9OX1JFU09VUkNFX05BTUUpLnJlc291cmNlTmFtZVxuICAgICAgICAgICAgICB9KmAsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICksXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG5cbiAgICAgIHBvbGljeVN0YXRlbWVudHMucHVzaChuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAnZXZlbnRzOlB1dFRhcmdldHMnLCAnZXZlbnRzOlB1dFJ1bGUnLCAnZXZlbnRzOkRlc2NyaWJlUnVsZScsXG4gICAgICAgIF0sXG4gICAgICAgIHJlc291cmNlczogW1xuICAgICAgICAgIHN0YWNrLmZvcm1hdEFybihcbiAgICAgICAgICAgIHtcbiAgICAgICAgICAgICAgc2VydmljZTogJ2V2ZW50cycsXG4gICAgICAgICAgICAgIHJlc291cmNlOiAncnVsZScsXG4gICAgICAgICAgICAgIHJlc291cmNlTmFtZTogJ1N0ZXBGdW5jdGlvbnNHZXRFdmVudHNGb3JTdGVwRnVuY3Rpb25zRXhlY3V0aW9uUnVsZScsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICksXG4gICAgICAgIF0sXG4gICAgICB9KSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIHBvbGljeVN0YXRlbWVudHM7XG4gIH1cbn0iXX0=