"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.SonarCodeScanner = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_codebuild_1 = require("aws-cdk-lib/aws-codebuild");
const aws_events_1 = require("aws-cdk-lib/aws-events");
const aws_events_targets_1 = require("aws-cdk-lib/aws-events-targets");
const aws_iam_1 = require("aws-cdk-lib/aws-iam");
const aws_secretsmanager_1 = require("aws-cdk-lib/aws-secretsmanager");
const constructs_1 = require("constructs");
const sonarqube_commands_1 = require("./sonarqube-commands");
const unpackSourceAndArtifacts = (includeGlobsForScan) => [
    'export BUILT_ARTIFACT_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r \'.builds[0].secondaryArtifacts[] | select(.artifactIdentifier == "Synth__") | .location\' | awk \'{sub("arn:aws:s3:::","s3://")}1\' $1`',
    "export SYNTH_SOURCE_URI=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r '.builds[0].sourceVersion' | awk '{sub(\"arn:aws:s3:::\",\"s3://\")}1' $1`",
    "aws s3 cp $SYNTH_SOURCE_URI source.zip",
    "aws s3 cp $BUILT_ARTIFACT_URI built.zip",
    "unzip source.zip -d src",
    "unzip built.zip -d built",
    "rm source.zip built.zip",
    `rsync -a built/* src --include="*/" ${includeGlobsForScan
        ? includeGlobsForScan.map((g) => `--include ${g}`).join(" ")
        : ""} --include="**/coverage/**" --include="**/cdk.out/**" --exclude="**/node_modules/**/*" --exclude="**/.env/**" --exclude="*" --prune-empty-dirs`,
];
const owaspScan = () => `npx owasp-dependency-check --format HTML --out src/reports --exclude '**/.git/**/*' --scan src --enableExperimental --bin /tmp/dep-check --disableRetireJS`;
const cfnNagScan = (cdkOutDir, cfnNagIgnorePath) => cdkOutDir
    ? `cfn_nag ${cfnNagIgnorePath ? `--deny-list-path=${cfnNagIgnorePath}` : ""} built/${cdkOutDir}/**/*.template.json --output-format=json > src/reports/cfn-nag-report.json`
    : 'echo "skipping cfn_nag as no cdkOutDir was specified.';
class SonarCodeScanner extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const sonarQubeToken = new aws_secretsmanager_1.Secret(this, "SonarQubeToken");
        const synthBuildProject = aws_codebuild_1.Project.fromProjectArn(this, "SynthBuildProject", props.synthBuildArn);
        const validationProject = new aws_codebuild_1.Project(this, "ValidationProject", {
            environment: {
                buildImage: aws_codebuild_1.LinuxBuildImage.STANDARD_5_0,
            },
            environmentVariables: {
                SONARQUBE_TOKEN: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.SECRETS_MANAGER,
                    value: sonarQubeToken.secretArn,
                },
                SONARQUBE_ENDPOINT: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeEndpoint,
                },
                PROJECT_NAME: {
                    type: aws_codebuild_1.BuildEnvironmentVariableType.PLAINTEXT,
                    value: props.sonarqubeProjectName,
                },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: "0.2",
                env: {
                    shell: "bash",
                },
                phases: {
                    install: {
                        commands: ["npm install -g aws-cdk", "gem install cfn-nag"],
                    },
                    build: {
                        commands: [
                            "export RESOLVED_SOURCE_VERSION=`aws codebuild batch-get-builds --ids $SYNTH_BUILD_ID | jq -r '.builds[0].resolvedSourceVersion'`",
                            ...unpackSourceAndArtifacts(props.includeGlobsForScan),
                            ...sonarqube_commands_1.createSonarqubeProject(props),
                            "mkdir -p src/reports",
                            owaspScan(),
                            cfnNagScan(props.cdkOutDir, props.cfnNagIgnorePath),
                            "cd src",
                            sonarqube_commands_1.sonarqubeScanner(props.excludeGlobsForScan),
                            ...sonarqube_commands_1.generateSonarqubeReports(),
                            ...(props.preArchiveCommands || []),
                        ],
                    },
                },
            }),
        });
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["codebuild:BatchGetBuilds"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [synthBuildProject.projectArn],
        }));
        validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            actions: ["s3:GetObject*"],
            effect: aws_iam_1.Effect.ALLOW,
            resources: [props.artifactBucketArn, `${props.artifactBucketArn}/**`],
        }));
        props.artifactBucketKeyArn &&
            validationProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
                actions: ["kms:Decrypt", "kms:DescribeKey"],
                effect: aws_iam_1.Effect.ALLOW,
                resources: [props.artifactBucketKeyArn],
            }));
        synthBuildProject.onBuildSucceeded("OnSynthSuccess", {
            target: new aws_events_targets_1.CodeBuildProject(validationProject, {
                event: aws_events_1.RuleTargetInput.fromObject({
                    environmentVariablesOverride: [
                        {
                            name: "SYNTH_BUILD_ID",
                            type: "PLAINTEXT",
                            value: aws_events_1.EventField.fromPath("$.detail.build-id"),
                        },
                    ],
                }),
            }),
        });
        new aws_cdk_lib_1.CfnOutput(this, "SonarqubeSecretArn", {
            exportName: "SonarqubeSecretArn",
            value: sonarQubeToken.secretArn,
        });
    }
}
exports.SonarCodeScanner = SonarCodeScanner;
_a = JSII_RTTI_SYMBOL_1;
SonarCodeScanner[_a] = { fqn: "aws-prototyping-sdk.pdk_pipeline.SonarCodeScanner", version: "0.0.54" };
//# sourceMappingURL=data:application/json;base64,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