"use strict";
/**
 *  Copyright 2019 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3ToStepFunction = void 0;
const defaults = require("@aws-solutions-constructs/core");
const aws_events_rule_step_function_1 = require("@aws-solutions-constructs/aws-events-rule-step-function");
const core_1 = require("@aws-cdk/core");
const cloudtrail = require("@aws-cdk/aws-cloudtrail");
class S3ToStepFunction extends core_1.Construct {
    /**
     * @summary Constructs a new instance of the S3ToStepFunction class.
     * @param {cdk.App} scope - represents the scope for all the resources.
     * @param {string} id - this is a a scope-unique id.
     * @param {S3ToStepFunctionProps} props - user provided props for the construct
     * @since 0.9.0
     * @access public
     */
    constructor(scope, id, props) {
        super(scope, id);
        [this.s3Bucket, this.s3LoggingBucket] = defaults.buildS3Bucket(this, {
            existingBucketObj: props.existingBucketObj,
            bucketProps: props.bucketProps
        });
        if (!props.hasOwnProperty('deployCloudTrail') || props.deployCloudTrail === true) {
            [this.cloudtrailBucket, this.cloudtrailLoggingBucket] = defaults.buildS3Bucket(this, {}, 'CloudTrail');
            this.cloudtrail = new cloudtrail.Trail(this, 'S3EventsTrail', {
                bucket: this.cloudtrailBucket
            });
            this.cloudtrail.addS3EventSelector([{
                    bucket: this.s3Bucket
                }], {
                readWriteType: cloudtrail.ReadWriteType.ALL,
                includeManagementEvents: false
            });
        }
        let _eventRuleProps = {};
        if (props.eventRuleProps) {
            _eventRuleProps = props.eventRuleProps;
        }
        else {
            // By default the CW Events Rule will filter any 's3:PutObject' events for the S3 Bucket
            _eventRuleProps = {
                eventPattern: {
                    source: ['aws.s3'],
                    detailType: ['AWS API Call via CloudTrail'],
                    detail: {
                        eventSource: [
                            "s3.amazonaws.com"
                        ],
                        eventName: [
                            "PutObject"
                        ],
                        requestParameters: {
                            bucketName: [
                                this.s3Bucket.bucketName
                            ]
                        }
                    }
                }
            };
        }
        const eventsRuleToStepFunction = new aws_events_rule_step_function_1.EventsRuleToStepFunction(this, 'test-events-rule-step-function-stack', {
            stateMachineProps: props.stateMachineProps,
            eventRuleProps: _eventRuleProps
        });
        this.stateMachine = eventsRuleToStepFunction.stateMachine;
        this.stateMachineLogGroup = eventsRuleToStepFunction.stateMachineLogGroup;
        this.cloudwatchAlarms = eventsRuleToStepFunction.cloudwatchAlarms;
    }
}
exports.S3ToStepFunction = S3ToStepFunction;
//# sourceMappingURL=data:application/json;base64,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