# -*- coding: utf-8 -*-
from setuptools import setup

package_dir = \
{'': 'src'}

packages = \
['kafka_connect']

package_data = \
{'': ['*']}

install_requires = \
['click>=8.1.3,<9.0.0', 'requests>=2.25,<3.0']

entry_points = \
{'console_scripts': ['kafka-connect = kafka_connect.cli:cli',
                     'kc = kafka_connect.cli:cli']}

setup_kwargs = {
    'name': 'kafka-connect-py',
    'version': '0.10.1',
    'description': 'A client for the Confluent Platform Kafka Connect REST API.',
    'long_description': '[![PyPI](https://img.shields.io/pypi/v/kafka_connect_py.svg)](https://pypi.org/project/kafka-connect-py/)\n\n# Kafka Connect Python\n\nThe Kafka Connect REST API allows you to manage connectors that move data between Apache Kafka and other systems.\n\nThe Kafka Connect command line tool, also known as `kc` or `kafka-connect`, allows users to manage their Kafka Connect cluster and connectors. With this tool, users can retrieve information about the cluster and connectors, create new connectors, update existing connectors, delete connectors, and perform other actions.\n\nThis project aims to supported all features of the [Kafka Connect REST API](https://docs.confluent.io/platform/current/connect/references/restapi.html#kconnect-rest-interface).\n\n## Install\n\n```bash\npip install kafka-connect-py\n```\n\n## Command Line Usage\n\n### Getting Basic Connect Cluster Information\n\nTo get basic Connect cluster information including the worker version, the commit it’s on, and its Kafka cluster ID, use the following command:\n\n```bash\nkc info\n```\n\n### Listing Installed Plugins\n\nTo list the plugins installed on the worker, use the following command:\n\n```bash\nkc plugins\n```\n\nTo format the result of the installed plugin list for easier readability, pipe the output to the jq command:\n\n```bash\nkc plugins | jq \'.\'\n```\n\n### Create a Connector Instance\n\nTo create a connector instance with JSON data containing the connector’s configuration:\n\n```bash\nkc update source-debezium-orders-00 -d \'{\n    "connector.class": "io.debezium.connector.mysql.MySqlConnector",\n    "value.converter": "io.confluent.connect.json.JsonSchemaConverter",\n    "value.converter.schemas.enable": "true",\n    "value.converter.schema.registry.url": "\'$SCHEMA_REGISTRY_URL\'",\n    "value.converter.basic.auth.credentials.source": "\'$BASIC_AUTH_CREDENTIALS_SOURCE\'",\n    "value.converter.basic.auth.user.info": "\'$SCHEMA_REGISTRY_BASIC_AUTH_USER_INFO\'",\n    "database.hostname": "mysql",\n    "database.port": "3306",\n    "database.user": "debezium",\n    "database.password": "dbz",\n    "database.server.id": "42",\n    "database.server.name": "asgard",\n    "table.whitelist": "demo.orders",\n    "database.history.kafka.bootstrap.servers": "\'$BOOTSTRAP_SERVERS\'",\n    "database.history.consumer.security.protocol": "SASL_SSL",\n    "database.history.consumer.sasl.mechanism": "PLAIN",\n    "database.history.consumer.sasl.jaas.config": "org.apache.kafka.common.security.plain.PlainLoginModule required username=\\"\'$CLOUD_KEY\'\\" password=\\"\'$CLOUD_SECRET\'\\";",\n    "database.history.producer.security.protocol": "SASL_SSL",\n    "database.history.producer.sasl.mechanism": "PLAIN",\n    "database.history.producer.sasl.jaas.config": "org.apache.kafka.common.security.plain.PlainLoginModule required username=\\"\'$CLOUD_KEY\'\\" password=\\"\'$CLOUD_SECRET\'\\";",\n    "database.history.kafka.topic": "dbhistory.demo",\n    "topic.creation.default.replication.factor": "3",\n    "topic.creation.default.partitions": "3",\n    "decimal.handling.mode": "double",\n    "include.schema.changes": "true",\n    "transforms": "unwrap,addTopicPrefix",\n    "transforms.unwrap.type": "io.debezium.transforms.ExtractNewRecordState",\n    "transforms.addTopicPrefix.type":"org.apache.kafka.connect.transforms.RegexRouter",\n    "transforms.addTopicPrefix.regex":"(.*)",\n    "transforms.addTopicPrefix.replacement":"mysql-debezium-$1"\n}\'\n```\n\nOr create/update a connector instance with a JSON file:\n\n```bash\nkc update <connector> --config-file <config_file>\n```\n\n### List Connector Instances\n\nUse the following command to list of all extant connectors:\n\n```bash\nkc list [--expand=info|status] [--pattern=regex] [--state=running|paused|unassigned|failed]\n```\n\n\n### Inspect Config and Status for a Connector\n\nInspect the config for a given connector as follows:\n\n```bash\nkc config sink-elastic-orders-00\n```\n\nYou can also look at a connector’s status. While the config command shows a connector’s static configuration, the status shows the connector as a runtime entity:\n\n```bash\nkc status sink-elastic-orders-00\n```\n\nYou can also use `list` with the `--expand=status` option to show the status of many connectors at once. We can filter down the response using a regex pattern and/or connector state. \n\nUse the following to show all connector names prefixed with the word `sink-` and that are in a `FAILED` connector state.\n\n```bash\nkc list --expand=status -p sink-.* -s failed\n```\n\n### Delete a Connector\n\nIf something is wrong in your setup and you don’t think a config change would help, or if you simply don’t need a connector to run anymore, you can delete it by name:\n\n```bash\nkc delete sink-elastic-orders-00\n```\n\nThe `delete` sub-command also supports multiple deletions using the `--all` option. On its own it will apply the sub-command to all connectors.\n\nThe following will delete all connector names prefixed with the word `sink-` and that are in a `PAUSED` connector state.\n\n```bash\nkc delete --all --pattern sink-.* -s paused\n```\n\nThe --all option is supported by several sub-commands, including `delete`, `restart`, `resume`, and `pause`. However, for better testing and control over the outcome of your actions, we recommend using the list filtering option before executing any of these sub-commands. This way, you can ensure that your filters are working as intended and avoid unintended consequences. To use list filtering, simply run the `list` sub-command and apply your filters.\n\n### Update a Connector\n\nAs mentioned above, if there’s a connector to update, you can use the `update` sub-command to amend the configuration (see [Create a Connector Instance](https://github.com/aidanmelen/kafka-connect-py/blob/main/README.md#create-a-connector-instance) above). Because update is used to both create and update connectors, it’s the standard command that you should use most of the time (which also means that you don’t have to completely rewrite your configs).\n\n### Inspect Task Details\n\nThe following command returns the connector status:\n\n```bash\nkc status source-debezium-orders-00 | jq \'.\'\n```\n\nIf your connector fails, the details of the failure belong to the task. So to inspect the problem, you’ll need to find the stack trace for the task. The task is the entity that is actually running the connector and converter code, so the state for the stack trace lives in it.\n\n```bash\nkc task-status source-debezium-orders-00 <task-id> | jq \'.\'\n```\n\n### Restart the Connector and Tasks\n\nIf after inspecting a task, you have determined that it has failed and you have fixed the reason for the failure (perhaps restarted a database), you can restart the connector with the following:\n\n```\nkc restart source-debezium-orders-00\n```\n\nKeep in mind though that restarting the connector doesn’t restart all of its tasks. You will also need to restart the failed task and then get its status again as follows:\n\n```bash\nkc task-status source-debezium-orders-00 <task-id> \n```\n\nWhat\'s more, you can restart the connector and all its failed tasks with the following:\n\n```bash\nkc restart source-debezium-orders-00 --include-tasks --failed-only\n```\n\nand check the status again:\n\n```bash\nkc status source-debezium-orders-00 | jq \'.\'\n```\n\n### Pause and Resume a Connector\n\nUnlike restarting, pausing a connector does pause its tasks. This happens asynchronously, though, so when you pause a connector, you can’t rely on it pausing all of its tasks at exactly the same time. The tasks are running in a thread pool, so there’s no fancy mechanism to make this happen simultaneously.\n\nA connector and its tasks can be paused as follows:\n\n```bash\nkc pause source-debezium-orders-00\n```\n\nJust as easily, a connector and its tasks can be resumed:\n\n```bash\nkc resume source-debezium-orders-00\n```\n\n### Display All of a Connector’s Tasks\n\nA convenient way to display all of a connector’s tasks at once is as follows:\n\n```bash\nkc list-tasks source-debezium-orders-00 | jq \'.\'\n```\n\nThis information is similar to what you can get from other APIs, but it is broken down by task, and configs for each are shown.\nGet a List of Topics Used by a Connector\n\nAs of Apache Kafka 2.5, it is possible to get a list of topics used by a connector:\n\n```bash\nkc list-topics | jq \'.\'\n```\n\nThis shows the topics that a connector is consuming from or producing to. This may not be particularly useful for connectors that are consuming from or producing to a single topic. However, some developers, for example, use regular expressions for topic names in Connect, so this is a major benefit in situations where topic names are derived computationally.\n\nThis could also be useful with a source connector that is using SMTs to dynamically change the topic names to which it is producing.\n\n## Python\n\n```python\n# Import the class\nfrom kafka_connect import KafkaConnect\n\nimport json\n\n# Instantiate the client\nclient = KafkaConnect(url="http://localhost:8083")\n\n# Get the version and other details of the Kafka Connect cluster\ncluster = client.get_cluster_info()\nprint(cluster)\n\n# Get a list of active connectors\nconnectors = client.list_connectors(expand="status")\nprint(json.dumps(connectors, indent=2))\n\n# Create a new connector\nconfig = {\n    "name": "my-connector",\n    "config": {\n        "connector.class": "io.confluent.connect.jdbc.JdbcSourceConnector",\n        "tasks.max": "1",\n        "connection.url": "jdbc:postgresql://localhost:5432/mydatabase",\n        "connection.user": "myuser",\n        "connection.password": "mypassword",\n        "table.whitelist": "mytable",\n        "mode": "timestamp+incrementing",\n        "timestamp.column.name": "modified_at",\n        "validate.non.null": "false",\n        "incrementing.column.name": "id",\n        "topic.prefix": "my-connector-",\n    },\n}\nresponse = client.create_connector(config)\nprint(response)\n\n# Update an existing connector\nnew_config = {\n    "config": {\n        "connector.class": "io.confluent.connect.jdbc.JdbcSourceConnector",\n        "tasks.max": "1",\n        "connection.url": "jdbc:postgresql://localhost:5432/mydatabase",\n        "connection.user": "myuser",\n        "connection.password": "mypassword",\n        "table.whitelist": "mytable",\n        "mode": "timestamp+incrementing",\n        "timestamp.column.name": "modified_at",\n        "validate.non.null": "false",\n        "incrementing.column.name": "id",\n        "topic.prefix": "my-connector-",\n    },\n}\nresponse = client.update_connector("my-connector", new_config)\nprint(response)\n\n# Get status for a connector\nresponse = client.get_connector_status("my-connector")\nprint(json.dumps(response, indent=2))\n\n# Restart a connector\nresponse = client.restart_connector("my-connector")\nprint(response)\n\n# Delete a connector\nresponse = client.delete_connector("my-connector")\nprint(response)\n```\n\n## Tests\n\n```bash\npython3 -m unittest tests/test_kafka_connect.py -v\n..............................\n----------------------------------------------------------------------\nRan 30 tests in 0.034s\n\nOK\n```\n\n## Credits\n\nThe entire [Command Line Usage](https://github.com/aidanmelen/kafka-connect-py/blob/main/README.md#command-line-usage) section was copied directly from the Confluence\'s [Kafka Connect’s REST API](https://developer.confluent.io/learn-kafka/kafka-connect/rest-api/) course.',
    'author': 'Aidan Melen',
    'author_email': 'aidan-melen@protonmail.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/aidanmelen/kafka-connect-py',
    'package_dir': package_dir,
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'entry_points': entry_points,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
