"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const compress_content_lambda_shared_1 = require("../shared/compress-content.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
async function handler(event, context) {
    var _a;
    console.log('Event:', JSON.stringify(event, null, 2));
    const indexedPackages = new Map();
    const packageNames = new Set();
    const packageMajorVersions = new Set();
    const perLanguage = new Map();
    /**
     * Records the status of a particular package, package major version, package
     * version, and package version submodule in the per-language state storage.
     * Whenever a new entry is added, a `MISSING` entry is automatically inserted
     * for the other languages (unless another entry already exists).
     *
     * If a submodule is provided, only that submodule's availability is updated.
     */
    function recordPerLanguage(language, status, pkgName, pkgMajor, pkgVersion, submodule) {
        for (const lang of language_1.DocumentationLanguage.ALL) {
            doRecordPerLanguage(perLanguage, lang, 
            // If the language is NOT the registered one, then we insert "MISSING".
            lang === language ? status : "Missing" /* MISSING */, pkgName, pkgMajor, pkgVersion, submodule);
        }
    }
    const bucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    for await (const key of relevantObjectKeys(bucket)) {
        const [, name, version] = constants.STORAGE_KEY_FORMAT_REGEX.exec(key);
        packageNames.add(name);
        const majorVersion = `${name}@${new semver_1.SemVer(version).major}`;
        packageMajorVersions.add(majorVersion);
        const fullName = `${name}@${version}`;
        // Ensure the package is fully registered for per-language status, even if no doc exists yet.
        for (const language of language_1.DocumentationLanguage.ALL) {
            recordPerLanguage(language, "Missing" /* MISSING */, name, majorVersion, fullName);
        }
        if (!indexedPackages.has(fullName)) {
            indexedPackages.set(fullName, {});
        }
        const status = indexedPackages.get(fullName);
        if (key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            status.metadataPresent = true;
        }
        else if (key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            status.tarballPresent = true;
        }
        else if (key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            status.assemblyPresent = true;
        }
        else if (key.endsWith(constants.UNINSTALLABLE_PACKAGE_SUFFIX)) {
            status.uninstallable = true;
        }
        else {
            let identified = false;
            for (const language of language_1.DocumentationLanguage.ALL) {
                const matchJson = submoduleKeyRegexp(language, 'json').exec(key);
                const matchMd = submoduleKeyRegexp(language, 'md').exec(key);
                if (matchJson != null) {
                    const [, submodule, isUnsupported] = matchJson;
                    if (status.submodules == null) {
                        status.submodules = new Set();
                    }
                    status.submodules.add(`${fullName}.${submodule}`);
                    recordPerLanguage(language, isUnsupported ? "Unsupported" /* UNSUPPORTED */ : "Supported" /* SUPPORTED */, name, majorVersion, fullName, submodule);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language, undefined, 'json'))) {
                    recordPerLanguage(language, "Supported" /* SUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
                else if (key.endsWith(constants.notSupportedKeySuffix(language, undefined, 'json'))) {
                    recordPerLanguage(language, "Unsupported" /* UNSUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
                else if (key.endsWith(constants.corruptAssemblyKeySuffix(language, undefined, 'json'))) {
                    recordPerLanguage(language, "CorruptAssembly" /* CORRUPT_ASSEMBLY */, name, majorVersion, fullName);
                    identified = true;
                    // Currently we generate both JSON files and markdown files, so for now
                    // we record JSON files as the source of truth, but still identify
                    // markdown files so they are not counted as unknown.
                }
                else if (matchMd != null) {
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language, undefined, 'md'))) {
                    identified = true;
                }
                else if (key.endsWith(constants.notSupportedKeySuffix(language, undefined, 'md'))) {
                    identified = true;
                }
                else if (key.endsWith(constants.corruptAssemblyKeySuffix(language, undefined, 'md'))) {
                    identified = true;
                }
            }
            if (!identified) {
                status.unknownObjects = (_a = status.unknownObjects) !== null && _a !== void 0 ? _a : [];
                status.unknownObjects.push(key);
            }
        }
    }
    const reports = [];
    function createReport(reportKey, packageVersions) {
        const report = JSON.stringify(packageVersions, null, 2);
        const { buffer, contentEncoding } = compress_content_lambda_shared_1.compressContent(Buffer.from(report));
        console.log(`Uploading list to s3://${bucket}/${reportKey}`);
        reports.push(aws.s3().putObject({
            Body: buffer,
            Bucket: bucket,
            ContentEncoding: contentEncoding,
            ContentType: 'application/json',
            Expires: new Date(Date.now() + 300000),
            Key: reportKey,
            Metadata: {
                'Lambda-Run-Id': context.awsRequestId,
                'Lambda-Log-Group-Name': context.logGroupName,
                'Lambda-Log-Stream-Name': context.logStreamName,
            },
        }).promise());
    }
    await aws_embedded_metrics_1.metricScope((metrics) => () => {
        var _a, _b, _c;
        // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
        metrics.setDimensions();
        const missingMetadata = new Array();
        const missingAssembly = new Array();
        const missingTarball = new Array();
        const uninstallable = new Array();
        const unknownObjects = new Array();
        const submodules = new Array();
        for (const [name, status] of indexedPackages.entries()) {
            if (!status.metadataPresent) {
                missingMetadata.push(name);
            }
            if (!status.assemblyPresent) {
                missingAssembly.push(name);
            }
            if (!status.tarballPresent) {
                missingTarball.push(name);
            }
            if (status.uninstallable) {
                uninstallable.push(name);
            }
            if ((_b = (_a = status.unknownObjects) === null || _a === void 0 ? void 0 : _a.length) !== null && _b !== void 0 ? _b : 0 > 0) {
                unknownObjects.push(...status.unknownObjects);
            }
            for (const submodule of (_c = status.submodules) !== null && _c !== void 0 ? _c : []) {
                submodules.push(submodule);
            }
        }
        metrics.setProperty('detail', { missingMetadata, missingAssembly, missingTarball, unknownObjects });
        metrics.putMetric("UninstallablePackageCount" /* UNINSTALLABLE_PACKAGE_COUNT */, uninstallable.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */, missingMetadata.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */, missingAssembly.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */, missingTarball.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageCount" /* PACKAGE_COUNT */, packageNames.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */, packageMajorVersions.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, indexedPackages.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("SubmoduleCount" /* SUBMODULE_COUNT */, submodules.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */, unknownObjects.length, aws_embedded_metrics_1.Unit.Count);
        createReport(constants.UNINSTALLABLE_PACKAGES_REPORT, uninstallable);
    })();
    for (const entry of Array.from(perLanguage.entries())) {
        await aws_embedded_metrics_1.metricScope((metrics) => async (language, data) => {
            console.log('');
            console.log('##################################################');
            console.log(`### Start of data for ${language}`);
            metrics.setDimensions({ [constants_1.LANGUAGE_DIMENSION]: language.toString() });
            for (const forStatus of [
                "Supported" /* SUPPORTED */,
                "Unsupported" /* UNSUPPORTED */,
                "Missing" /* MISSING */,
                "CorruptAssembly" /* CORRUPT_ASSEMBLY */,
            ]) {
                for (const [key, statuses] of Object.entries(data)) {
                    let filtered = Array.from(statuses.entries()).filter(([, status]) => forStatus === status);
                    let metricName = METRIC_NAME_BY_STATUS_AND_GRAIN[forStatus][key];
                    if ((forStatus === "Missing" /* MISSING */ && metricName === "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */)
                        || (forStatus === "CorruptAssembly" /* CORRUPT_ASSEMBLY */ && metricName === "CorruptAssemblyPackageVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_VERSIONS */)) {
                        // generate reports for missing/corrupt only for package versions granularity
                        const reportKey = forStatus === "Missing" /* MISSING */ ?
                            constants.missingDocumentationReport(language) :
                            constants.corruptAssemblyReport(language);
                        createReport(reportKey, filtered.map(([name]) => name).sort());
                    }
                    console.log(`${forStatus} ${key} for ${language}: ${filtered.length} entries`);
                    metrics.putMetric(metricName, filtered.length, aws_embedded_metrics_1.Unit.Count);
                }
            }
            console.log(`### End of data for ${language}`);
            console.log('##################################################');
            console.log('');
        })(...entry);
    }
    for (const report of reports) {
        await report;
    }
}
exports.handler = handler;
async function* relevantObjectKeys(bucket) {
    var _a;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        for (const { Key } of (_a = response.Contents) !== null && _a !== void 0 ? _a : []) {
            if (Key == null) {
                continue;
            }
            yield Key;
        }
        request.ContinuationToken = response.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
/**
 * This function obtains a regular expression with a capture group that allows
 * determining the submodule name from a submodule documentation key, and
 * another to determine whether the object is an "unsupported beacon" or not.
 */
function submoduleKeyRegexp(language, fileExt) {
    // We use a placeholder to be able to insert the capture group once we have
    // fully quoted the key prefix for Regex safety.
    const placeholder = '<SUBMODULENAME>';
    // We obtain the standard key suffix.
    const keyPrefix = constants.docsKeySuffix(language, placeholder, fileExt);
    // Finally, assemble the regular expression with the capture group.
    return new RegExp(`.*${reQuote(keyPrefix).replace(placeholder, '(.+)')}(${reQuote(constants.NOT_SUPPORTED_SUFFIX)})?$`);
    /**
     * Escapes all "speacial meaning" characters in a string, so it can be used as
     * part of a regular expression.
     */
    function reQuote(str) {
        return str.replace(/([+*.()?$[\]])/g, '\\$1');
    }
}
const METRIC_NAME_BY_STATUS_AND_GRAIN = {
    ["Missing" /* MISSING */]: {
        ["packages" /* PACKAGES */]: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
    },
    ["Unsupported" /* UNSUPPORTED */]: {
        ["packages" /* PACKAGES */]: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
    },
    ["Supported" /* SUPPORTED */]: {
        ["packages" /* PACKAGES */]: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
    },
    ["CorruptAssembly" /* CORRUPT_ASSEMBLY */]: {
        ["packages" /* PACKAGES */]: "CorruptAssemblyPackageCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "CorruptAssemblyMajorVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "CorruptAssemblyPackageVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "CorruptAssemblySubmoduleCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_SUBMODULES */,
    },
};
/**
 * Registers the information for the provided language. A "MISSING" status
 * will be ignored if another status was already registered for the same
 * entity. An "UNSUPPORTED" status will be ignored if a "SUPPORTED" status
 * was already registered for the same entity.
 *
 * If a submodule is provided, only that submodule's availability is updated.
 */
function doRecordPerLanguage(perLanguage, language, status, pkgName, pkgMajor, pkgVersion, submodule) {
    if (!perLanguage.has(language)) {
        perLanguage.set(language, {
            ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: new Map(),
            ["packages" /* PACKAGES */]: new Map(),
            ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: new Map(),
            ["package versions" /* PACKAGE_VERSIONS */]: new Map(),
        });
    }
    const data = perLanguage.get(language);
    // If there is a submodule, only update the submodule domain.
    const outputDomains = submodule
        ? [
            [data["package version submodules" /* PACKAGE_VERSION_SUBMODULES */], `${pkgVersion}.${submodule}`],
        ]
        : [
            [data["package major versions" /* PACKAGE_MAJOR_VERSIONS */], pkgMajor],
            [data["package versions" /* PACKAGE_VERSIONS */], pkgVersion],
            [data["packages" /* PACKAGES */], pkgName],
        ];
    for (const [map, name] of outputDomains) {
        switch (status) {
            case "Missing" /* MISSING */:
                // If we already have a status, don't override it with "MISSING".
                if (!map.has(name)) {
                    map.set(name, status);
                }
                break;
            case "Supported" /* SUPPORTED */:
                // If thr package is "supported", this always "wins"
                map.set(name, status);
                break;
            case "Unsupported" /* UNSUPPORTED */:
            case "CorruptAssembly" /* CORRUPT_ASSEMBLY */:
                // If we already have a status, only override with if it was "MISSING".
                if (!map.has(name) || map.get(name) === "Missing" /* MISSING */) {
                    map.set(name, status);
                }
                break;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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