"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Inventory = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const constants_1 = require("../shared/constants");
const canary_1 = require("./canary");
const constants_2 = require("./constants");
/**
 * Periodically computes an inventory of all indexed packages into the storage
 * bucket, and produces metrics with an overview of the index' state.
 */
class Inventory extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.rate = (_a = props.scheduleRate) !== null && _a !== void 0 ? _a : core_1.Duration.minutes(15);
        this.canary = new canary_1.Canary(this, 'Resource', {
            description: '[ConstructHub/Inventory] A canary that periodically inspects the list of indexed packages',
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: props.bucket.bucketName,
            },
            logRetention: props.logRetention,
            memorySize: 10240,
            timeout: this.rate,
        });
        const grantRead = props.bucket.grantRead(this.canary);
        const grantWriteMissing = props.bucket.grantWrite(this.canary, constants_1.MISSING_DOCUMENTATION_REPORT_PATTERN);
        const grantWriteCorruptAssembly = props.bucket.grantWrite(this.canary, constants_1.CORRUPT_ASSEMBLY_REPORT_PATTERN);
        const grantWriteUnInstallable = props.bucket.grantWrite(this.canary, constants_1.UNINSTALLABLE_PACKAGES_REPORT);
        const rule = new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule: aws_events_1.Schedule.rate(this.rate),
            targets: [new aws_events_targets_1.LambdaFunction(this.canary)],
        });
        rule.node.addDependency(grantRead, grantWriteMissing);
        rule.node.addDependency(grantRead, grantWriteCorruptAssembly);
        rule.node.addDependency(grantRead, grantWriteUnInstallable);
        props.monitoring.addLowSeverityAlarm('Inventory Canary is not Running', this.canary.metricInvocations({ period: this.rate }).createAlarm(this, 'Not Running', {
            alarmName: `${this.node.path}/NotRunning`,
            alarmDescription: [
                'The inventory canary is not running!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to function: ${deep_link_1.lambdaFunctionUrl(this.canary)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        props.monitoring.addLowSeverityAlarm('Inventory Canary is failing', this.canary.metricErrors({ period: this.rate }).createAlarm(this, 'Failures', {
            alarmName: `${this.node.path}/Failures`,
            alarmDescription: [
                'The inventory canary is failing!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to function: ${deep_link_1.lambdaFunctionUrl(this.canary)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
        }));
    }
    get function() {
        return this.canary;
    }
    metricMissingPackageMetadataCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricMissingAssemblyCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricPackageCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageCount" /* PACKAGE_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricPackageMajorCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricMissingPackageTarballCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricUninstallablePackageCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "UninstallablePackageCount" /* UNINSTALLABLE_PACKAGE_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricSubmoduleCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "SubmoduleCount" /* SUBMODULE_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    metricUnknownObjectCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages for which all versions are missing a documnetation artifact
     * (whether supported or not) for the provided `DocumentationLanguage`.
     */
    metricMissingPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions for which all versions are missing a
     * documnetation artifact (whether supported or not) for the provided
     * `DocumentationLanguage`.
     */
    metricMissingMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that are missing a documnetation artifact
     * (whether supported or not) for the provided `DocumentationLanguage`.
     */
    metricMissingPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that are missing a documnetation
     * artifact (whether supported or not) for the provided
     * `DocumentationLanguage`.
     */
    metricMissingSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages that have at least one version for which there is
     * available documentation in the provided `DocumentationLanguage`.
     */
    metricSupportedPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions that have at least one version for
     * which there is available documentation in the provided
     * `DocumentationLanguage`.
     */
    metricSupportedMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that have available documentation in the
     * provided `DocumentationLanguage`.
     */
    metricSupportedPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that have available documentation
     * in the provided `DocumentationLanguage`.
     */
    metricSupportedSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that do not support the provided
     * `DocumentationLanguage`, and hence cannot have documentation for it.
     */
    metricUnsupportedSubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of packages that have a language specific corrupt assembly.
     */
    metricCorruptAssemblyPackageCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "CorruptAssemblyPackageCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_PACKAGES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package major versions that have a language specific corrupt assembly.
     */
    metricCorruptAssemblyMajorVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "CorruptAssemblyMajorVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_MAJORS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package versions that have a language specific corrupt assembly.
     */
    metricCorruptAssemblyPackageVersionCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "CorruptAssemblyPackageVersionCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_VERSIONS */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
    /**
     * The count of package version submodules that have a language specific corrupt assembly.
     */
    metricCorruptAssemblySubmoduleCount(language, opts) {
        return new aws_cloudwatch_1.Metric({
            period: this.rate,
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensions: {
                [constants_2.LANGUAGE_DIMENSION]: language.toString(),
            },
            metricName: "CorruptAssemblySubmoduleCount" /* PER_LANGUAGE_CORRUPT_ASSEMBLY_SUBMODULES */,
            namespace: constants_2.METRICS_NAMESPACE,
        });
    }
}
exports.Inventory = Inventory;
//# sourceMappingURL=data:application/json;base64,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