"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Orchestration = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const tasks = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../../deep-link");
const runbook_url_1 = require("../../runbook-url");
const _lambda_architecture_1 = require("../_lambda-architecture");
const catalog_builder_1 = require("../catalog-builder");
const constants_1 = require("../shared/constants");
const transliterator_1 = require("../transliterator");
const needs_catalog_update_1 = require("./needs-catalog-update");
const redrive_state_machine_1 = require("./redrive-state-machine");
/**
 * This retry policy is used for all items in the state machine and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * After 30 attempts, given the parameters, the last attempt will wait just
 * under 16 minutes, which should be enough for currently running Lambda
 * functions to complete (or time out after 15 minutes). The total time spent
 * waiting between retries by this time is just over 3 hours. This is a lot of
 * time, but in extreme burst situations (i.e: reprocessing everything), this
 * is actually a good thing.
 */
const THROTTLE_RETRY_POLICY = { backoffRate: 1.1, interval: core_1.Duration.minutes(1), maxAttempts: 30 };
/**
 * This retry policy is used for transliteration tasks and allows ample
 * retry attempts in order to avoid having to implement a custom backpressure
 * handling mehanism.
 *
 * This is meant as a stop-gap until we can implement a more resilient system,
 * which likely will involve more SQS queues, but will probably need to be
 * throughoutly vetted before it is rolled out everywhere.
 *
 * The interval is determined by the execution duration we currently expect from jsii-docgen.
 * See https://github.com/cdklabs/jsii-docgen/blob/main/test/docgen/view/documentation.test.ts#L13.
 *
 * We don't apply backoff because this inevitably causes longer wait times than desired, and unfortunately
 * there is no way to configure max interval. In addition, since a task duration is fairly stable,
 * we can assume capacity will free up after roughly 2 minutes.
 *
 * Combined with a two minute interval, and a backoff factor of 1, 200 attempts gives us just under 7 hours to complete
 * a full reprocessing workflow, which should be sufficient.
 */
const DOCGEN_THROTTLE_RETRY_POLICY = { backoffRate: 1, interval: core_1.Duration.minutes(2), maxAttempts: 200 };
/**
 * Orchestrates the backend processing tasks using a StepFunctions State Machine.
 */
class Orchestration extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.deadLetterQueue = new aws_sqs_1.Queue(this, 'DLQ', {
            encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
            retentionPeriod: core_1.Duration.days(14),
            visibilityTimeout: core_1.Duration.minutes(15),
        });
        props.monitoring.addLowSeverityAlarm('Backend Orchestration Dead-Letter Queue is not empty', new aws_cloudwatch_1.MathExpression({
            expression: 'm1 + m2',
            label: 'Dead-Letter Queue not empty',
            usingMetrics: {
                m1: this.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                m2: this.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(this, 'DLQAlarm', {
            alarmName: `${this.deadLetterQueue.node.path}/NotEmpty`,
            alarmDescription: [
                'Backend orchestration dead-letter queue is not empty.',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to queue: ${deep_link_1.sqsQueueUrl(this.deadLetterQueue)}`,
                'Warning: State Machines executions that sent messages to the DLQ will not show as "failed".',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        const sendToDeadLetterQueue = new tasks.SqsSendMessage(this, 'Send to Dead Letter Queue', {
            messageBody: aws_stepfunctions_1.TaskInput.fromJsonPathAt('$'),
            queue: this.deadLetterQueue,
            resultPath: aws_stepfunctions_1.JsonPath.DISCARD,
        }).next(new aws_stepfunctions_1.Succeed(this, 'Sent to DLQ'));
        const ignore = new aws_stepfunctions_1.Pass(this, 'Ignore');
        this.catalogBuilder = new catalog_builder_1.CatalogBuilder(this, 'CatalogBuilder', props);
        const addToCatalog = new tasks.LambdaInvoke(this, 'Add to catalog.json', {
            lambdaFunction: this.catalogBuilder.function,
            resultPath: '$.catalogBuilderOutput',
            resultSelector: {
                'ETag.$': '$.Payload.ETag',
                'VersionId.$': '$.Payload.VersionId',
            },
        })
            // This has a concurrency of 1, so we want to aggressively retry being throttled here.
            .addRetry({ errors: ['Lambda.TooManyRequestsException'], ...THROTTLE_RETRY_POLICY })
            .addCatch(sendToDeadLetterQueue, { errors: ['Lambda.TooManyRequestsException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' });
        const needsCatalogUpdateFunction = new needs_catalog_update_1.NeedsCatalogUpdate(this, 'NeedsCatalogUpdate', {
            architecture: _lambda_architecture_1.gravitonLambdaIfAvailable(this),
            description: '[ConstructHub/Orchestration/NeedsCatalogUpdate] Determines whether a package version requires a catalog update',
            environment: { CATALOG_BUCKET_NAME: props.bucket.bucketName, CATALOG_OBJECT_KEY: constants_1.CATALOG_KEY },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(1),
        });
        props.bucket.grantRead(needsCatalogUpdateFunction);
        // Check whether the catalog needs updating. If so, trigger addToCatalog.
        const addToCatalogIfNeeded = new tasks.LambdaInvoke(this, 'Check whether catalog needs udpating', {
            lambdaFunction: needsCatalogUpdateFunction,
            payloadResponseOnly: true,
            resultPath: '$.catalogNeedsUpdating',
        })
            .addRetry({
            errors: [
                'Lambda.TooManyRequestsException',
                'Lambda.Unknown',
            ],
            ...THROTTLE_RETRY_POLICY,
        })
            .addCatch(sendToDeadLetterQueue, { errors: ['Lambda.TooManyRequestsException', 'Lambda.Unknown'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' })
            .next(new aws_stepfunctions_1.Choice(this, 'Is catalog update needed?')
            .when(aws_stepfunctions_1.Condition.booleanEquals('$.catalogNeedsUpdating', true), addToCatalog)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.ecsCluster = new aws_ecs_1.Cluster(this, 'Cluster', {
            containerInsights: true,
            enableFargateCapacityProviders: true,
            vpc: props.vpc,
        });
        this.transliterator = new transliterator_1.Transliterator(this, 'Transliterator', props);
        const definition = new aws_stepfunctions_1.Pass(this, 'Track Execution Infos', {
            inputPath: '$$.Execution',
            parameters: {
                'Id.$': '$.Id',
                'Name.$': '$.Name',
                'RoleArn.$': '$.RoleArn',
                'StartTime.$': '$.StartTime',
            },
            resultPath: '$.$TaskExecution',
        })
            .next(new aws_stepfunctions_1.Pass(this, 'Prepare doc-gen ECS Command', {
            parameters: { 'command.$': 'States.Array(States.JsonToString($))' },
            resultPath: '$.docGen',
        }))
            .next(this.transliterator.createEcsRunTask(this, 'Generate docs', {
            cluster: this.ecsCluster,
            inputPath: '$.docGen.command',
            resultPath: '$.docGenOutput',
            // aws-cdk-lib succeeds in roughly 1 hour, so this should give us
            // enough of a buffer and prorably account for all other libraries out there.
            timeout: core_1.Duration.hours(2),
            vpcSubnets: props.vpcSubnets,
            securityGroups: props.vpcSecurityGroups,
            // The task code sends a heartbeat back every minute, but in rare
            // cases the first heartbeat may take a while to come back due to
            // the time it takes to provision the task in the cluster, so we
            // give a more generous buffer here.
            heartbeat: core_1.Duration.minutes(10),
        })
            // Do not retry NoSpaceLeftOnDevice errors, these are typically not transient.
            .addRetry({ errors: ['jsii-docgen.NoSpaceLeftOnDevice'], maxAttempts: 0 })
            .addRetry({
            errors: [
                'ECS.AmazonECSException',
                'ECS.InvalidParameterException',
                'jsii-docgen.NpmError.E429',
                'jsii-codgen.NpmError.EPROTO',
            ],
            ...DOCGEN_THROTTLE_RETRY_POLICY,
        })
            .addRetry({
            errors: ['jsii-docgen.NpmError.ETARGET'],
            // We'll wait longer between retries. This is to account for CodeArtifact's lag behind npm
            backoffRate: 2,
            interval: core_1.Duration.minutes(5),
            maxAttempts: 3,
        })
            .addRetry({
            errors: ['States.Timeout'],
            // To compensate we'll give more retries and pause between them in
            // case it's just a transient issue.
            maxAttempts: 5,
        })
            .addRetry({ maxAttempts: 3 })
            .addCatch(ignore, { errors: [constants_1.UNPROCESSABLE_PACKAGE_ERROR_NAME] })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.Timeout'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['ECS.AmazonECSException', 'ECS.InvalidParameterException'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.TaskFailed'], resultPath: '$.error' })
            .addCatch(sendToDeadLetterQueue, { errors: ['States.ALL'], resultPath: '$.error' })
            .next(addToCatalogIfNeeded));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.days(1),
            tracingEnabled: true,
        });
        if (props.vpc) {
            // Ensure the State Machine does not get to run before the VPC can be used.
            this.stateMachine.node.addDependency(props.vpc.internetConnectivityEstablished);
        }
        props.monitoring.addHighSeverityAlarm('Backend Orchestration Failed', this.stateMachine.metricFailed()
            .createAlarm(this, 'OrchestrationFailed', {
            alarmName: `${this.stateMachine.node.path}/${this.stateMachine.metricFailed().metricName}`,
            alarmDescription: [
                'Backend orchestration failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to state machine: ${deep_link_1.stateMachineUrl(this.stateMachine)}`,
                'Warning: messages that resulted in a failed exectuion will NOT be in the DLQ!',
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
        }));
        // This function is intended to be manually triggered by an operrator to
        // attempt redriving messages from the DLQ.
        this.redriveFunction = new redrive_state_machine_1.RedriveStateMachine(this, 'Redrive', {
            description: '[ConstructHub/Redrive] Manually redrives all messages from the backend dead letter queue',
            environment: {
                STATE_MACHINE_ARN: this.stateMachine.stateMachineArn,
                QUEUE_URL: this.deadLetterQueue.queueUrl,
            },
            memorySize: 1024,
            timeout: core_1.Duration.minutes(15),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        this.stateMachine.grantStartExecution(this.redriveFunction);
        this.deadLetterQueue.grantConsumeMessages(this.redriveFunction);
        // The workflow is intended to be manually triggered by an operator to
        // reprocess all package versions currently in store through the orchestrator.
        this.regenerateAllDocumentation = new RegenerateAllDocumentation(this, 'RegenerateAllDocumentation', {
            bucket: props.bucket,
            stateMachine: this.stateMachine,
        }).stateMachine;
    }
    metricEcsTaskCount(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'TaskCount',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'CpuUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsCpuUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % CPU utilization from the CPU units utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mCpuUtilized, 0) / FILL(mCpuReserved, REPEAT)',
            usingMetrics: {
                mCpuReserved: this.metricEcsCpuReserved(),
                mCpuUtilized: this.metricEcsCpuUtilized(),
            },
        });
    }
    metricEcsMemoryReserved(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryReserved',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilized(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'MemoryUtilized',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsMemoryUtilization(opts) {
        return new aws_cloudwatch_1.MathExpression({
            ...opts,
            // Calculates the % memory utilization from the RAM utilization &
            // reservation. FILL is used to make a non-sparse time-series (the metrics
            // are not emitted if no task runs)
            expression: '100 * FILL(mMemoryUtilized, 0) / FILL(mMemoryReserved, REPEAT)',
            usingMetrics: {
                mMemoryReserved: this.metricEcsMemoryReserved(),
                mMemoryUtilized: this.metricEcsMemoryUtilized(),
            },
        });
    }
    metricEcsNetworkRxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkRxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
    metricEcsNetworkTxBytes(opts) {
        return new aws_cloudwatch_1.Metric({
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            dimensionsMap: { ClusterName: this.ecsCluster.clusterName },
            metricName: 'NetworkTxBytes',
            namespace: 'ECS/ContainerInsights',
        });
    }
}
exports.Orchestration = Orchestration;
class RegenerateAllDocumentation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const processVersions = new aws_stepfunctions_1.Choice(this, 'Get package versions page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First versions page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each key prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Start Orchestration Workflow', {
            stateMachine: props.stateMachine,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                bucket: props.bucket.bucketName,
                assembly: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.ASSEMBLY_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                metadata: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.METADATA_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
                package: { key: aws_stepfunctions_1.JsonPath.stringAt(`States.Format('{}${constants_1.PACKAGE_KEY_SUFFIX.substr(1)}', $.Prefix)`) },
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.REQUEST_RESPONSE,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })));
        processVersions.next(new aws_stepfunctions_1.Choice(this, 'Has more versions?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processVersions)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Success')));
        const processPackageVersions = new aws_stepfunctions_1.StateMachine(this, 'PerPackage', {
            definition: processVersions,
            timeout: core_1.Duration.hours(1),
            tracingEnabled: true,
        });
        // This workflow is broken into two sub-workflows because otherwise it hits the 25K events limit
        // of StepFunction executions relatively quickly.
        const processNamespace = new aws_stepfunctions_1.Choice(this, 'Get @scope page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First @scope page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each @scope/pkg', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new tasks.StepFunctionsStartExecution(this, 'Process scoped package', {
            stateMachine: processPackageVersions,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] })));
        processNamespace.next(new aws_stepfunctions_1.Choice(this, 'Has more packages?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), processNamespace)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'All Done')));
        const start = new aws_stepfunctions_1.Choice(this, 'Get prefix page')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), new tasks.CallAwsService(this, 'Next prefixes page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                ContinuationToken: aws_stepfunctions_1.JsonPath.stringAt('$.response.NextContinuationToken'),
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] }))
            .otherwise(new tasks.CallAwsService(this, 'First prefix page', {
            service: 's3',
            action: 'listObjectsV2',
            iamAction: 's3:ListBucket',
            iamResources: [props.bucket.bucketArn],
            parameters: {
                Bucket: props.bucket.bucketName,
                Delimiter: '/',
                Prefix: constants_1.STORAGE_KEY_PREFIX,
            },
            resultPath: '$.response',
        }).addRetry({ errors: ['S3.SdkClientException'] })).afterwards()
            .next(new aws_stepfunctions_1.Map(this, 'For each prefix', { itemsPath: '$.response.CommonPrefixes', resultPath: aws_stepfunctions_1.JsonPath.DISCARD })
            .iterator(new aws_stepfunctions_1.Choice(this, 'Is this a @scope/ prefix?')
            .when(aws_stepfunctions_1.Condition.stringMatches('$.Prefix', `${constants_1.STORAGE_KEY_PREFIX}@*`), processNamespace)
            .otherwise(new tasks.StepFunctionsStartExecution(this, 'Process unscoped package', {
            stateMachine: processPackageVersions,
            associateWithParent: true,
            input: aws_stepfunctions_1.TaskInput.fromObject({
                Prefix: aws_stepfunctions_1.JsonPath.stringAt('$.Prefix'),
            }),
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.RUN_JOB,
        }).addRetry({ errors: ['StepFunctions.ExecutionLimitExceeded'] }))
            .afterwards()));
        start.next(new aws_stepfunctions_1.Choice(this, 'Has more prefixes?')
            .when(aws_stepfunctions_1.Condition.isPresent('$.response.NextContinuationToken'), start)
            .otherwise(new aws_stepfunctions_1.Succeed(this, 'Done')));
        this.stateMachine = new aws_stepfunctions_1.StateMachine(this, 'Resource', {
            definition: start,
            stateMachineName: stateMachineNameFrom(this.node.path),
            timeout: core_1.Duration.hours(4),
            tracingEnabled: true,
        });
        props.bucket.grantRead(processPackageVersions);
        props.bucket.grantRead(this.stateMachine);
    }
}
/**
 * This turns a node path into a valid state machine name, to try and improve
 * the StepFunction's AWS console experience while minimizing the risk for
 * collisons.
 */
function stateMachineNameFrom(nodePath) {
    // Poor man's replace all...
    return nodePath.split(/[^a-z0-9+!@.()=_'-]+/i).join('.');
}
//# sourceMappingURL=data:application/json;base64,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