"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Transliterator = void 0;
const aws_ecs_1 = require("@aws-cdk/aws-ecs");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_logs_1 = require("@aws-cdk/aws-logs");
const aws_stepfunctions_1 = require("@aws-cdk/aws-stepfunctions");
const aws_stepfunctions_tasks_1 = require("@aws-cdk/aws-stepfunctions-tasks");
const core_1 = require("@aws-cdk/core");
const s3 = require("../../s3");
const constants = require("../shared/constants");
const language_1 = require("../shared/language");
const transliterator_1 = require("./transliterator");
/**
 * Transliterates jsii assemblies to various other languages.
 */
class Transliterator extends core_1.Construct {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        const repository = ((_a = props.vpcEndpoints) === null || _a === void 0 ? void 0 : _a.codeArtifact) && props.vpcEndpoints.codeArtifactApi
            ? (_b = props.codeArtifact) === null || _b === void 0 ? void 0 : _b.throughVpcEndpoint(props.vpcEndpoints.codeArtifactApi, props.vpcEndpoints.codeArtifact) : props.codeArtifact;
        const bucket = props.vpcEndpoints
            ? s3.throughVpcEndpoint(props.bucket, props.vpcEndpoints.s3)
            : props.bucket;
        const environment = {
            // temporaty hack to generate construct-hub compliant markdown.
            // see https://github.com/cdklabs/jsii-docgen/blob/master/src/docgen/render/markdown.ts#L172
            HEADER_SPAN: 'true',
            // Set embedded metrics format environment to "Local", to have a consistent experience.
            AWS_EMF_ENVIRONMENT: 'Local',
        };
        if ((_c = props.vpcEndpoints) === null || _c === void 0 ? void 0 : _c.codeArtifactApi) {
            // Those are returned as an array of HOSTED_ZONE_ID:DNS_NAME... We care
            // only about the DNS_NAME of the first entry in that array (which is
            // the AZ-agnostic DNS name).
            environment.CODE_ARTIFACT_API_ENDPOINT = core_1.Fn.select(1, core_1.Fn.split(':', core_1.Fn.select(0, props.vpcEndpoints.codeArtifactApi.vpcEndpointDnsEntries)));
        }
        if (props.codeArtifact) {
            environment.CODE_ARTIFACT_DOMAIN_NAME = props.codeArtifact.repositoryDomainName;
            environment.CODE_ARTIFACT_DOMAIN_OWNER = props.codeArtifact.repositoryDomainOwner;
            environment.CODE_ARTIFACT_REPOSITORY_ENDPOINT = props.codeArtifact.repositoryNpmEndpoint;
        }
        this.logGroup = new aws_logs_1.LogGroup(this, 'LogGroup', { retention: props.logRetention });
        this.containerDefinition = new transliterator_1.Transliterator(this, 'Resource', {
            environment,
            logging: aws_ecs_1.LogDrivers.awsLogs({ logGroup: this.logGroup, streamPrefix: 'transliterator' }),
            taskDefinition: new aws_ecs_1.FargateTaskDefinition(this, 'TaskDefinition', {
                cpu: 4096,
                memoryLimitMiB: 8192,
            }),
        });
        repository === null || repository === void 0 ? void 0 : repository.grantReadFromRepository(this.taskDefinition.taskRole);
        // The task handler reads & writes to this bucket.
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.ASSEMBLY_KEY_SUFFIX}`);
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.PACKAGE_KEY_SUFFIX}`);
        bucket.grantRead(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.UNINSTALLABLE_PACKAGE_SUFFIX}`);
        bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.UNINSTALLABLE_PACKAGE_SUFFIX}`);
        bucket.grantDelete(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.UNINSTALLABLE_PACKAGE_SUFFIX}`);
        for (const language of language_1.DocumentationLanguage.ALL) {
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.NOT_SUPPORTED_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.NOT_SUPPORTED_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
            bucket.grantWrite(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
            bucket.grantDelete(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language)}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
            bucket.grantDelete(this.taskDefinition.taskRole, `${constants.STORAGE_KEY_PREFIX}*${constants.docsKeySuffix(language, '*')}${constants.CORRUPT_ASSEMBLY_SUFFIX}`);
        }
        const executionRole = this.taskDefinition.obtainExecutionRole();
        (_d = props.vpcEndpoints) === null || _d === void 0 ? void 0 : _d.ecrApi.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ecr:GetAuthorizationToken',
            ],
            resources: ['*'],
            principals: [executionRole],
            sid: 'Allow-ECR-ReadOnly',
        }));
        (_e = props.vpcEndpoints) === null || _e === void 0 ? void 0 : _e.ecr.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'ecr:BatchCheckLayerAvailability',
                'ecr:GetDownloadUrlForLayer',
                'ecr:BatchGetImage',
            ],
            // We cannot get the ECR repository info from an asset... So scoping down to same-account repositories instead...
            resources: [core_1.Stack.of(this).formatArn({ service: 'ecr', resource: 'repository', arnFormat: core_1.ArnFormat.SLASH_RESOURCE_NAME, resourceName: '*' })],
            principals: [executionRole],
            sid: 'Allow-ECR-ReadOnly',
        }));
        (_f = props.vpcEndpoints) === null || _f === void 0 ? void 0 : _f.cloudWatchLogs.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'logs:CreateLogStream',
                'logs:PutLogEvents',
            ],
            resources: [
                core_1.Stack.of(this).formatArn({ service: 'logs', resource: 'log-group', arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME, resourceName: this.logGroup.logGroupName }),
                core_1.Stack.of(this).formatArn({ service: 'logs', resource: 'log-group', arnFormat: core_1.ArnFormat.COLON_RESOURCE_NAME, resourceName: `${this.logGroup.logGroupName}:log-stream:*` }),
            ],
            principals: [executionRole],
            sid: 'Allow-Logging',
        }));
        (_g = props.vpcEndpoints) === null || _g === void 0 ? void 0 : _g.stepFunctions.addToPolicy(new aws_iam_1.PolicyStatement({
            effect: aws_iam_1.Effect.ALLOW,
            actions: [
                'states:SendTaskFailure',
                'states:SendTaskHeartbeat',
                'states:SendTaskSuccess',
            ],
            resources: ['*'],
            principals: [this.taskDefinition.taskRole],
            sid: 'Allow-StepFunctions-Callbacks',
        }));
    }
    get taskDefinition() {
        return this.containerDefinition.taskDefinition;
    }
    createEcsRunTask(scope, id, opts) {
        return new aws_stepfunctions_tasks_1.EcsRunTask(scope, id, {
            // The container sends heartbeats every minute, but when the runloop will
            // actually get to submitting it is fairly dependent on the async
            // workload's nature... so we don't rely on it all too strongly, and
            // default to a 5 minutes timeout here as a minimal protection. Options
            // may override this liberally if they know better.
            heartbeat: core_1.Duration.minutes(5),
            ...opts,
            containerOverrides: [{
                    containerDefinition: this.containerDefinition,
                    command: aws_stepfunctions_1.JsonPath.listAt('$'),
                    environment: [
                        { name: 'SFN_TASK_TOKEN', value: aws_stepfunctions_1.JsonPath.taskToken },
                    ],
                }],
            integrationPattern: aws_stepfunctions_1.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
            launchTarget: new aws_stepfunctions_tasks_1.EcsFargateLaunchTarget({ platformVersion: aws_ecs_1.FargatePlatformVersion.VERSION1_4 }),
            subnets: opts.vpcSubnets,
            securityGroups: opts.securityGroups,
            taskDefinition: this.taskDefinition,
        });
    }
}
exports.Transliterator = Transliterator;
//# sourceMappingURL=data:application/json;base64,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