"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DomainRedirect = void 0;
const acm = require("@aws-cdk/aws-certificatemanager");
const cf = require("@aws-cdk/aws-cloudfront");
const origins = require("@aws-cdk/aws-cloudfront-origins");
const route53 = require("@aws-cdk/aws-route53");
const route53_targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
/**
 * Redirects one domain to another domain using S3 and CloudFront.
 */
class DomainRedirect extends core_1.Construct {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        const bucket = this.getOrCreateBucket(props.targetDomainName);
        const sourceDomainName = props.source.hostedZone.zoneName;
        const cert = (_a = props.source.certificate) !== null && _a !== void 0 ? _a : new acm.DnsValidatedCertificate(this, 'Certificate', {
            domainName: sourceDomainName,
            hostedZone: props.source.hostedZone,
        });
        const dist = new cf.Distribution(this, 'Distribution', {
            domainNames: [sourceDomainName],
            defaultBehavior: { origin: new origins.S3Origin(bucket) },
            certificate: cert,
            minimumProtocolVersion: cf.SecurityPolicyProtocol.TLS_V1_2_2018,
        });
        // IPv4
        new route53.ARecord(this, 'ARecord', {
            zone: props.source.hostedZone,
            target: route53.RecordTarget.fromAlias(new route53_targets.CloudFrontTarget(dist)),
            comment: 'Created by the AWS CDK',
        });
        // IPv6
        new route53.AaaaRecord(this, 'AaaaRecord', {
            zone: props.source.hostedZone,
            target: route53.RecordTarget.fromAlias(new route53_targets.CloudFrontTarget(dist)),
            comment: 'Created by the AWS CDK',
        });
    }
    /**
     * Gets or creates a bucket which redirects all requests to the given domain name.
     *
     * @param domainName The target domain name
     * @returns An S3 bucket
     */
    getOrCreateBucket(domainName) {
        var _a;
        const buckets = this.getOrCreateBucketsScope();
        const id = `RedirectBucket-${domainName}`;
        return (_a = buckets.node.tryFindChild(id)) !== null && _a !== void 0 ? _a : new s3.Bucket(buckets, id, {
            websiteRedirect: {
                hostName: domainName,
            },
        });
    }
    /**
     * Returns a singleton construct scope (stack-level) that includes all the
     * buckets used for domain redirection.
     *
     * @returns A construct
     */
    getOrCreateBucketsScope() {
        var _a;
        const stack = core_1.Stack.of(this);
        const scopeId = 'DomainRedirectBucketsA177hj';
        return (_a = stack.node.tryFindChild(scopeId)) !== null && _a !== void 0 ? _a : new core_1.Construct(stack, scopeId);
    }
}
exports.DomainRedirect = DomainRedirect;
//# sourceMappingURL=data:application/json;base64,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