"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmJs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const aws_sqs_1 = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const deep_link_1 = require("../deep-link");
const metric_utils_1 = require("../metric-utils");
const runbook_url_1 = require("../runbook-url");
const storage_1 = require("../s3/storage");
const canary_1 = require("./npmjs/canary");
const constants_lambda_shared_1 = require("./npmjs/constants.lambda-shared");
const npm_js_follower_1 = require("./npmjs/npm-js-follower");
const stage_and_notify_1 = require("./npmjs/stage-and-notify");
/**
 * The periodicity at which the NpmJs follower will run. This MUST be a valid
 * CloudWatch Metric grain, as this will also be the period of the CloudWatch
 * alarm that montiors the health of the follower.
 */
const FOLLOWER_RUN_RATE = core_1.Duration.minutes(5);
/**
 * A package source that gets package data from the npmjs.com package registry.
 */
class NpmJs {
    constructor(props = {}) {
        this.props = props;
    }
    bind(scope, { baseUrl, denyList, ingestion, licenseList, monitoring, queue, repository }) {
        var _b, _c, _d;
        repository === null || repository === void 0 ? void 0 : repository.addExternalConnection('public:npmjs');
        const storageFactory = storage_1.S3StorageFactory.getOrCreate(scope);
        const bucket = this.props.stagingBucket || storageFactory.newBucket(scope, 'NpmJs/StagingBucket', {
            blockPublicAccess: aws_s3_1.BlockPublicAccess.BLOCK_ALL,
            enforceSSL: true,
            lifecycleRules: [{ prefix: "staged/" /* STAGED_KEY_PREFIX */, expiration: core_1.Duration.days(30) }],
        });
        bucket.grantRead(ingestion);
        const stager = new stage_and_notify_1.StageAndNotify(scope, 'NpmJs-StageAndNotify', {
            deadLetterQueue: new aws_sqs_1.Queue(scope, 'StagerDLQ', {
                encryption: aws_sqs_1.QueueEncryption.KMS_MANAGED,
                retentionPeriod: core_1.Duration.days(14),
                visibilityTimeout: core_1.Duration.minutes(15),
            }),
            description: `[${scope.node.path}/NpmJS-StageAndNotify] Stages tarballs to S3 and notifies ConstructHub`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                QUEUE_URL: queue.queueUrl,
            },
            memorySize: 10024,
            retryAttempts: 2,
            timeout: core_1.Duration.minutes(5),
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(stager);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(stager);
        queue.grantSendMessages(stager);
        stager.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(stager.deadLetterQueue, { batchSize: 1, enabled: false }));
        const follower = new npm_js_follower_1.NpmJsFollower(scope, 'NpmJs', {
            description: `[${scope.node.path}/NpmJs] Periodically query npmjs.com index for new packages`,
            environment: {
                AWS_EMF_ENVIRONMENT: 'Local',
                BUCKET_NAME: bucket.bucketName,
                FUNCTION_NAME: stager.functionName,
            },
            memorySize: 10024,
            reservedConcurrentExecutions: 1,
            timeout: FOLLOWER_RUN_RATE,
            tracing: aws_lambda_1.Tracing.ACTIVE,
        });
        bucket.grantReadWrite(follower, constants_lambda_shared_1.MARKER_FILE_NAME);
        denyList === null || denyList === void 0 ? void 0 : denyList.grantRead(follower);
        licenseList.grantRead(follower);
        stager.grantInvoke(follower);
        const rule = new aws_events_1.Rule(scope, 'NpmJs/Schedule', {
            description: `${scope.node.path}/NpmJs/Schedule`,
            schedule: aws_events_1.Schedule.rate(FOLLOWER_RUN_RATE),
            targets: [new aws_events_targets_1.LambdaFunction(follower)],
        });
        this.registerAlarms(scope, follower, stager, monitoring, rule);
        return {
            name: follower.node.path,
            links: [
                { name: 'NpmJs Follower', url: deep_link_1.lambdaFunctionUrl(follower), primary: true },
                { name: 'Marker Object', url: deep_link_1.s3ObjectUrl(bucket, constants_lambda_shared_1.MARKER_FILE_NAME) },
                { name: 'Stager', url: deep_link_1.lambdaFunctionUrl(stager) },
                { name: 'Stager DLQ', url: deep_link_1.sqsQueueUrl(stager.deadLetterQueue) },
            ],
            dashboardWidgets: [
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Follower Health',
                        left: [
                            metric_utils_1.fillMetric(follower.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(follower.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            this.metricRemainingTime({ label: 'Remaining Time' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Stager Health',
                        left: [
                            metric_utils_1.fillMetric(stager.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(stager.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            stager.metricDuration({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ], [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Follower',
                        left: [
                            metric_utils_1.fillMetric(this.metricChangeCount({ label: 'Change Count' }), 0),
                            metric_utils_1.fillMetric(this.metricUnprocessableEntity({ label: 'Unprocessable' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            metric_utils_1.fillMetric(this.metricNpmJsChangeAge({ label: 'Lag to npmjs.com' }), 'REPEAT'),
                            metric_utils_1.fillMetric(this.metricPackageVersionAge({ label: 'Package Version Age' }), 'REPEAT'),
                        ],
                        rightYAxis: { label: 'Milliseconds', min: 0, showUnits: false },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'CouchDB Changes',
                        left: [
                            metric_utils_1.fillMetric(this.metricLastSeq({ label: 'Last Sequence Number' }), 'REPEAT'),
                        ],
                        period: core_1.Duration.minutes(5),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Stager Dead-Letter Queue',
                        left: [
                            metric_utils_1.fillMetric(stager.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }), 0),
                            metric_utils_1.fillMetric(stager.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }), 0),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            stager.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message' }),
                        ],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    ...(((_b = this.props.enableCanary) !== null && _b !== void 0 ? _b : true)
                        ? this.registerCanary(follower, (_c = this.props.canaryPackage) !== null && _c !== void 0 ? _c : 'construct-hub-probe', (_d = this.props.canarySla) !== null && _d !== void 0 ? _d : core_1.Duration.minutes(5), bucket, baseUrl, monitoring)
                        : []),
                ],
            ],
        };
    }
    /**
     * The average time it took to process a changes batch.
     */
    metricBatchProcessingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.AVERAGE,
            ...opts,
            metricName: "BatchProcessingTime" /* BATCH_PROCESSING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of changes that were processed.
     */
    metricChangeCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "ChangeCount" /* CHANGE_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The last sequence number that was processed. This metric can be used to
     * discover when a sequence reset has happened in the CouchDB instance.
     */
    metricLastSeq(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "LastSeq" /* LAST_SEQ */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    metricNpmJsChangeAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "NpmJsChangeAge" /* NPMJS_CHANGE_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The age of the oldest package version that was processed.
     */
    metricPackageVersionAge(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
            ...opts,
            metricName: "PackageVersionAge" /* PACKAGE_VERSION_AGE */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were inspected.
     */
    metricPackageVersionCount(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "PackageVersionCount" /* PACKAGE_VERSION_COUNT */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The total count of package versions that were deemed relevant.
     */
    metricRelevantPackageVersions(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "RelevantPackageVersions" /* RELEVANT_PACKAGE_VERSIONS */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of time that was remaining when the lambda returned in order to
     * avoid hitting a timeout.
     */
    metricRemainingTime(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(5),
            statistic: aws_cloudwatch_1.Statistic.MINIMUM,
            ...opts,
            metricName: "RemainingTime" /* REMAINING_TIME */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    /**
     * The amount of changes that were not processed due to having an invalid
     * format.
     */
    metricUnprocessableEntity(opts) {
        return new aws_cloudwatch_1.Metric({
            period: core_1.Duration.minutes(1),
            statistic: aws_cloudwatch_1.Statistic.SUM,
            ...opts,
            metricName: "UnprocessableEntity" /* UNPROCESSABLE_ENTITY */,
            namespace: constants_lambda_shared_1.METRICS_NAMESPACE,
        });
    }
    registerAlarms(scope, follower, stager, monitoring, schedule) {
        const failureAlarm = follower.metricErrors().createAlarm(scope, 'NpmJs/Follower/Failures', {
            alarmName: `${scope.node.path}/NpmJs/Follower/Failures`,
            alarmDescription: [
                'The NpmJs follower function failed!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 3,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.MISSING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Follower Failures', failureAlarm);
        const notRunningAlarm = follower.metricInvocations({ period: FOLLOWER_RUN_RATE })
            .createAlarm(scope, 'NpmJs/Follower/NotRunning', {
            alarmName: `${scope.node.path}/NpmJs/Follower/NotRunning`,
            alarmDescription: [
                'The NpmJs follower function is not running!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        monitoring.addHighSeverityAlarm('NpmJs/Follower Not Running', notRunningAlarm);
        // The period for this alarm needs to match the scheduling interval of the
        // follower, otherwise the metric will be too sparse to properly detect
        // problems.
        const noChangeAlarm = this.metricChangeCount({ period: FOLLOWER_RUN_RATE })
            .createAlarm(scope, 'NpmJs/Follower/NoChanges', {
            alarmName: `${scope.node.path}/NpmJs/Follower/NoChanges`,
            alarmDescription: [
                'The NpmJs follower function is no discovering any changes from CouchDB!',
                '',
                `RunBook: ${runbook_url_1.RUNBOOK_URL}`,
                '',
                `Direct link to Lambda function: ${deep_link_1.lambdaFunctionUrl(follower)}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.LESS_THAN_THRESHOLD,
            evaluationPeriods: 2,
            threshold: 1,
            // If the metric is not emitted, it can be assumed to be zero.
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
        });
        monitoring.addLowSeverityAlarm('Np npmjs.com changes discovered', noChangeAlarm);
        const dlqNotEmptyAlarm = new aws_cloudwatch_1.MathExpression({
            expression: 'mVisible + mHidden',
            usingMetrics: {
                mVisible: stager.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ period: core_1.Duration.minutes(1) }),
                mHidden: stager.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ period: core_1.Duration.minutes(1) }),
            },
        }).createAlarm(scope, `${scope.node.path}/NpmJs/Stager/DLQNotEmpty`, {
            alarmName: `${scope.node.path}/NpmJs/Stager/DLQNotEmpty`,
            alarmDescription: [
                'The NpmJS package stager is failing - its dead letter queue is not empty',
                '',
                `Link to the lambda function: ${deep_link_1.lambdaFunctionUrl(stager)}`,
                `Link to the dead letter queue: ${deep_link_1.sqsQueueUrl(stager.deadLetterQueue)}`,
                '',
                `Runbook: ${runbook_url_1.RUNBOOK_URL}`,
            ].join('/n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
            evaluationPeriods: 1,
            threshold: 1,
            treatMissingData: aws_cloudwatch_1.TreatMissingData.NOT_BREACHING,
        });
        monitoring.addLowSeverityAlarm('NpmJs/Stager DLQ Not Empty', dlqNotEmptyAlarm);
        // Finally - the "not running" alarm depends on the schedule (it won't run until the schedule
        // exists!), and the schedule depends on the failure alarm existing (we don't want it to run
        // before we can know it is failing). This means the returned `IDependable` effectively ensures
        // all alarms have been provisionned already! Isn't it nice!
        notRunningAlarm.node.addDependency(schedule);
        schedule.node.addDependency(failureAlarm);
    }
    registerCanary(scope, packageName, visibilitySla, bucket, constructHubBaseUrl, monitoring) {
        const canary = new canary_1.NpmJsPackageCanary(scope, 'Canary', { bucket, constructHubBaseUrl, packageName });
        const alarm = new aws_cloudwatch_1.MathExpression({
            expression: 'MAX([mDwell, mTTC])',
            period: core_1.Duration.minutes(1),
            usingMetrics: {
                mDwell: canary.metricDwellTime(),
                mTTC: canary.metricTimeToCatalog(),
            },
        }).createAlarm(canary, 'Alarm', {
            alarmName: `${canary.node.path}/SLA-Breached`,
            alarmDescription: [
                `New versions of ${packageName} have been published over ${visibilitySla.toHumanString()} ago and are still not visible in construct hub`,
                `Runbook: ${runbook_url_1.RUNBOOK_URL}`,
            ].join('\n'),
            comparisonOperator: aws_cloudwatch_1.ComparisonOperator.GREATER_THAN_THRESHOLD,
            evaluationPeriods: 2,
            // If there is no data, the canary might not be running, so... *Chuckles* we're in danger!
            treatMissingData: aws_cloudwatch_1.TreatMissingData.BREACHING,
            threshold: visibilitySla.toSeconds(),
        });
        monitoring.addHighSeverityAlarm('New version visibility SLA breached', alarm);
        return [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Package Canary',
                left: [
                    canary.metricDwellTime({ label: 'Dwell Time' }),
                    canary.metricTimeToCatalog({ label: 'Time to Catalog' }),
                ],
                leftAnnotations: [{
                        color: '#ff0000',
                        label: `SLA (${visibilitySla.toHumanString()})`,
                        value: visibilitySla.toSeconds(),
                    }],
                leftYAxis: { min: 0 },
                right: [
                    canary.metricTrackedVersionCount({ label: 'Tracked Version Count' }),
                ],
                rightYAxis: { min: 0 },
            }),
        ];
    }
}
exports.NpmJs = NpmJs;
_a = JSII_RTTI_SYMBOL_1;
NpmJs[_a] = { fqn: "construct-hub.sources.NpmJs", version: "0.3.262" };
//# sourceMappingURL=data:application/json;base64,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