"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const os = require("os");
const path = require("path");
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const fs = require("fs-extra");
const docgen = require("jsii-docgen");
const aws = require("../shared/aws.lambda-shared");
const code_artifact_lambda_shared_1 = require("../shared/code-artifact.lambda-shared");
const compress_content_lambda_shared_1 = require("../shared/compress-content.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const shell_out_lambda_shared_1 = require("../shared/shell-out.lambda-shared");
const constants_1 = require("./constants");
const util_1 = require("./util");
const ASSEMBLY_KEY_REGEX = new RegExp(`^${constants.STORAGE_KEY_PREFIX}((?:@[^/]+/)?[^/]+)/v([^/]+)${constants.ASSEMBLY_KEY_SUFFIX}$`);
// Capture groups:                                                     ┗━━━━━━━━━1━━━━━━━┛  ┗━━2━━┛
/**
 * This function receives an S3 event, and for each record, proceeds to download
 * the `.jsii` assembly the event refers to, transliterates it to the language,
 * configured in `TARGET_LANGUAGE`, and uploads the resulting `.jsii.<lang>`
 * object to S3.
 *
 * @param event   an S3 event payload
 * @param context a Lambda execution context
 *
 * @returns nothing
 */
function handler(event) {
    console.log(`Event: ${JSON.stringify(event, null, 2)}`);
    // We'll need a writable $HOME directory, or this won't work well, because
    // npm will try to write stuff like the `.npmrc` or package caches in there
    // and that'll bail out on EROFS if that fails.
    return ensureWritableHome(async () => {
        var _a, _b;
        const endpoint = process.env.CODE_ARTIFACT_REPOSITORY_ENDPOINT;
        if (!endpoint) {
            console.log('No CodeArtifact endpoint configured - using npm\'s default registry');
        }
        else {
            console.log(`Using CodeArtifact registry: ${endpoint}`);
            const domain = env_lambda_shared_1.requireEnv('CODE_ARTIFACT_DOMAIN_NAME');
            const domainOwner = process.env.CODE_ARTIFACT_DOMAIN_OWNER;
            const apiEndpoint = process.env.CODE_ARTIFACT_API_ENDPOINT;
            await code_artifact_lambda_shared_1.logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint });
        }
        // Set up NPM shared cache directory (https://docs.npmjs.com/cli/v7/using-npm/config#cache)
        const npmCacheDir = process.env.NPM_CACHE;
        if (npmCacheDir) {
            // Create it if it does not exist yet...
            await fs.mkdirp(npmCacheDir);
            console.log(`Using shared NPM cache at: ${npmCacheDir}`);
            await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `cache=${npmCacheDir}`);
        }
        const created = new Array();
        const [, packageName, packageVersion] = (_a = event.assembly.key.match(ASSEMBLY_KEY_REGEX)) !== null && _a !== void 0 ? _a : [];
        if (packageName == null) {
            throw new Error(`Invalid object key: "${event.assembly.key}". It was expected to match ${ASSEMBLY_KEY_REGEX}!`);
        }
        const packageFqn = `${packageName}@${packageVersion}`;
        console.log(`Source Bucket:  ${event.bucket}`);
        console.log(`Source Key:     ${event.assembly.key}`);
        console.log(`Source Version: ${event.assembly.versionId}`);
        console.log(`Fetching assembly: ${event.assembly.key}`);
        const assemblyResponse = await aws.s3().getObject({ Bucket: event.bucket, Key: event.assembly.key }).promise();
        if (!assemblyResponse.Body) {
            throw new Error(`Response body for assembly at key ${event.assembly.key} is empty`);
        }
        const assembly = JSON.parse(assemblyResponse.Body.toString('utf-8'));
        const submodules = Object.keys((_b = assembly.submodules) !== null && _b !== void 0 ? _b : {}).map(s => s.split('.')[1]);
        console.log(`Fetching package: ${event.package.key}`);
        const tarballExists = await aws.s3ObjectExists(event.bucket, event.package.key);
        if (!tarballExists) {
            throw new Error(`Tarball does not exist at key ${event.package.key} in bucket ${event.bucket}.`);
        }
        const readStream = aws.s3().getObject({ Bucket: event.bucket, Key: event.package.key }).createReadStream();
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'packages-'));
        const tarball = path.join(tmpdir, 'package.tgz');
        await util_1.writeFile(tarball, readStream);
        const cwd = process.cwd();
        try {
            process.chdir(tmpdir);
            console.log(`Preparing package install at ${tmpdir}`);
            const installResult = await shell_out_lambda_shared_1.shellOutWithOutput('npm', 'install', tarball, '--ignore-scripts', '--no-bin-links', '--no-save', '--include=dev', '--no-package-lock', '--json');
            if (installResult.exitCode !== 0) {
                if (installResult.signal) {
                    throw new Error(`"npm install" was killed by signal ${installResult.signal}`);
                }
                const toThrow = new Error(`"npm install" failed with exit code ${installResult.exitCode}`);
                toThrow.name = 'jsii-docgen.NpmError';
                try {
                    const json = JSON.parse(installResult.stdout.toString('utf-8'));
                    console.error(`Installation result: ${JSON.stringify(json, null, 2)}`);
                    if (json.error.code) {
                        toThrow.name += `.${json.error.code}`;
                    }
                }
                catch (error) {
                    console.error(`Installation result: ${installResult.stdout.toString('utf-8')}`);
                }
                throw toThrow;
            }
        }
        finally {
            process.chdir(cwd);
        }
        const packageDir = path.join(tmpdir, 'node_modules', packageName);
        console.log(`Generating documentation from ${packageDir}...`);
        for (const language of language_1.DocumentationLanguage.ALL) {
            if (event.languages && !event.languages[language.toString()]) {
                console.log(`Skipping language ${language} as it was not requested!`);
                continue;
            }
            const isSupported = language === language_1.DocumentationLanguage.TYPESCRIPT || assembly.targets[language.targetName];
            if (!isSupported) {
                console.error(`Package ${packageName}@${packageVersion} does not support ${language}, skipping!`);
                console.log(`Assembly targets: ${JSON.stringify(assembly.targets, null, 2)}`);
                for (const submodule of [undefined, ...submodules]) {
                    const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(language, submodule)) + constants.NOT_SUPPORTED_SUFFIX;
                    const response = await uploadFile(event.bucket, key, event.assembly.versionId);
                    created.push({ bucket: event.bucket, key, versionId: response.VersionId });
                }
                continue;
            }
            const generateDocs = aws_embedded_metrics_1.metricScope((metrics) => async (lang) => {
                metrics.setDimensions();
                metrics.setNamespace(constants_1.METRICS_NAMESPACE);
                const uploads = new Map();
                const docs = await docgen.Documentation.forProject(packageDir, {
                    assembliesDir: tmpdir,
                    language: docgen.Language.fromString(lang.name),
                });
                function renderAndDispatch(submodule) {
                    console.log(`Rendering documentation in ${lang} for ${packageFqn} (submodule: ${submodule})`);
                    const page = Buffer.from(docs.render({ submodule, linkFormatter: linkFormatter(docs) }).render());
                    metrics.putMetric("DocumentSizeBytes" /* DOCUMENT_SIZE */, page.length, aws_embedded_metrics_1.Unit.Bytes);
                    const { buffer: body, contentEncoding } = compress_content_lambda_shared_1.compressContent(page);
                    metrics.putMetric("CompressedDocumentSizeBytes" /* COMPRESSED_DOCUMENT_SIZE */, body.length, aws_embedded_metrics_1.Unit.Bytes);
                    const key = event.assembly.key.replace(/\/[^/]+$/, constants.docsKeySuffix(lang, submodule));
                    console.log(`Uploading ${key}`);
                    const upload = uploadFile(event.bucket, key, event.assembly.versionId, body, contentEncoding);
                    uploads.set(key, upload);
                }
                renderAndDispatch();
                for (const submodule of submodules) {
                    renderAndDispatch(submodule);
                }
                for (const [key, upload] of uploads.entries()) {
                    const response = await upload;
                    created.push({ bucket: event.bucket, key, versionId: response.VersionId });
                    console.log(`Finished uploading ${key} (Version ID: ${response.VersionId})`);
                }
            });
            await generateDocs(language);
        }
        return created;
    });
}
exports.handler = handler;
async function ensureWritableHome(cb) {
    // Since $HOME is not set, or is not writable, we'll just go make our own...
    const fakeHome = await fs.mkdtemp(path.join(os.tmpdir(), 'fake-home'));
    console.log(`Made temporary $HOME directory: ${fakeHome}`);
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = fakeHome;
        return await cb();
    }
    finally {
        process.env.HOME = oldHome;
        await fs.remove(fakeHome);
        console.log(`Cleaned-up temporary $HOME directory: ${fakeHome}`);
    }
}
function uploadFile(bucket, key, sourceVersionId, body, contentEncoding) {
    return aws.s3().putObject({
        Bucket: bucket,
        Key: key,
        Body: body,
        CacheControl: 'public, max-age=300, must-revalidate, proxy-revalidate',
        ContentEncoding: contentEncoding,
        ContentType: 'text/markdown; charset=UTF-8',
        Metadata: {
            'Origin-Version-Id': sourceVersionId !== null && sourceVersionId !== void 0 ? sourceVersionId : 'N/A',
        },
    }).promise();
}
/**
 * A link formatter to make sure type links redirect to the appropriate package
 * page in the webapp.
 */
function linkFormatter(docs) {
    function _formatter(type) {
        const packageName = type.source.assembly.name;
        const packageVersion = type.source.assembly.version;
        // the webapp sanitizes anchors - so we need to as well when
        // linking to them.
        const hash = sanitize(type.fqn);
        if (docs.assembly.name === packageName) {
            // link to the same package - just add the hash
            return `#${hash}`;
        }
        // cross link to another package
        return `/packages/${packageName}/v/${packageVersion}?lang=${type.language.toString()}${type.submodule ? `&submodule=${type.submodule}` : ''}#${hash}`;
    }
    return _formatter;
}
function sanitize(input) {
    return input
        .toLowerCase()
        .replace(/[^a-zA-Z0-9 ]/g, '')
        .replace(/ /g, '-');
}
;
//# sourceMappingURL=data:application/json;base64,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