"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.DjangoEcs = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const acm = require("@aws-cdk/aws-certificatemanager");
const ec2 = require("@aws-cdk/aws-ec2");
const ecs = require("@aws-cdk/aws-ecs");
const patterns = require("@aws-cdk/aws-ecs-patterns");
const elbv2 = require("@aws-cdk/aws-elasticloadbalancingv2");
const logs = require("@aws-cdk/aws-logs");
const route53 = require("@aws-cdk/aws-route53");
const route53targets = require("@aws-cdk/aws-route53-targets");
const s3 = require("@aws-cdk/aws-s3");
const cdk = require("@aws-cdk/core");
const database_1 = require("./common/database");
const elasticache_1 = require("./common/elasticache");
const vpc_1 = require("./common/vpc");
const beat_1 = require("./ecs/celery/beat");
const worker_1 = require("./ecs/celery/worker");
const tasks_1 = require("./ecs/tasks");
/**
 * Configures a Django project using ECS Fargate.
 *
 * @stability stable
 */
class DjangoEcs extends cdk.Construct {
    /**
     * @stability stable
     */
    constructor(scope, id, props) {
        var _b, _c;
        super(scope, id);
        /**
         * VPC must have public, private and isolated subnets
         *
         * If you don't provide a VPC, a new VPC will be created
         */
        if (!props.vpc) {
            const applicationVpc = new vpc_1.ApplicationVpc(scope, 'AppVpc');
            this.vpc = applicationVpc.vpc;
        }
        else {
            const vpc = props.vpc;
            this.vpc = vpc;
        }
        /**
         * static files bucket name is derived from the Construct id if not provided
         */
        const staticFilesBucket = new s3.Bucket(scope, 'StaticBucket', {
            bucketName: props === null || props === void 0 ? void 0 : props.bucketName,
        });
        this.staticFileBucket = staticFilesBucket;
        /**
         * ECS cluster
         */
        this.cluster = new ecs.Cluster(scope, 'EcsCluster', { vpc: this.vpc });
        /**
         * task definition construct
         */
        const taskDefinition = new ecs.TaskDefinition(scope, 'TaskDefinition', {
            compatibility: ecs.Compatibility.FARGATE,
            cpu: '256',
            memoryMiB: '512',
        });
        /**
         * Container image used in web API, celery worker and management task containers
         */
        this.image = new ecs.AssetImage(props.imageDirectory);
        /**
         * RDS managed database using PostgreSQL
         */
        const database = new database_1.RdsPostgresInstance(scope, 'RdsPostgresInstance', {
            vpc: this.vpc,
            dbSecretName: 'dbSecret',
        });
        /**
         * A security group in the VPC for our application (ECS Fargate services and tasks)
         * Allow the application services to access the RDS security group
         */
        const appSecurityGroup = new ec2.SecurityGroup(scope, 'appSecurityGroup', {
            vpc: this.vpc,
        });
        /**
         * ElastiCache Redis Cluster for caching, celery message brokering
         */
        const elastiCacheRedis = new elasticache_1.ElastiCacheCluster(scope, 'ElastiCacheCluster', {
            vpc: this.vpc,
            appSecurityGroup,
        });
        const environment = {
            AWS_STORAGE_BUCKET_NAME: staticFilesBucket.bucketName,
            POSTGRES_SERVICE_HOST: database.rdsPostgresInstance.dbInstanceEndpointAddress,
            DB_SECRET_NAME: database.dbSecretName,
            DEBUG: '0',
            DJANGO_SETTINGS_MODULE: 'backend.settings.production',
            REDIS_SERVICE_HOST: elastiCacheRedis.elastiCacheCluster.attrRedisEndpointAddress,
        };
        taskDefinition.addContainer('backendContainer', {
            image: this.image,
            environment,
            command: props.webCommand,
            portMappings: [{
                    containerPort: 8000,
                    hostPort: 8000,
                }],
            logging: ecs.LogDriver.awsLogs({
                logRetention: logs.RetentionDays.ONE_DAY,
                streamPrefix: 'BackendContainer',
            }),
        });
        new tasks_1.managementCommandTask(scope, 'migrate', {
            image: this.image,
            command: ['python3', 'manage.py', 'migrate', '--no-input'],
            appSecurityGroup,
            environment,
            dbSecret: database.secret,
            cluster: this.cluster,
            // this command will run automatically on every deployment
            run: true,
            vpc: this.vpc,
        });
        new tasks_1.managementCommandTask(scope, 'collectstatic', {
            image: this.image,
            command: ['python3', 'manage.py', 'collectstatic', '--no-input'],
            appSecurityGroup,
            environment,
            dbSecret: database.secret,
            cluster: this.cluster,
            vpc: this.vpc,
        });
        /**
         * Use celery beat if it is configured in props
         */
        if ((_b = props.useCeleryBeat) !== null && _b !== void 0 ? _b : false) {
            new beat_1.CeleryBeat(scope, 'CeleryBeat', {
                dbSecret: database.secret,
                image: this.image,
                command: [
                    'celery',
                    '--app=backend.celery_app:app',
                    'beat',
                    '--loglevel=INFO',
                    '--pidfile=/code/celerybeat.pid',
                ],
                environment,
                cluster: this.cluster,
                securityGroups: [appSecurityGroup],
            });
        }
        ;
        /**
         * Celery worker
         *
         * TODO: refactor to support defining multiple queues
         */
        new worker_1.CeleryWorker(scope, 'CeleryWorkerDefaultQueue', {
            image: this.image,
            command: [
                'celery',
                '-A',
                'backend.celery_app:app',
                'worker',
                '-l',
                'info',
                '-Q',
                'celery',
                '-n',
                'worker-celery@%h',
            ],
            environment,
            cluster: this.cluster,
            securityGroups: [appSecurityGroup],
            dbSecret: database.secret,
        });
        /**
         * Lookup Certificate from ARN or generate
         * Deploy external-dns and related IAM resource if a domain name is included
         */
        let certificate = undefined;
        let hostedZone = undefined;
        if (props.domainName) {
            hostedZone = route53.HostedZone.fromLookup(scope, 'hosted-zone', {
                domainName: props.domainName,
            });
            /**
             * Lookup or request ACM certificate depending on value of certificateArn
             */
            if (props.certificateArn) {
                // lookup ACM certificate from ACM certificate ARN
                certificate = acm.Certificate.fromCertificateArn(scope, 'certificate', props.certificateArn);
            }
            else {
                // request a new certificate
                certificate = new acm.Certificate(this, 'SSLCertificate', {
                    domainName: props.domainName,
                    validation: acm.CertificateValidation.fromDns(hostedZone),
                });
            }
        }
        /**
          * ECS load-balanced fargate service
         */
        const albfs = new patterns.ApplicationLoadBalancedFargateService(scope, 'AlbFargateService', {
            cluster: this.cluster,
            taskDefinition,
            securityGroups: [appSecurityGroup],
            desiredCount: 1,
            assignPublicIp: true,
            redirectHTTP: (_c = !!props.domainName) !== null && _c !== void 0 ? _c : false,
            protocol: props.domainName ? elbv2.ApplicationProtocol.HTTPS : elbv2.ApplicationProtocol.HTTP,
            certificate: props.domainName ? certificate : undefined,
        });
        database.secret.grantRead(albfs.taskDefinition.taskRole);
        const albLogsBucket = new s3.Bucket(scope, `${id}-alb-logs`);
        albfs.loadBalancer.logAccessLogs(albLogsBucket);
        /**
       * Health check for the application load balancer
       */
        albfs.targetGroup.configureHealthCheck({
            path: '/api/health-check/',
        });
        /**
         * Allows the app security group to communicate with the RDS security group
         */
        database.rdsSecurityGroup.addIngressRule(appSecurityGroup, ec2.Port.tcp(5432));
        /**
         * Grant the task defintion read-write access to static files bucket
         */
        staticFilesBucket.grantReadWrite(albfs.taskDefinition.taskRole);
        new cdk.CfnOutput(this, 'bucketName', { value: staticFilesBucket.bucketName });
        new cdk.CfnOutput(this, 'apiUrl', { value: albfs.loadBalancer.loadBalancerFullName });
        if (props.domainName) {
            new route53.ARecord(scope, 'ARecord', {
                target: route53.RecordTarget.fromAlias(new route53targets.LoadBalancerTarget(albfs.loadBalancer)),
                zone: hostedZone,
                recordName: props.domainName,
            });
        }
    }
}
exports.DjangoEcs = DjangoEcs;
_a = JSII_RTTI_SYMBOL_1;
DjangoEcs[_a] = { fqn: "django-cdk.DjangoEcs", version: "0.0.22" };
//# sourceMappingURL=data:application/json;base64,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